package ch.sahits.game.openpatrician.display.gameplay.impl;

import ch.sahits.game.graphic.image.IDataImageLoader;
import ch.sahits.game.graphic.image.IPolygonScaling;
import ch.sahits.game.graphic.image.ImageScaleState;
import ch.sahits.game.graphic.image.model.ImageData;
import ch.sahits.game.graphic.image.model.NamedPolygon;
import com.google.common.base.Preconditions;
import javafx.event.EventHandler;
import javafx.scene.input.MouseEvent;
import javafx.scene.shape.Polygon;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map.Entry;

public abstract class ScenePolygonInitialzer {
	protected static final Logger logger = LogManager.getLogger(ScenePolygonInitialzer.class);

	private final String SCENE_IMAGE_NAME;
	@Autowired
	@Qualifier("xmlImageLoader")
	protected IDataImageLoader xmlLoader;
	@Autowired
	@Qualifier("polygonScaling")
	private IPolygonScaling scaler;

	/**
	 * Comparator based on descending order of the entries key values.
	 * @author Andi Hotz, (c) Sahits GmbH, 2013
	 * Created on Nov 13, 2013
	 *
	 */
	protected static class PolygonComparator implements Comparator<Entry<Integer, NamedPolygon>> {
		@Override
		public int compare(Entry<Integer, NamedPolygon> o1,
				Entry<Integer, NamedPolygon> o2) {
			return o2.getKey().compareTo(o1.getKey());
		}
		
	}
	/**
	 * Initialize the polygons defined in the metadata considering the proper scaling and cropping.
	 * @param state current state
	 * @return List of polygons.
	 */
	public List<Polygon> initialzePolygons(ImageScaleState state) {
		ImageData imageData = xmlLoader.getImageData(SCENE_IMAGE_NAME);
		Preconditions.checkNotNull(imageData, "There is not imageData for "+SCENE_IMAGE_NAME);
		ImageData recalculated = scaler.recalculatePolygons(imageData, state);
		List<Polygon> polygons = new ArrayList<>(recalculated.getPolygons().size());
		List<Entry<Integer, NamedPolygon>> sortedPolygons = new ArrayList<>(recalculated.getPolygons().entrySet());
		sortedPolygons.sort(new PolygonComparator());
		for (Entry<Integer, NamedPolygon> item : sortedPolygons) {
			NamedPolygon poly = item.getValue();
			logger.info("Add polygon: "+poly);
			addAction(poly);
			polygons.add(poly);
		}
		return polygons;
	}
	/**
	 * Add an event to the polygon.
	 * @param poly polygon
	 */
	private void addAction(NamedPolygon poly) {
		final String name = poly.getName();
		EventHandler<MouseEvent> eventHandler = createEventHandler(name);
		poly.setOnMousePressed(eventHandler);
		// TODO aho Nov 15, 2013: add other handler like escape for dialog closing
	}
	/**
	 * Create an event handler for the clicking on the polygon with the given name.
	 * @param name of the polygon
	 * @return EventHandler for the mouse click event
	 */
	protected abstract EventHandler<MouseEvent> createEventHandler(final String name);

	public ScenePolygonInitialzer(String sceneName) {
		SCENE_IMAGE_NAME = sceneName;
	}

}