package ch.sahits.game.openpatrician.display.javafx;

import ch.sahits.game.openpatrician.display.gameplay.impl.EScene;
import ch.sahits.game.openpatrician.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.openpatrician.event.EViewChangeEvent;
import ch.sahits.game.openpatrician.event.NavigationStateChange;
import ch.sahits.game.openpatrician.event.NoticeBoardClose;
import ch.sahits.game.openpatrician.event.data.ENavigationStateChange;
import ch.sahits.game.openpatrician.javafx.model.EDialogType;
import ch.sahits.game.openpatrician.model.initialisation.StartNewGameBean;
import ch.sahits.game.openpatrician.sound.IPlayableTrack;
import ch.sahits.game.openpatrician.sound.ITrackLoader;
import ch.sahits.game.openpatrician.sound.ITrackPlayer;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.DependentInitialisation;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.ObjectPropertyType;
import com.google.common.base.Preconditions;
import com.google.common.eventbus.AsyncEventBus;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;

/**
 * Event handler for the city hall scene.
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Jan 03, 2015
 */
@Component
@Lazy
@DependentInitialisation(StartNewGameBean.class)
@ClassCategory(EClassCategory.HANDLER)
public class ChurchSceneEventHandler implements ISceneEventHandler {
    @Autowired
    @Qualifier("clientEventBus")
    private AsyncEventBus clientEventBus;
    @Autowired
    private ITrackLoader trackLoader;
    @Autowired
    private ITrackPlayer soundPlayer;
    @ObjectPropertyType(EScene.class)
    private ObjectProperty<EScene> currentScene = new SimpleObjectProperty<>(this, "currentScene", EScene.PORT);
    private IDialogContoller dialogContoller; // this instance exists not at startup

    private boolean navigationDisabled = false;
    @PostConstruct
    private void initialize() {
        IPlayableTrack backgroundSound = trackLoader.loadTrack("organ");
        currentScene.addListener((observable, oldScene, newScene) -> {
            if (newScene == EScene.CHURCH) {
                soundPlayer.add(backgroundSound);
                soundPlayer.play();
            } else {
                soundPlayer.stop();
                soundPlayer.clearPlayList();
            }
        });
    }
    @Override
    public ObjectProperty<EScene> currentSceneProperty() {
        return currentScene;
    }

    @Override
    public void setDialogContoller(IDialogContoller dialogContoller) {
        this.dialogContoller = dialogContoller;
    }

    @Override
    public void handleEvent(Object eventNotice) {
        Preconditions.checkNotNull(dialogContoller, "Dialog controller must be initialized first");
        if (eventNotice instanceof ViewChangeCityPlayerProxyJFX) {
            ViewChangeCityPlayerProxyJFX cityProxy = (ViewChangeCityPlayerProxyJFX) eventNotice;
            // Scene change market place
            if (cityProxy.getViewChangeEvent() == EViewChangeEvent.MAIN_VIEW_MARKET && !navigationDisabled) {
                clientEventBus.post(new NoticeBoardClose());
                currentScene.set(EScene.MARKET);
                dialogContoller.changeScene();
            }
        }
        if (eventNotice instanceof EDialogType) {
            if (eventNotice == EDialogType.CHURCH_PRAY) {
                pray();
            } else {
                dialogContoller.replaceDialog((EDialogType) eventNotice);
            }
        }
    }

    private void pray() {
        // Close any open dialogs
        dialogContoller.closeDialog();
        // Disable navigation
        NavigationStateChange changeEvent = new NavigationStateChange(ENavigationStateChange.DISABLE_NAVIGATION);
        clientEventBus.post(changeEvent);
        navigationDisabled = true;
        // Play pray track
        IPlayableTrack track = trackLoader.loadTrack("pater-noster");


        // Enable navigation
        soundPlayer.stop();
        track.play();
        ChangeListener<Boolean> playingChangeListener = new ChangeListener<>() {
            @Override
            public void changed(ObservableValue<? extends Boolean> observable, Boolean oldValue, Boolean newValue) {
                NavigationStateChange changeBackEvent = new NavigationStateChange(ENavigationStateChange.ENABLE_NAVIGATION);
                clientEventBus.post(changeBackEvent);
                soundPlayer.play();
                navigationDisabled = false;
                track.playingProperty().removeListener(this);
            }
        };
        track.playingProperty().addListener(playingChangeListener);

    }
}
