package ch.sahits.game.openpatrician.display.javafx;

import ch.sahits.game.event.ViewChangeEvent;
import ch.sahits.game.event.data.PauseGame;
import ch.sahits.game.graphic.image.IDataImageLoader;
import ch.sahits.game.graphic.image.IImageUtilities;
import ch.sahits.game.graphic.image.ImageScaleState;
import ch.sahits.game.graphic.image.model.ImageData;
import ch.sahits.game.openpatrician.clientserverinterface.model.factory.GameFactory;
import ch.sahits.game.openpatrician.display.ClientViewState;
import ch.sahits.game.openpatrician.display.EViewState;
import ch.sahits.game.openpatrician.display.dialog.service.DialogFactory;
import ch.sahits.game.openpatrician.display.gameplay.impl.EScene;
import ch.sahits.game.openpatrician.display.gameplay.impl.PolygonInitializerFactory;
import ch.sahits.game.openpatrician.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.openpatrician.event.EViewChangeEvent;
import ch.sahits.game.openpatrician.event.data.DisplayEventVideo;
import ch.sahits.game.openpatrician.event.data.SwitchCity;
import ch.sahits.game.openpatrician.javafx.control.EventMediaPlayer;
import ch.sahits.game.openpatrician.javafx.dialog.Dialog;
import ch.sahits.game.openpatrician.javafx.model.EDialogType;
import ch.sahits.game.openpatrician.model.IGame;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.map.IMap;
import ch.sahits.game.openpatrician.model.ship.IConvoy;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.ui.DialogTemplate;
import ch.sahits.game.openpatrician.model.ui.IDialogState;
import ch.sahits.game.openpatrician.utilities.IRebinabable;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.DialogCloseing;
import ch.sahits.game.openpatrician.utilities.annotation.DialogOpening;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.ObjectPropertyType;
import ch.sahits.game.openpatrician.utilities.annotation.UniquePrototype;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import ch.sahits.game.openpatrician.utilities.spring.DependentAnnotationConfigApplicationContext;
import ch.sahits.game.openpatrician.utilities.spring.DialogScope;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import javafx.application.Platform;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleDoubleProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.event.EventHandler;
import javafx.geometry.Dimension2D;
import javafx.geometry.Point2D;
import javafx.scene.Group;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.input.KeyCode;
import javafx.scene.input.KeyEvent;
import javafx.scene.layout.StackPane;
import javafx.scene.paint.Color;
import javafx.scene.shape.Polygon;
import javafx.scene.shape.Rectangle;
import javafx.scene.text.Font;
import lombok.Getter;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

@UniquePrototype
@ClassCategory(EClassCategory.PROTOTYPE_BEAN)
public class MainGameView extends Group implements  IDialogContoller, IRebinabable {
	private static final Logger logger = LogManager.getLogger(MainGameView.class);
    public static final int MINMIMAL_DISPLAY_HEIGHT = 766;
    /**
	 * Use low level byte to indicate what is displayed instead of an enum.
	 */
	private double controlWidth;
	private double controlHeight;

	@Autowired
	private IImageUtilities imageUtils;
	@Autowired
    @Qualifier("xmlImageLoader")
	private IDataImageLoader xmlLoader;
    @Autowired
    private Locale locale;

    @Autowired
    private MessageSource messageSource;
    @Autowired
	private DialogFactory dialogFactory;
	private Rectangle placeHolder;
	private BaseMainGameImageView imgView;
	@Getter
	private Dialog dialog;

	private EventMediaPlayer eventViewer = null;
    @Autowired
    private DialogScope dialogScope;
	@ObjectPropertyType(EScene.class)
	private ObjectProperty<EScene> currentScene = new SimpleObjectProperty<>(this, "currentScene", EScene.PORT);
    @Autowired
    private SceneEventHandlerFactory sceneEventHandlerFactory;
    @Autowired
    private PolygonInitializerFactory polygonInitFactory;
    @Autowired
    private ClientViewState viewState;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    @Qualifier("clientEventBus")
    private AsyncEventBus clientEventBus;
    @Autowired
    @Qualifier("timerEventBus")
    private AsyncEventBus timerEventBus;
    @Autowired
    @Qualifier("uiTimer")
    private ScheduledExecutorService uiTimer;
	@Autowired
	private DisplayMessageOverlay dispMesgOverlay;
	@Autowired
	private DependentAnnotationConfigApplicationContext context;
	@Autowired
	private GameFactory gameFactory;
    @Autowired
    private IMap map;
    private StackPane stack;
    private SeamapImageView seaMapView;
//    private EventHandler<KeyEvent> closeKeyHandler;

    public MainGameView(double width, double heigth) {
		super();
		setManaged(false);
		controlWidth = width;
		controlHeight = heigth;
	}

	@PostConstruct
	private void createComponents() {

        setUpImageView();

        sceneEventHandlerFactory.currentSceneProperty().bindBidirectional(currentScene);
        sceneEventHandlerFactory.setUpDialogController(this);

        polygonInitFactory.currentSceneProperty().bindBidirectional(currentScene);

        stack = new StackPane();
		stack.getChildren().add(imgView);
		placeHolder = new Rectangle(controlWidth, controlHeight);
		placeHolder.setFill(Color.BLACK);
		dispMesgOverlay.setLayoutY(10);
		dispMesgOverlay.setLayoutX(10);
		getChildren().addAll(placeHolder, stack, dispMesgOverlay);
        clientEventBus.register(this);
        clientServerEventBus.register(this);
        timerEventBus.register(this);
		setOnMouseClicked(event -> {
			IGame game = gameFactory.getGame();
			game.normalSpeed();
		});
        EventHandler<KeyEvent> closeKeyHandler = event -> {
            if (event.getCode().equals(KeyCode.ESCAPE)) {
                Platform.runLater(() -> {
                    try {
                        closeEventView();
                    } catch (RuntimeException e) {
                        logger.error("Failed to close event view", e);
                    }
                });
            }
        };
        addEventHandler(KeyEvent.KEY_PRESSED, closeKeyHandler);
	}
	@PreDestroy
	private void unregister() {
		clientEventBus.unregister(this);
        clientServerEventBus.unregister(this);
        timerEventBus.unregister(this);
	}
	/**
	 * Retrieve the name for the image to be displayed in the scene
	 * @return name of the image tag to be associated with the current scene.
	 */
	private String getImageNameFromScene() {
		switch (currentScene.get()) {
		case MARKET:
			return "images/scene/marketPlaceScene";
		case PORT:
			return "images/scene/portScene";
		case SHIPYARD:
			return "images/scene/shipYard";
		case TAVERN:
			return "images/scene/tavernIterior";
		case CITY_HALL:
			return "images/scene/cityhall";
		case LOANER:
			return "images/scene/loaner";
		case CHURCH:
			return "images/scene/churchInterior";
        case GUILD:
            return "images/scene/guildInterior";
        case ARMORY:
            return "images/scene/Armory";
		default:
			throw new RuntimeException(currentScene.get() + " is not implemented");
		}
	}

	private void setUpImageView() {
		String imageName = getImageNameFromScene();
		Image tmpImg = xmlLoader.getImage(imageName);
		ImageData imgData = xmlLoader.getImageData(imageName);
		Dimension2D targetDim = new Dimension2D(controlWidth, controlHeight);
		ImageScaleState state = new ImageScaleState(new Dimension2D(tmpImg.getWidth(), tmpImg.getHeight()), targetDim, imgData.getCrop(), imgData.getMaxCrop());
		logger.debug("State of the port scene before: " + state);

        // // todo: andi 13/12/13: it would be simpler to bound the values instead of reinitializing them
        imgView = new MainGameImageView(controlWidth, controlHeight, imageUtils.cropAndScale(imageName, state), state);
		List<Polygon> polygons = polygonInitFactory.getScenePolygonInitializer().initialzePolygons(state);
		imgView.resetPolygons(polygons);
        viewState.setState(EViewState.CITY);

        logger.debug("State of the port scene after : " + state);
	}
    private void setUpSeamapImageView() {
        IHumanPlayer player = viewState.getPlayer();
        Image mapImage = imageUtils.createMapWithCities(map, player);
        final Font openPatrician18 = Font.font("OpenPatrician", 18);
        mapImage = imageUtils.addCityNames(map, mapImage, openPatrician18);
        Dimension2D origDim = new Dimension2D(mapImage.getWidth(), mapImage.getHeight());
        double scale = controlHeight/origDim.getHeight();
        Dimension2D targetDim = new Dimension2D(origDim.getWidth()*scale, origDim.getHeight()*scale);
        mapImage = imageUtils.scale(mapImage, targetDim, scale);

        Point2D coordinates = viewState.getCurrentCityProxy().get().getCity().getCoordinates();
        coordinates = new Point2D(coordinates.getX() * scale, coordinates.getY() * scale);

        if (seaMapView == null) {
           seaMapView = (SeamapImageView) context.getBean("seamapImageView", new Object[]{mapImage, controlWidth, controlHeight, coordinates, scale});
        } else {
            seaMapView.resetImage(mapImage,controlWidth, controlHeight, scale);

        }

        imgView = seaMapView;
        viewState.setState(EViewState.MAP);
    }

	public void widthChange(double oldWidth, double newWidth) {
        List<Polygon> polygons = new ArrayList<>();

        Dimension2D origDim = map.getDimension();
        double scale = controlHeight/origDim.getHeight();
        if (imgView instanceof MainGameImageView) {
            String imageName = getImageNameFromScene();
            Image tmpImg = xmlLoader.getImage(imageName);
            ImageData imgData = xmlLoader.getImageData(imageName);
            Dimension2D targetDim = new Dimension2D(newWidth, controlHeight);
            ImageScaleState state = new ImageScaleState(new Dimension2D(tmpImg.getWidth(), tmpImg.getHeight()), targetDim, imgData.getCrop(), imgData.getMaxCrop());
            ImageView tmpImgView = imageUtils.cropAndScale(imageName, state);
            ((MainGameImageView) imgView).setImageView(newWidth, controlHeight, tmpImgView, state);
            polygons = polygonInitFactory.getScenePolygonInitializer().initialzePolygons(state);
        }
        if (imgView instanceof SeamapImageView) {

            Dimension2D targetDim = new Dimension2D(origDim.getWidth()*scale, origDim.getHeight()*scale);
            IHumanPlayer player = viewState.getPlayer();
            Image mapImage = imageUtils.createMapWithCities(map, player);
            final Font openPatrician18 = Font.font("OpenPatrician", 18);
            mapImage = imageUtils.addCityNames(map, mapImage, openPatrician18);
            mapImage = imageUtils.scale(mapImage, targetDim, scale);
            ((SeamapImageView)imgView).resetImage(mapImage,newWidth, controlHeight, scale);
        }
		imgView.resetPolygons(polygons);
		placeHolder.setWidth(newWidth);
		this.controlWidth = newWidth;
	}


	public void heightChange(double oldHeight, double newHeigth) {
        List<Polygon> polygons = new ArrayList<>();
        Dimension2D origDim = map.getDimension();
        double scale = newHeigth/origDim.getHeight();
        if (imgView instanceof MainGameImageView) {
            String imageName = getImageNameFromScene();
            Image tmpImg = xmlLoader.getImage(imageName);
            ImageData imgData = xmlLoader.getImageData(imageName);
            Dimension2D targetDim = new Dimension2D(controlWidth, newHeigth);
            ImageScaleState state = new ImageScaleState(new Dimension2D(tmpImg.getWidth(), tmpImg.getHeight()), targetDim, imgData.getCrop(), imgData.getMaxCrop());
            ImageView tmpImgView = imageUtils.cropAndScale(imageName, state);
            ((MainGameImageView) imgView).setImageView(controlWidth, newHeigth, tmpImgView, state);
            polygons = polygonInitFactory.getScenePolygonInitializer().initialzePolygons(state);
        }
        if (imgView instanceof SeamapImageView) {
            Dimension2D targetDim = new Dimension2D(origDim.getWidth()*scale, origDim.getHeight()*scale);
            IHumanPlayer player = viewState.getPlayer();
            Image mapImage = imageUtils.createMapWithCities(map, player);
            final Font openPatrician18 = Font.font("OpenPatrician", 18);
            mapImage = imageUtils.addCityNames(map, mapImage, openPatrician18);
            mapImage = imageUtils.scale(mapImage, targetDim, scale);
            ((SeamapImageView)imgView).resetImage(mapImage,controlWidth, newHeigth, scale);
        }
        imgView.resetPolygons(polygons);
		placeHolder.setHeight(newHeigth);
		this.controlHeight = newHeigth;
	}
    @Subscribe
    public void handleViewChange(ViewChangeEvent event) {
        if (event.getAddresse().equals(MainGameView.class)) {
            if (event.getEventNotice() instanceof DialogTemplate) {
                setNewDialog((DialogTemplate) event.getEventNotice());
            } else if (event.getEventNotice() instanceof IDialogState) {
                setNewDialog((IDialogState) event.getEventNotice());
            } else if (event.getEventNotice() instanceof ViewChangeCityPlayerProxyJFX
                    && ((ViewChangeCityPlayerProxyJFX)event.getEventNotice()).getViewChangeEvent() == EViewChangeEvent.MAIN_VIEW_SEA_MAP) {
                currentScene.set(EScene.SEAMAP);
                changeScene();
            } else {
                sceneEventHandlerFactory.getSceneEventHandler().handleEvent(event.getEventNotice());
            }
        }
    }
    @Subscribe
    public void handleDialogStateEvent(IDialogState dialogState) {
	    setNewDialog(dialogState);
    }

    /**
     * Replace the current dialog with a new one.
     * @param dialogType of the new dialog
     * @param params varargs to construct the new dialog
     */
    @Override
    public void replaceDialog(EDialogType dialogType, Object...params) {
        closeDialog();
		while (dialog != null) {
			try {
				Thread.sleep(5);
			} catch (InterruptedException e) {
				// Do not care
			}
		}
        setNewDialog(dialogType, params);
    }

    /**
     * Close the currently open dialog
     */
    @Override
    @DialogCloseing
    public void closeDialog() {
		if (Platform.isFxApplicationThread()) {
			closeDialogUnwrapped();
		} else {
			Platform.runLater(this::closeDialogUnwrapped);
		}
    }

	private void closeDialogUnwrapped() {
		if (dialog != null) {
            dialog.close();
			context.removePrototypeBean(dialog);
			getChildren().remove(dialog);
			dialog = null;
			dialogScope.closeScope();
		} else {
            logger.warn("Tried to close the dialog but it was already null");
        }
	}

	/**
     * Set up the new dialog.
     * @param dialogType new dialog type
     * @param params varargs to create the dialog
     */
    @Override
    @DialogOpening
    public void setNewDialog(EDialogType dialogType, Object...params) {
        if (params.length == 0) {
            dialog = dialogFactory.getDialog(dialogType, viewState.getCurrentCityProxy().get());
        } else if (params.length == 2) {
            if (params[0] instanceof IConvoy && params[1] instanceof IShip) {
                dialog = dialogFactory.getDialog(dialogType, viewState.getCurrentCityProxy().get(), (IConvoy)params[0], (IShip)params[1]);
            } else {
                throw new IllegalArgumentException("Cannot handle parameters "+params[0]+" and "+params[1]);
            }
        }
		dialog.setDialogType(dialogType);
		setNewDialog();
    }

    /**
     * Set up the new dialog based on a dialog template.
     * @param template dialog template for the new dialog
     */
    @Override
    @DialogOpening
    public void setNewDialog(DialogTemplate template) {
        closeDialogBeforeReplacing();
        dialog = dialogFactory.getDialog(template);
        setNewDialog();
    }

    /**
     * Set up the new dialog based on a dialog state.
     * @param dialogState dialog state for the new dialog
     */
    @Override
    @DialogOpening
    public void setNewDialog(IDialogState dialogState) {
        closeDialogBeforeReplacing();
        dialog = dialogFactory.getDialog(dialogState);
        setNewDialog();
    }

    private void closeDialogBeforeReplacing() {
        while (dialog != null) {
            closeDialog();
            try {
                Thread.sleep(5);
            } catch (InterruptedException e) {
                // Do not care
            }
        }
    }

    /**
     * Set up the dialog that was initialized. This method should only be called from #setNewDialog(EDialogType) or
     * #setNewDialog(DialogTemplate).
     */
    private void setNewDialog() {
        dialogScope.openScope();
        dialog.setLayoutX((controlWidth - Dialog.WIDTH) / 2);
        dialog.setLayoutY((controlHeight - Dialog.HEIGHT) / 2);
		if (Platform.isFxApplicationThread()) {
		    closeEventView();
			getChildren().add(dialog);
		} else {
			Platform.runLater(() -> {
                closeEventView();
				if (dialog != null) {
					getChildren().add(dialog);
				}
			});
		}
    }

    /**
     * Handle the displaying of an event video.
     * @param event display event video, containg the parameters to initiate the video to be played
     */
    @Subscribe
    public void handleEventVideoDisplay(DisplayEventVideo event) {
        if (Platform.isFxApplicationThread()) {
            displayEventVideoUnwrapped(event);
        } else {
            Platform.runLater(() -> displayEventVideoUnwrapped(event));
        }
    }

    private void displayEventVideoUnwrapped(DisplayEventVideo event) {
        closeEventView();
        String title = messageSource.getMessage(event.getTitleKey(), event.getTitleParams(), locale.getCurrentLocal());
        String description = messageSource.getMessage(event.getDescriptionKey(), event.getDescriptionParams(), locale.getCurrentLocal());

        eventViewer = new EventMediaPlayer(event.getMediaType(), new SimpleDoubleProperty(controlWidth));
        eventViewer.setTitle(title);
        eventViewer.setDescription(description);
        eventViewer.setLayoutX((0.35 * controlWidth) / 2);
        eventViewer.setLayoutY(30);
        getChildren().add(eventViewer);

        uiTimer.schedule(() -> Platform.runLater(this::closeEventView), event.getDurationInSeconds() + 5, TimeUnit.SECONDS);
    }

    /**
     * Close the event view and stop playback. This method must be called from
     * within the FX application thread.
     */
    private void closeEventView() {
        if (eventViewer != null) {
            eventViewer.stop();
            getChildren().remove(eventViewer);
            eventViewer = null;
        }
    }
    @Subscribe
    public void handleGamePause(PauseGame event) {
        Platform.runLater(this::closeEventView);
    }

    /**
     * Change the scene image
     */
    @Override
    public void changeScene() {
		if (Platform.isFxApplicationThread()) {
			changeSceneUnwrapped();
			closeEventView();
		} else {
			Platform.runLater(() -> {
			    changeSceneUnwrapped();
			    closeEventView();
			});
		}
     }

	private void changeSceneUnwrapped() {
		closeDialog();
		stack.getChildren().removeAll(imgView);
        if (currentScene.get() == EScene.SEAMAP) {
            setUpSeamapImageView();
        } else {
            setUpImageView();
        }
        stack.getChildren().add(imgView);
    }

	@Override
	public void rebind() {
        if (currentScene.get() != EScene.PORT) {
            ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.MAIN_VIEW_PORT);
            handleViewChange(new ViewChangeEvent(MainGameView.class, proxy));
        }
        if (seaMapView != null) {
            seaMapView.removeShipIcons();
        }
	}
	@Subscribe
	public void handleSwitchToCity(SwitchCity event) {
		currentScene.set(EScene.PORT);
		changeScene();
	}
}
