package ch.sahits.game.openpatrician.display.scene;

import ch.sahits.game.event.data.ResumeGame;
import ch.sahits.game.openpatrician.clientserverinterface.model.factory.GameFactory;
import ch.sahits.game.openpatrician.display.SceneChangeable;
import ch.sahits.game.openpatrician.display.service.GameOptionsService;
import ch.sahits.game.openpatrician.display.service.UIFactory;
import ch.sahits.game.openpatrician.event.EGameStatusChange;
import ch.sahits.game.openpatrician.event.GameStateChange;
import ch.sahits.game.openpatrician.event.data.NewGame;
import ch.sahits.game.openpatrician.javafx.OpenPatricianScene;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianSpinner;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianStoneButton;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.Difficulty;
import ch.sahits.game.openpatrician.model.EGameSpeed;
import ch.sahits.game.openpatrician.model.EObjective;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.UniquePrototype;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import javafx.application.Platform;
import javafx.collections.FXCollections;
import javafx.event.EventHandler;
import javafx.geometry.HPos;
import javafx.geometry.Pos;
import javafx.scene.control.Label;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.ColumnConstraints;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.StackPane;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

/**
 * Scene for initally selecting a savegame, load it and start it.
 */
@UniquePrototype
@ClassCategory(EClassCategory.PROTOTYPE_BEAN)
public class LoadGameScene extends BaseStartupScene {
	private final Logger logger = LogManager.getLogger(getClass());
	@Autowired
	private UIFactory uiFactory;
	@Autowired
	private GameFactory gameFactory;
	@Autowired
	private Locale locale;
	@Autowired
	private MessageSource messageSource;
	@Autowired
	private GameOptionsService gameOptions;
	@Autowired
	private Date gameDate;

	@Autowired
	@Qualifier("serverClientEventBus")
	private AsyncEventBus clientServerEventBus;
	@Autowired
	@Qualifier("timerEventBus")
	private AsyncEventBus timerEventBus;
	@Autowired
	private SceneChangeService sceneChanger;

	// Need to inject the font as for custom controls it is not stylable
	private SceneChangeable sceneChangeable;
	private OpenPatricianSpinner lodableGames;

	public LoadGameScene() {
		super(new StackPane());
		getRoot().getStylesheets().add(this.getClass().getResource(getStyleSheetFilename()).toExternalForm());
	}


	public SceneChangeable getSceneChangeable() {
		return sceneChangeable;
	}

	public void setSceneChangeable(SceneChangeable sceneChangeable) {
		this.sceneChangeable = sceneChangeable;
	}
	protected String getStyleSheetFilename() {
		return "newGame.css";
	}
    @PreDestroy
    private void unregister() {
        clientServerEventBus.unregister(this);
    }
    @PostConstruct
	private void createControls() {
		gameFactory.getGame();
		StackPane root = (StackPane) getRoot();
		root.setId("LoadGameStackPane");

		GridPane grid = new GridPane();
		grid.getStyleClass().add("grid");
		grid.setAlignment(Pos.CENTER);
		grid.getColumnConstraints().add(new ColumnConstraints(200)); // column 1
		grid.getColumnConstraints().add(new ColumnConstraints(300)); // column 2
		grid.getColumnConstraints().add(new ColumnConstraints(250)); // column 3
		grid.getColumnConstraints().add(new ColumnConstraints(100)); // column 4

		// Load
		Label loadLbl = new Label(messageSource.getMessage("ch.sahits.game.openpatrician.display.scene.InGameOptionsScene.load", new Object[]{}, locale.getCurrentLocal()));
		grid.add(loadLbl, 0, 0);

		lodableGames = new OpenPatricianSpinner();
		lodableGames.selectedIndexProperty().setValue(0);
		lodableGames.getStyleClass().add("defaultTextSize24");
		lodableGames.setMaxWidth(200);
		grid.add(lodableGames, 1, 0);
		GridPane.setHalignment(lodableGames, HPos.LEFT);
		lodableGames.setOptions(FXCollections.observableArrayList(gameOptions.getSavegames()));
		lodableGames.setMaxWidth(250);
		OpenPatricianStoneButton loadBtn = new OpenPatricianStoneButton(messageSource.getMessage("ch.sahits.game.openpatrician.display.scene.InGameOptionsScene.load", new Object[]{}, locale.getCurrentLocal()));
		loadBtn.getStyleClass().add("defaultTextSize24");
		grid.add(loadBtn, 3, 0);
		GridPane.setHalignment(loadBtn, HPos.LEFT);
		loadBtn.setOnAction(createLoadAction());




		root.getChildren().add(grid);

		clientServerEventBus.register(this);
	}

	@Subscribe
	public void handleGameStateChange(GameStateChange stateChange) {
		if (stateChange.getStatusChange() == EGameStatusChange.GAME_INITIALISATION_COMPLETE) {
			logger.info("Load and replace the game");
			  gameOptions.load(lodableGames.getSelectedValue());
		}
		if (stateChange.getStatusChange() == EGameStatusChange.GAME_LOADED) {

			uiFactory.invalidate();

			final MainGameScene mainGameScene1 = uiFactory.getMainGameScene(getRoot().getWidth(), getRoot().getHeight());
			mainGameScene1.initializeGameView(null);
            Platform.runLater(() ->
                changeScene(mainGameScene1)
            );
			timerEventBus.post(new ResumeGame());
		}
	}


	private EventHandler<MouseEvent> createLoadAction() {
		return event -> {
			try {
				NewGame newGameDTO = NewGame.builder()
						.difficulty(Difficulty.CHANDLER)
						.firstName("FirstName")
						.lastName("Lastname")
						.hometown("BREMEN")
						.objective(EObjective.ENDLESS)
						.speed(EGameSpeed.SLOW)
						.startYear(1300)
						.male(true)
						.mapName("/standartHanseaticMap.xml")
						.environmentInitialisation(true) // environment is not yet initialized
						.build();
				// notify server
				// This will instanciate the UI, so it can be replaced properly
				gameDate.setStartYear(Date.DEFAULT_INVALID_START_YEAR); // will be changed
				final double width = getRoot().getWidth();
				final double height = getRoot().getHeight();
				logger.debug("Change into main game scene with dimensions: {}x{}", width, height);
				final MainGameScene mainGameScene = uiFactory.getMainGameScene(width, height);
				mainGameScene.initializeGameView(null); // force initialisation of MainGameView

				GameStateChange stateChange = new GameStateChange(EGameStatusChange.LOAD_GAME_INIT);
				stateChange.setStateChangeData(newGameDTO);
				clientServerEventBus.post(stateChange);

				// Request the view change
				while (gameDate.getStartYear() == null) {
					try {
						Thread.sleep(1);
					} catch (InterruptedException e) {
						logger.warn("Interrupted while waiting for the startyear to be set.");
					}
				}
			} catch (RuntimeException e) {
				logger.error("Failed to load new game", e);
			}


		};
	}
	@Override
	public void changeScene(OpenPatricianScene scene) {
		sceneChanger.changeScene(sceneChangeable, scene);
	}


}
