package ch.sahits.game.openpatrician.display.service;

import ch.sahits.game.graphic.image.IMapImageServiceFacade;
import ch.sahits.game.graphic.image.model.MapGrid;
import ch.sahits.game.openpatrician.clientserverinterface.model.PathInterpolatorMap;
import ch.sahits.game.openpatrician.clientserverinterface.model.VesselPositionUpdateData;
import ch.sahits.game.openpatrician.clientserverinterface.service.MapService;
import ch.sahits.game.openpatrician.clientserverinterface.service.PathInterpolator;
import ch.sahits.game.openpatrician.engine.EngineConfiguration;
import ch.sahits.game.openpatrician.engine.sea.IPathConverter;
import ch.sahits.game.openpatrician.javafx.model.BezierCurveControls;
import ch.sahits.game.openpatrician.javafx.model.BezierPath;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.sea.TravellingVessels;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import javafx.geometry.Point2D;
import javafx.scene.paint.Color;
import javafx.scene.shape.CubicCurveTo;
import javafx.scene.shape.MoveTo;
import javafx.scene.shape.Path;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * Create a Bezier path based on a set of points.
 * The implementation is based on work described in
 * http://devmag.org.za/2011/06/23/bzier-path-algorithms/
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Jan 05, 2016
 */
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class BezierPathConverter implements IPathConverter {
    private final Logger logger = LogManager.getLogger(getClass());
    @Autowired
    private IMapImageServiceFacade mapImageService;
    @Autowired
    private TravellingVessels vessels;
    @Autowired
    private PathInterpolatorMap interpolators;
    @Autowired
    private Date date;
    @Autowired
    private MapService mapService;
    @Override
    public Optional<Path> createPath(INavigableVessel vessel, List<Point2D> pointedPath, double scale) {
        List<Point2D> reducedList = reduceLinePoints(pointedPath);
        List<Point2D> controlPoints = interpolate(applyScale(reducedList, scale), 0.5f);
        BezierPath pathModel = convertToPathModel(controlPoints);
        Path path = null;
        if (vessel.getOwner() instanceof IHumanPlayer) {
            path = new Path();
            path.setStroke(Color.RED);

            path.setStrokeWidth(2);
            if (!pathModel.getBezierCurves().isEmpty()) {
                Point2D p = pathModel.getStartpoint();
                MoveTo moveTo = new MoveTo();
                moveTo.setX(p.getX());
                moveTo.setY(p.getY());
                path.getElements().add(moveTo);
                for (BezierCurveControls segment : pathModel.getBezierCurves()) {
                    CubicCurveTo cubicTo = new CubicCurveTo();
                    cubicTo.setControlX1(segment.getControlPoint1().getX());
                    cubicTo.setControlY1(segment.getControlPoint1().getY());
                    cubicTo.setControlX2(segment.getControlPoint2().getX());
                    cubicTo.setControlY2(segment.getControlPoint2().getY());
                    cubicTo.setX(segment.getPoint().getX());
                    cubicTo.setY(segment.getPoint().getY());
                    path.getElements().add(cubicTo);
                }
            }
        }
        logger.trace("Add travelling vessel {} of {} {}", vessel.getName(), vessel.getOwner().getName(), vessel.getOwner().getLastName());
        Optional<Path> optPath = Optional.ofNullable(path);
        vessels.addVessel(vessel, optPath, reducedList);
        final PathInterpolator pathInterpolator = new PathInterpolator(reducedList);
        long duration = calculateDuration(vessel, pathInterpolator, 1);
        long nbTicks = duration / EngineConfiguration.CLOCK_TICK_INTERVALL_MS;
        double fractionPerTick = 1.0/nbTicks;
        pathInterpolator.setTravelFractionPerTick(fractionPerTick);
        VesselPositionUpdateData data = new VesselPositionUpdateData(pathInterpolator,pointedPath.get(0), pointedPath.get(pointedPath.size() - 1), duration);
        data.setAnimationStartMS(System.currentTimeMillis());

        interpolators.put(vessel, data);
        logger.trace("Add path interpolator from {} to {}, nb interpolators={}, update fraction per tick={}", reducedList.get(0), reducedList.get(reducedList.size() - 1), interpolators.size(), fractionPerTick);
        return optPath;
    }

    @Override
    public long calculateDuration(INavigableVessel vessel, PathInterpolator interpolator, double fraction) {
        double speedKmPerH = vessel.getCurrentSpeed();
        double distanceInPixels = interpolator.getTotalLength() * fraction;
        double distanceInKm = mapService.convertToDistenceInKm(distanceInPixels);
        double tickUpdateInHours = date.getTickUpdate()/60.0; // tick update is in minutes
        double inGameAnimationDurationInHours = distanceInKm/speedKmPerH;
        long nbTicks = Math.round(Math.ceil(inGameAnimationDurationInHours / tickUpdateInHours));
        logger.debug("Distance of {}km can be traveled in {}h at {}km/h, which results in {} clock ticks or {}ms, fraction to travel={}",distanceInKm, inGameAnimationDurationInHours, speedKmPerH, nbTicks, EngineConfiguration.CLOCK_TICK_INTERVALL_MS * nbTicks, fraction);
        return EngineConfiguration.CLOCK_TICK_INTERVALL_MS * nbTicks;
    }

    private List<Point2D> applyScale(List<Point2D> pointedPath, double scale) {
        ArrayList<Point2D> list = new ArrayList<>();
        for (Point2D p : pointedPath) {
            list.add(new Point2D(p.getX()*scale, p.getY()*scale));
        }
        return list;
    }

    /**
     * Calculate the controlpoints for a given list of points on the curve.
     * @param segmentPoints List of points that make up the curve
     * @param scale to be used for the length of the handles
     * @return List of controlpoints including the points on the line.
     */
    List<Point2D> interpolate(List<Point2D> segmentPoints, float scale) {
        List<Point2D> controlPoints = new ArrayList<>();

        if (segmentPoints.size() < 2) {
            return controlPoints;
        }

        for (int i = 0; i < segmentPoints.size(); i++) {
            if (i == 0) { // is first
                Point2D p1 = segmentPoints.get(i);
                Point2D p2 = segmentPoints.get(i + 1);

                Point2D tangent = p2.subtract(p1);
                Point2D q1 = p1.add(tangent.multiply(scale));

                controlPoints.add(p1);
                controlPoints.add(q1);
            } else if (i == segmentPoints.size() - 1) { //last
                Point2D p0 = segmentPoints.get(i - 1);
                Point2D p1 = segmentPoints.get(i);
                Point2D tangent = p1.subtract(p0);
                Point2D q0 = p1.subtract(tangent.multiply(scale));

                controlPoints.add(q0);
                controlPoints.add(p1);
            } else {
                Point2D p0 = segmentPoints.get(i - 1);
                Point2D p1 = segmentPoints.get(i);
                Point2D p2 = segmentPoints.get(i + 1);
                Point2D tangent = p2.subtract(p0).normalize();
                Point2D q0 = p1.subtract(tangent.multiply(scale * p1.subtract(p0).magnitude()));
                Point2D q1 = p1.add(tangent.multiply(scale * p2.subtract(p1).magnitude()));

                controlPoints.add(q0);
                controlPoints.add(p1);
                controlPoints.add(q1);
            }
        }

        return controlPoints;
    }

    private BezierPath convertToPathModel(List<Point2D> points) {
        BezierPath bezierPath = new BezierPath(points.get(0));
        int i = 1;
        while (i < points.size()) {
            Point2D outgoingControl = points.get(i++);
            Point2D incomingControl = points.get(i++);
            Point2D segmentEndPoint = points.get(i++);
            BezierCurveControls segment = new BezierCurveControls(segmentEndPoint, outgoingControl, incomingControl);
            bezierPath.addSegment(segment);
        }
        return bezierPath;
    }

    /**
     * Reduce the number of points that are needed to describe the path.
     * Points that do not change the direction of the path can be eliminated if:
     * <ul>
     *     <li>The distance to the last point left is larger than the distance to the shore</li>
     * </ul>
     * @param initialPoints initial point of the line
     * @return list of points making up the path
     */
    @Override
    public List<Point2D> reduceLinePoints(List<Point2D> initialPoints) {
        Map<Point2D, Distances> distance = new HashMap<>();
        for (int i = 0; i < initialPoints.size() - 1; i++) {
            Point2D currentPoint = initialPoints.get(i);
            Point2D nextPoint = initialPoints.get(i + 1);
            Point2D v1 = nextPoint.subtract(currentPoint);
            double max = currentPoint.distance(nextPoint);
            for (int j = i+1; j < initialPoints.size(); j++) {
                Point2D next = initialPoints.get(j);
                Point2D v2 = next.subtract(currentPoint);
                if (v1.angle(v2) < 1) {
                    // line continues in the same direction
                    max = currentPoint.distance(next);
                }
            }
            double shoreDistance = MapGrid.CHECK_DISTANCE;
            try {
                shoreDistance = mapImageService.distanceToShore(currentPoint, max);
            } catch (IOException e) {
                logger.warn("Failed to calculate the distance to the shore", e);
            }
            final Distances value = new Distances(shoreDistance, max);
            distance.put(currentPoint, value);
        }
        distance.put(initialPoints.get(initialPoints.size()-1), new Distances(Double.MAX_VALUE, Double.MAX_VALUE));
        LinkedList<Point2D> reducedList = new LinkedList<>();
        for (int i = 0; i < initialPoints.size(); i++) {
            Point2D currentPoint = initialPoints.get(i);
            boolean isCorner;
            if (i == 0 || i == initialPoints.size() - 1) {
                isCorner = true;
            } else {
                Point2D before = initialPoints.get(i - 1);
                Point2D next = initialPoints.get(i + 1);
                Point2D v1 = currentPoint.subtract(before);
                Point2D v2 = next.subtract(currentPoint);
                isCorner = v1.angle(v2) >= 1;
            }
            if (isCorner) {
                reducedList.add(currentPoint);
            } else {
                Distances dist = distance.get(currentPoint);
                double distanceToLastPoint = reducedList.getLast().distance(currentPoint);
                if (distanceToLastPoint >= dist.distanceToShore && dist.distanceToShore < dist.distanceToCornor) {
                    reducedList.add(currentPoint);
                }
            }

        }

        return reducedList;
    }



    private static class Distances {
        private final double distanceToShore;
        private final double distanceToCornor;

        public Distances(double distanceToShore, double distanceToCornor) {
            this.distanceToShore = distanceToShore;
            this.distanceToCornor = distanceToCornor;
        }

    }
}
