package ch.sahits.game.openpatrician.display;

import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.display.model.CityPlayerProxyJFX;
import ch.sahits.game.openpatrician.event.data.NewGameClient;
import ch.sahits.game.openpatrician.event.data.SwitchCity;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.PlayerList;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.initialisation.StartNewGameBean;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.DependentInitialisation;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.ObjectPropertyType;
import ch.sahits.game.openpatrician.utilities.annotation.OptionalType;
import ch.sahits.game.openpatrician.utilities.service.IPostLoadOperation;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleObjectProperty;
import lombok.Getter;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.List;
import java.util.Optional;

/**
 * This is the state that represents the actual state of what the client currently displayes.
 * This differeciates between the actual states:
 * <ul>
 *     <li>City</li>
 *     <li>Map</li>
 *     <li>Sea</li>
 * </ul>
 * As well for the city there is additional information on which city.
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Feb 15, 2014
 */
@Component
@Lazy
@DependentInitialisation(StartNewGameBean.class)
@ClassCategory({EClassCategory.SINGLETON_BEAN})
public class ClientViewState implements IPostLoadOperation {
    @XStreamOmitField
    private final Logger logger = LogManager.getLogger(getClass());

    /** Main state */
    @ObjectPropertyType(EViewState.class)
    private ObjectProperty<EViewState> state  = new SimpleObjectProperty<>(EViewState.CITY);
    /** proxy for the city is only present when the state is city */
    @Getter
    @OptionalType(ICityPlayerProxyJFX.class)
    private Optional<ICityPlayerProxyJFX> currentCityProxy;
    /** The player behind the client */
    @Getter
    private IHumanPlayer player;

    @Autowired
    @Qualifier("serverClientEventBus")
    @XStreamOmitField
    private AsyncEventBus clientServerEventBus;
    @Autowired
    @Qualifier("clientEventBus")
    @XStreamOmitField
    protected AsyncEventBus clientEventBus;
    @Autowired
    @XStreamOmitField
    private ApplicationContext context;

    @PostConstruct
    private void initEventBus() {
        clientServerEventBus.register(this);
        clientEventBus.register(this);
        try {
            CheatKeyEventListener listener = context.getBean(CheatKeyEventListener.class);
            listener.setViewState(this);
            listener.setContext(context);
        } catch (Exception e) {
            /// the listener may not be available
        }
    }
    @PreDestroy
    private void unregister() {
        logger.debug("Unregister event bus in PreDestroy");
        clientServerEventBus.unregister(this);
        clientEventBus.unregister(this);
    }

    /**
     * Initialize the player that goes with this client. The player does not change later on,
     * @param newGameDTO parameter object for the new game
     */
    @Subscribe
    public void initializeState(NewGameClient newGameDTO) {
        this.player = newGameDTO.getPlayer();
        state.setValue(EViewState.CITY);
        IShip activeShip = null;
        for (IShip ship : player.getFleet()) {
            if (ship.isAvailable()) {
                activeShip = ship;
                break;
            }
        }
        ICityPlayerProxyJFX proxy = (ICityPlayerProxyJFX) context.getBean("cityPlayerProxy", player.getHometown(), player, activeShip);
        currentCityProxy = Optional.of(proxy);
    }
    @Subscribe
    public void handleSwitchToCity(SwitchCity event) {
        if (currentCityProxy.isPresent() && !event.getToCity().equals(currentCityProxy.get().getCity())) {
            final ICityPlayerProxyJFX proxy = currentCityProxy.get();
            final ICity city = event.getToCity();
            proxy.setCity(city);
            // Update the ships for that city
            proxy.getPlayersNavalVessels().clear();
            final List<INavigableVessel> ships = proxy.getPlayer().findShips(city);
            proxy.getPlayersNavalVessels().addAll(ships);
            if (!ships.isEmpty()) {
                proxy.activateShip(ships.get(0));
            }
        }
    }

    public EViewState getState() {
        return state.get();
    }

    public ObjectProperty<EViewState> stateProperty() {
        return state;
    }

    public void setState(EViewState state) {
        this.state.set(state);
    }

    @Override
    public void postLoad() {
        state.setValue(EViewState.CITY);
        PlayerList players = context.getBean(PlayerList.class);
        for (IPlayer curPlayer : players) {
            if (curPlayer instanceof IHumanPlayer) {
                this.player = (IHumanPlayer) curPlayer;
                break;
            }
        }
        CityPlayerProxyJFX proxy = (CityPlayerProxyJFX) context.getBean("cityPlayerProxy", player.getHometown(), player, null);
        currentCityProxy = Optional.of(proxy);
        proxy.postLoad();
    }
}
