package ch.sahits.game.openpatrician.display.event.handler.impl;

import ch.sahits.game.openpatrician.display.event.data.PatrolInternalState;
import ch.sahits.game.openpatrician.display.event.task.ClientTaskFactory;
import ch.sahits.game.openpatrician.event.data.ShipEntersPortEvent;
import ch.sahits.game.openpatrician.event.handler.ShipEntersPortEventListener;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.DisplayMessage;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import com.google.common.eventbus.AsyncEventBus;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.time.LocalDateTime;
import java.util.Optional;
import java.util.Random;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Feb 21, 2014
 */
@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
class NextPatrolStation extends ShipEntersPortEventListener {
    /** Next station may be null if it is the last */
    private final PatrolInternalState state;
    private final IPlayer player;
    private final LocalDateTime deadline;
    private final AttackListener atackListener;
    @Autowired
    private Date date;
    @Autowired
    @XStreamOmitField
    private Random rnd;
    @Autowired
    @Qualifier("clientEventBus")
    @XStreamOmitField
    private AsyncEventBus clientEventBus;
    @Autowired
    @XStreamOmitField
    private ClientTaskFactory taskFactory;
    @Autowired
    @XStreamOmitField
    private Locale locale;
    @Autowired
    @XStreamOmitField
    private MessageSource messageSource;

    /**
     * Create the next patrol station listener. This listener adds itself to the event queue and removes the first city from the
     * patrol state.
     * @param ship on patrol
     * @param city next city
     * @param state patrol state
     * @param player current player (ship owner)
     * @param deadline until the ship has to arrive
     * @param atackListener attack listener
     */
    public NextPatrolStation(IShip ship, ICity city, PatrolInternalState state, IPlayer player, LocalDateTime deadline, AttackListener atackListener) {
        super(ship, city);
        this.state = state;
        this.player = player;
        this.deadline = deadline;
        this.atackListener = atackListener;
        state.removeFirstCity();

    }
    @PostConstruct
    private void initEventBus() {
        clientServerEventBus.register(this);
    }
    @PreDestroy
    private void unregisterEventBus() { clientServerEventBus.unregister(this);}
    @Override
    public void handleShipEntersPort(ShipEntersPortEvent event) {
        super.handleShipEntersPort(event);
        LocalDateTime now = date.getCurrentDate();
        if (now.isAfter(deadline)) {
            clientEventBus.post(new DisplayMessage("ch.sahits.game.event.handler.impl.NextCourierPatrolStation.unreliable"));
            clientServerEventBus.unregister(this);
            clientServerEventBus.unregister(atackListener);
        }
    }


    @Override
    public boolean execute(ICity city) {
        Optional<ICity> nextCity = state.nextDestination();
        if (!nextCity.isPresent()){ // last leg
            int premium = state.getPremium()+state.getNumberOfAttacks()*state.getBonusPerAttack();
            if (player instanceof IHumanPlayer) {
                player.getCompany().updateCash(premium);
            } else {
                player.getCompany().updateCashDirectly(premium);
            }
            clientEventBus.post(new DisplayMessage("ch.sahits.game.event.handler.impl.NextPatrolStation.premium", premium));
            clientServerEventBus.unregister(atackListener);
            unregisterEventBus();
            clientServerEventBus.unregister(atackListener);
        } else { // next leg
            ICity next = nextCity.get();
            state.removeFirstCity();
            int minutes = rnd.nextInt(600);

            LocalDateTime postMessage = date.getCurrentDate().plusMinutes(minutes);
            taskFactory.getPostponedDisplayMessage(postMessage, "ch.sahits.game.event.handler.impl.NextCourierPatrolStation.nextStation", next.getName());
            LocalDateTime deadline = date.getCurrentDate().plusDays(6);
            new NextPatrolStation((IShip) getShip(), state.nextDestination().get(), state, player, deadline, atackListener);
        }
        return true;
    }}
