package ch.sahits.game.openpatrician.display.event.handler.impl;


import ch.sahits.game.openpatrician.clientserverinterface.service.DialogTemplateFactory;
import ch.sahits.game.openpatrician.clientserverinterface.service.DialogTemplateParameterSupplier;
import ch.sahits.game.openpatrician.clientserverinterface.service.EDialogTemplateType;
import ch.sahits.game.openpatrician.clientserverinterface.service.LinearDeadlinePremiumCalculator;
import ch.sahits.game.openpatrician.event.handler.ShipEntersPortEventListener;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.DisplayTemplateMessage;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.event.TargetedEvent;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.ui.DialogTemplate;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.time.LocalDateTime;

/**
 * The passenger leaves the ship when it arrives in the city and the player gets
 * a certain amount of money.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Feb 19, 2013
 *
 */
@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
class PassengerLeavesShipOnArrival extends ShipEntersPortEventListener {
	@Autowired
	@XStreamOmitField
	private DialogTemplateFactory dialogTemplateFactory;
	@Autowired
	private Date date;
	private final IPlayer shipsOwner;
	private final LocalDateTime deadline;
	private final EDialogTemplateType dialogTemplateType;
	private final DialogTemplateParameterSupplier dialogParameterSupplier;
	private final LinearDeadlinePremiumCalculator premiumCalculator;
	public PassengerLeavesShipOnArrival(IShip ship, ICity city, IPlayer shipsOwner, LocalDateTime deadline, EDialogTemplateType dialogType, DialogTemplateParameterSupplier dialogParameterSupplier, LinearDeadlinePremiumCalculator premiumCalculator) {
		super(ship, city);
		this.shipsOwner = shipsOwner;
		this.deadline = deadline;
		this.dialogTemplateType = dialogType;
		this.dialogParameterSupplier = dialogParameterSupplier;
		this.premiumCalculator = premiumCalculator;
	}
    @PostConstruct
    private void initializeEventBus() {
        clientServerEventBus.register(this);
    }
	@PreDestroy
	private void unregisterEventBus() { clientServerEventBus.unregister(this);}
	@Override
	public boolean execute(ICity city) {
		if (ignoreEventCity()) {
			// time has passed and passenger will leave.
			EDialogTemplateType templateType = resolveType(dialogTemplateType, false);
			DialogTemplate template = dialogTemplateFactory.createDialogTemplate(templateType, dialogParameterSupplier);
			DisplayTemplateMessage message = new DisplayTemplateMessage(getMessageBoardTitleKey(templateType), template, getMessageBoardTitleParameters(templateType, city));
			TargetedEvent tagetDisplayMsg = new TargetedEvent((IHumanPlayer)shipsOwner, message);
			clientServerEventBus.post(tagetDisplayMsg);
		} else {
			// Successful arrival at destination
			if (shipsOwner instanceof IHumanPlayer) {
				shipsOwner.getCompany().updateCash(premiumCalculator.get());
			} else {
				shipsOwner.getCompany().updateCashDirectly(premiumCalculator.get());
			}
			EDialogTemplateType templateType = resolveType(dialogTemplateType, true);
			DialogTemplate template = dialogTemplateFactory.createDialogTemplate(templateType, dialogParameterSupplier);
			DisplayTemplateMessage message = new DisplayTemplateMessage(getMessageBoardTitleKey(templateType), template, getMessageBoardTitleParameters(templateType, city));
			TargetedEvent tagetDisplayMsg = new TargetedEvent((IHumanPlayer)shipsOwner, message);
			clientServerEventBus.post(tagetDisplayMsg);

		}
		((IShip) getShip()).leavePassenger();
		unregisterEventBus();
		return true;
	}

	private Object[] getMessageBoardTitleParameters(EDialogTemplateType templateType, ICity city) {
		switch (templateType) {
			case ESCORT_LEAVES_SHIP_AFTER_DEADLINE:
			case ESCORT_LEAVES_SHIP_BEFORE_DEADLINE:
			case FUGITIVE_LEAVES_SHIP_AFTER_DEADLINE:
			case FUGITIVE_LEAVES_SHIP_BEFORE_DEADLINE:
			case TRAVELER_LEAVES_SHIP_AFTER_DEADLINE:
			case TRAVELER_LEAVES_SHIP_BEFORE_DEADLINE:
				return new Object[]{city.getName()};
			default:
				return new Object[0];
		}
	}

	private String getMessageBoardTitleKey(EDialogTemplateType templateType) {
		switch (templateType) {
			case ESCORT_LEAVES_SHIP_AFTER_DEADLINE:
				return "ch.sahits.game.openpatrician.display.dialog.TavernEscortDialog.dialogTemplate.leaving.unsuccessful.messageboard.title";
			case ESCORT_LEAVES_SHIP_BEFORE_DEADLINE:
				return "ch.sahits.game.openpatrician.display.dialog.TavernEscortDialog.dialogTemplate.leaving.successful.messageboard.title";
			case FUGITIVE_LEAVES_SHIP_AFTER_DEADLINE:
				return "ch.sahits.game.openpatrician.display.dialog.TavernFugitiveDialog.dialogTemplate.leaving.unsuccessful.messageboard.title";
			case FUGITIVE_LEAVES_SHIP_BEFORE_DEADLINE:
				return "ch.sahits.game.openpatrician.display.dialog.TavernFugitiveDialog.dialogTemplate.leaving.successful.messageboard.title";
			case TRAVELER_LEAVES_SHIP_AFTER_DEADLINE:
				return "ch.sahits.game.openpatrician.display.dialog.TavernTravelerDialog.dialogTemplate.leaving.unsuccessful.messageboard.title";
			case TRAVELER_LEAVES_SHIP_BEFORE_DEADLINE:
				return "ch.sahits.game.openpatrician.display.dialog.TavernTravelerDialog.dialogTemplate.leaving.successful.messageboard.title";
			default:
				throw new IllegalArgumentException("Unsupported template type: "+templateType);
		}
	}

	/**
	 * If the deadline has passed, the passenger will leave the ship in the next port not the destination city.
	 * @return true if the the current date is after the <code>deadline</code>
	 */
	@Override
	protected boolean ignoreEventCity() {
		return date.getCurrentDate().isAfter(deadline);
	}

	private EDialogTemplateType resolveType(EDialogTemplateType baseType, boolean successful) {
		switch (baseType) {
			case COURIER_SERVICE_FINISHED:
				if (successful) {
					return EDialogTemplateType.COURIER_SERVICE_FINISHED_SUCCESSFUL;
				} else {
					return EDialogTemplateType.COURIER_SERVICE_FINISHED_UNSUCCESSFUL;
				}
			case ESCORT_LEAVES_SHIP:
				if (successful) {
					return EDialogTemplateType.ESCORT_LEAVES_SHIP_BEFORE_DEADLINE;
				} else {
					return EDialogTemplateType.ESCORT_LEAVES_SHIP_AFTER_DEADLINE;
				}
			case FUGITIVE_LEAVES_SHIP:
				if (successful) {
					return EDialogTemplateType.FUGITIVE_LEAVES_SHIP_BEFORE_DEADLINE;
				} else {
					return EDialogTemplateType.FUGITIVE_LEAVES_SHIP_AFTER_DEADLINE;
				}
			case TRAVELER_LEAVES_SHIP:
				if (successful) {
					return EDialogTemplateType.TRAVELER_LEAVES_SHIP_BEFORE_DEADLINE;
				} else {
					return EDialogTemplateType.TRAVELER_LEAVES_SHIP_AFTER_DEADLINE;
				}
			default:
				throw new IllegalArgumentException("Unhandled dialog template type: "+baseType+" for success="+successful);
		}
	}
}
