package ch.sahits.game.openpatrician.display.javafx.control;

import ch.sahits.game.graphic.image.IDataImageLoader;
import ch.sahits.game.graphic.image.SpriteAnimation;
import ch.sahits.game.graphic.image.SpriteAnimationFactory;
import ch.sahits.game.openpatrician.clientserverinterface.model.factory.GameFactory;
import ch.sahits.game.openpatrician.display.ClientViewState;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.IGame;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.utilities.IRebinabable;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import javafx.animation.RotateTransition;
import javafx.beans.property.SimpleStringProperty;
import javafx.beans.property.StringProperty;
import javafx.beans.value.ChangeListener;
import javafx.geometry.VPos;
import javafx.scene.Group;
import javafx.scene.control.Label;
import javafx.scene.image.ImageView;
import javafx.scene.paint.Color;
import javafx.scene.shape.Rectangle;
import javafx.scene.text.Font;
import javafx.scene.text.Text;
import javafx.util.Duration;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;
/**
 * Status control displaying the city name, population, date and cash.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Nov 2, 2013
 *
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class GameStatus extends Group implements IRebinabable {
    private final Logger logger = LogManager.getLogger(getClass());

	private Text cashTxt;
	private Text dateTxt;
    @Autowired
    private Date gameDate;
    @Autowired
    private GameFactory gameFactory;

    @Autowired
    private SpriteAnimationFactory spriteAnimFactory;
    @Autowired
    @Qualifier("xmlImageLoader")
    private IDataImageLoader imageLoader;
    @Autowired
    private ClientViewState clientViewState;
    private ImageView timeGraphic;
    private ChangeListener<String> dateChangeListener;
    private ChangeListener<String> cashChangeListener;
    private Label time;
    private long lastDayChange = 0;

    public GameStatus() {
		setManaged(false);
        getStylesheets().add(getClass().getResource(getClass().getSimpleName()+".css").toExternalForm());
        cashTxt = new Text();
        dateTxt = new Text();

	}
    @PostConstruct
    private void initializeControl() {

        final Rectangle ph = new Rectangle(250, 50, Color.TRANSPARENT);
        final int spacing = 10;

        SimpleStringProperty date = createDateProperty();
        cashTxt.getStyleClass().addAll("Text");
        dateTxt.getStyleClass().addAll("Text");

        final ImageView cashPile = spriteAnimFactory.createImageViewForSprite("sprites/coin-pile");
        cashPile.setScaleY(0.5);
        cashPile.setScaleX(0.5);

        timeGraphic = new ImageView(imageLoader.getImage("icons/64/hourglass"));
        time = new Label();
        time.setGraphic(timeGraphic);
        time.setScaleX(0.5);
        time.setScaleY(0.5);
        time.setOnMouseReleased(event -> {
                IGame game = gameFactory.getGame();
                game.fastForward();
        });

        cashPile.setLayoutY(-14);
        cashTxt.setLayoutX(cashPile.getLayoutX() + 38 + spacing);
        cashTxt.setTextOrigin(VPos.BASELINE);
        cashTxt.setLayoutY(30);
        cashChangeListener = (observable, oldValue, newValue) -> {
            SpriteAnimation animation = spriteAnimFactory.createAnimation(cashPile, "sprites/coin-pile", true);
            animation.play();
        };
        cashTxt.textProperty().addListener(cashChangeListener);

        time.setLayoutY(-8);
        time.setLayoutX(ph.getWidth() - spacing - 32);
        dateTxt.setLayoutX(time.getLayoutX() - spacing - dateTxt.getBoundsInLocal().getWidth());
        dateTxt.setTextOrigin(VPos.BASELINE);
        dateTxt.setLayoutY(30);
        dateTxt.boundsInLocalProperty().addListener((observableValue, oldBounds, newBounds) -> {
            if (oldBounds.getWidth() != newBounds.getWidth()) {
                dateTxt.setLayoutX(time.getLayoutX() - spacing - newBounds.getWidth());
            }
        });
        dateTxt.textProperty().bind(date);
        dateTxt.textProperty().addListener((observable, oldText, newText) -> {
            long now = System.currentTimeMillis();
            long took = now - lastDayChange;
            logger.info("Date text changed {}->{} took {}ms", oldText, newText, took);
            lastDayChange = now;

        });
        dateChangeListener = (observableValue, s, t1) -> {
            RotateTransition rotation = new RotateTransition(Duration.millis(1200), timeGraphic);
            rotation.setByAngle(-360.0);
            rotation.setCycleCount(1);
            rotation.setAutoReverse(false);

            rotation.play();
        };
        dateTxt.textProperty().addListener(dateChangeListener);

        getChildren().addAll(ph, cashPile, cashTxt, dateTxt, time);

    }

    private SimpleStringProperty createDateProperty() {
        SimpleStringProperty date = new SimpleStringProperty(this, "date", "");
        boolean dateSet = false;
        while (!dateSet) {
            try {
                bindDate(date);
                dateSet = true;
            } catch (Exception e) {
                logger.debug("Failed to bind date, try again");
                try {
                    Thread.sleep(50);
                } catch (InterruptedException e1) {
                    logger.warn("Interrupped while waiting for date initialisation");
                }
            }
        }
        if (date.get().equals("") || date.get() == null) {
            logger.error("date property not initialized");
        }
        return date;
    }

    private void bindDate(SimpleStringProperty date) {
        date.bind(gameDate.dayDateBinding());
        gameDate.dayDateBinding().addListener((observable, oldValue, newValue) -> {
            date.unbind();
            date.bind(gameDate.dayDateBinding());
        });
    }

    public void bindToPlayer(IPlayer player) {
        cashTxt.textProperty().bind(player.getCompany().cashProperty().asString());
    }

	
	public void setFont(Font f) {
		cashTxt.setFont(f);
		dateTxt.setFont(f);
	}

    @Override
    public void rebind() {
        cashTxt.textProperty().removeListener(cashChangeListener);
        cashTxt.textProperty().unbind();
        cashTxt.textProperty().addListener(cashChangeListener);
        gameDate.invalidate();
        dateTxt.textProperty().removeListener(dateChangeListener);
        dateTxt.textProperty().unbind();
        StringProperty date = createDateProperty();
        dateTxt.textProperty().bind(date);
        dateTxt.textProperty().addListener(dateChangeListener);
    }
}
