package ch.sahits.game.openpatrician.display.scene;

import ch.sahits.game.event.data.PauseGame;
import ch.sahits.game.event.data.ResumeGame;
import ch.sahits.game.graphic.image.IFontLoader;
import ch.sahits.game.openpatrician.clientserverinterface.model.factory.GameFactory;
import ch.sahits.game.openpatrician.display.SceneChangeable;
import ch.sahits.game.openpatrician.display.service.GameOptionsService;
import ch.sahits.game.openpatrician.display.service.UIFactory;
import ch.sahits.game.openpatrician.event.EGameStatusChange;
import ch.sahits.game.openpatrician.event.GameStateChange;
import ch.sahits.game.openpatrician.event.data.NewGame;
import ch.sahits.game.openpatrician.javafx.OpenPatricianScene;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianSlider;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianSpinner;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianStoneButton;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianWoodenTextInput;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.Difficulty;
import ch.sahits.game.openpatrician.model.EGameSpeed;
import ch.sahits.game.openpatrician.model.EObjective;
import ch.sahits.game.openpatrician.model.IGame;
import ch.sahits.game.openpatrician.sound.ITrackPlayer;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.UniquePrototype;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import javafx.application.Platform;
import javafx.collections.FXCollections;
import javafx.event.EventHandler;
import javafx.geometry.HPos;
import javafx.geometry.Pos;
import javafx.scene.control.Label;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.ColumnConstraints;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.StackPane;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.util.StringUtils;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

/**
 * Options that can be accessed from within the game.
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Dec 27, 2014
 */
@UniquePrototype
@ClassCategory(EClassCategory.PROTOTYPE_BEAN)
public class InGameOptionsScene extends OpenPatricianScene {
    private final Logger logger = LogManager.getLogger(getClass());
    @Autowired
    private UIFactory uiFactory;
    @Autowired
    private GameFactory gameFactory;
    @Autowired
    private Locale locale;
    @Autowired
    private MessageSource messageSource;
    @Autowired
    private GameOptionsService gameOptions;
    @Autowired
    private Date date;
    @Autowired
    private IFontLoader fontLoader;
    @Autowired
    private ITrackPlayer soundPlayer;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    @Qualifier("timerEventBus")
    private AsyncEventBus timerEventBus;

    private IGame game;
    // Need to inject the font as for custom controls it is not stylable
    private SceneChangeable sceneChangeable;
    private OpenPatricianSlider speed;
    private EGameSpeed currentSpeed;
    private OpenPatricianSpinner lodableGames;
    private OpenPatricianSpinner saveableGames;
    private OpenPatricianWoodenTextInput newSaveGame;

    public InGameOptionsScene() {
        super(new StackPane());
        getRoot().getStylesheets().add(this.getClass().getResource(getStyleSheetFilename()).toExternalForm());
    }


    public SceneChangeable getSceneChangeable() {
        return sceneChangeable;
    }

    public void setSceneChangeable(SceneChangeable sceneChangeable) {
        this.sceneChangeable = sceneChangeable;
    }
    private String getStyleSheetFilename() {
        return "newGame.css";
    }
    @PreDestroy
    private void unregister() {
        clientServerEventBus.unregister(this);
    }
    @PostConstruct
    private void createControls() {
        clientServerEventBus.register(this);
        game = gameFactory.getGame();
        currentSpeed = game.getGameSpeed();
        StackPane root = (StackPane) getRoot();
        root.setId("IngameOptionsStackPane");

        GridPane grid = new GridPane();
        grid.getStyleClass().add("grid");
        grid.setAlignment(Pos.CENTER);
        grid.getColumnConstraints().add(new ColumnConstraints(200)); // column 1
        grid.getColumnConstraints().add(new ColumnConstraints(300)); // column 2
        grid.getColumnConstraints().add(new ColumnConstraints(300)); // column 3
        grid.getColumnConstraints().add(new ColumnConstraints(100)); // column 4

        // Load
        Label loadLbl = new Label(messageSource.getMessage("ch.sahits.game.openpatrician.display.scene.InGameOptionsScene.load", new Object[]{}, locale.getCurrentLocal()));
        loadLbl.getStyleClass().add("defaultTextSize24");
        grid.add(loadLbl, 0, 0);

        lodableGames = new OpenPatricianSpinner();
        lodableGames.setMaxWidth(200);
        lodableGames.selectedIndexProperty().setValue(0);
        grid.add(lodableGames, 1, 0);
        GridPane.setHalignment(lodableGames, HPos.LEFT);
        lodableGames.setMaxWidth(250);
        OpenPatricianStoneButton loadBtn = new OpenPatricianStoneButton(messageSource.getMessage("ch.sahits.game.openpatrician.display.scene.InGameOptionsScene.load", new Object[]{}, locale.getCurrentLocal()));
        loadBtn.getStyleClass().add("defaultTextSize24");
        loadBtn.setOnAction(createLoadAction());
        grid.add(loadBtn, 3, 0);
        GridPane.setHalignment(loadBtn, HPos.LEFT);

        // Save
        Label saveLbl = new Label(messageSource.getMessage("ch.sahits.game.openpatrician.display.scene.InGameOptionsScene.save", new Object[]{}, locale.getCurrentLocal()));
        saveLbl.getStyleClass().add("defaultTextSize24");
        grid.add(saveLbl, 0, 1);

        saveableGames = new OpenPatricianSpinner();
        saveableGames.setMaxWidth(200);
        saveableGames.selectedIndexProperty().setValue(0);
        grid.add(saveableGames, 1, 1);
        GridPane.setHalignment(saveableGames, HPos.LEFT);
        saveableGames.setMaxWidth(250);

        newSaveGame = new OpenPatricianWoodenTextInput("");
        newSaveGame.setSize(25);
        grid.add(newSaveGame, 2, 1);

        OpenPatricianStoneButton saveBtn = new OpenPatricianStoneButton(messageSource.getMessage("ch.sahits.game.openpatrician.display.scene.InGameOptionsScene.save", new Object[]{}, locale.getCurrentLocal()));
        saveBtn.getStyleClass().add("defaultTextSize24");
        grid.add(saveBtn, 3, 1);
        GridPane.setHalignment(saveBtn, HPos.LEFT);

        // Speed
        Label gameSpeedLbl = new Label(messageSource.getMessage("ch.sahits.game.openpatrician.display.scene.NewGameScene.gameSpeed", new Object[]{}, locale.getCurrentLocal()));
        grid.add(gameSpeedLbl, 0, 2);

        speed = new OpenPatricianSlider(200);
        speed.getStyleClass().add("defaultTextSize24");
        speed.setValues(gameOptions.getGameSpeedList());
        int index = 0;
        currentSpeed = game.getGameSpeed();
        for (int i = 0; i < EGameSpeed.values().length; i++) {
            if (currentSpeed.equals(EGameSpeed.values()[i])) {
                index = i;
                break;
            }
        }
        speed.selectedIndexProperty().setValue(index);

        grid.add(speed, 1, 2);
        GridPane.setHalignment(speed, HPos.LEFT);

        // Quit
        OpenPatricianStoneButton quit = new OpenPatricianStoneButton(messageSource.getMessage("ch.sahits.game.openpatrician.display.scene.InGameOptionsScene.quit", new Object[]{}, locale.getCurrentLocal()));
        quit.getStyleClass().add("defaultTextSize24");
        quit.setOnAction(mouseEvent -> {
            try {
                mouseEvent.consume();
                soundPlayer.clearPlayList();
                Platform.exit();
                System.exit(0);
            } catch (RuntimeException e) {
                logger.error("Failed to quit game", e);
            }
        });
        grid.add(quit, 0, 6);
        GridPane.setHalignment(quit, HPos.LEFT);
        // Back
        OpenPatricianStoneButton back = new OpenPatricianStoneButton(messageSource.getMessage("ch.sahits.game.openpatrician.display.scene.InGameOptionsScene.back", new Object[]{}, locale.getCurrentLocal()));
        back.getStyleClass().add("defaultTextSize24");
        grid.add(back, 3, 6);
        GridPane.setHalignment(back, HPos.RIGHT);
        back.setOnAction(getBackEvent());

        saveBtn.setOnAction(createSaveAction(saveableGames, newSaveGame, back));
        initializeSavedGames();


        root.getChildren().add(grid);
    }

    /**
     * Initialize the lists that depend on the saved games.
     */
    private void initializeSavedGames() {
        lodableGames.setOptions(FXCollections.observableArrayList(gameOptions.getSavegames()));
        saveableGames.setOptions(FXCollections.observableArrayList(gameOptions.getSavegames()));
    }

    public void initializeState() {
        timerEventBus.post(new PauseGame());
        initializeSavedGames();
    }

    private EventHandler<MouseEvent> getBackEvent() {
        return event -> {
            try {
                EGameSpeed gameSpeed = EGameSpeed.values()[speed.getSelectedIndex()];
                EGameSpeed currentSpeed = game.getGameSpeed();
                if (gameSpeed != currentSpeed) {
                    game.setGameSpeed(gameSpeed);
                }
                MainGameScene mainGame = uiFactory.getMainGameScene(getRoot().getWidth(), getRoot().getHeight());
                getSceneChangeable().changeScene(mainGame);
                soundPlayer.play();
                newSaveGame.setText("");
                timerEventBus.post(new ResumeGame());
            } catch (RuntimeException e) {
                logger.error("Failed to go back to game", e);
            }
        };
    }

    private EventHandler<MouseEvent> createSaveAction(OpenPatricianSpinner saveableGames, OpenPatricianWoodenTextInput newSaveGame, OpenPatricianStoneButton back) {
        return event -> {
            try {
                String savegameName = newSaveGame.getText();
                if (StringUtils.isEmpty(savegameName)) {
                    savegameName = saveableGames.getSelectedValue();
                }
                if (savegameName != null) {
                    gameOptions.save(savegameName);
                } else {
                    logger.warn("Game was not saved as no file name was specified.");
                }
                back.getOnAction().handle(event);
            } catch (RuntimeException e) {
                logger.error("Failed to save game", e);
            }
        };
    }
    private EventHandler<MouseEvent> createLoadAction() {
        return event -> {
            try {
                NewGame newGameDTO = NewGame.builder()
                        .difficulty(Difficulty.CHANDLER)
                        .firstName("FirstName")
                        .lastName("Lastname")
                        .hometown("BREMEN")
                        .objective(EObjective.ENDLESS)
                        .speed(EGameSpeed.SLOW)
                        .startYear(1300)
                        .male(true)
                        .mapName("/standartHanseaticMap.xml")
                        .environmentInitialisation(false)  // environment is already loaded
                        .build();
                // notify server
                // This will instanciate the UI, so it can be replaced properly
                date.resetStartYear(Date.DEFAULT_INVALID_START_YEAR); // will be changed

                GameStateChange stateChange = new GameStateChange(EGameStatusChange.LOAD_GAME_INIT);
                stateChange.setStateChangeData(newGameDTO);
                clientServerEventBus.post(stateChange);

                // Request the view change
                while (date.getStartYear() == null) {
                    try {
                        Thread.sleep(1);
                    } catch (InterruptedException e) {
                        logger.warn("Interrupted while waiting for the startyear to be set.");
                    }
                }
            } catch (RuntimeException e) {
                logger.error("Failed to load new game", e);
            }
        };
    }
    @Subscribe
    public void handleGameStateChange(GameStateChange stateChange) {
        if (stateChange.getStatusChange() == EGameStatusChange.GAME_INITIALISATION_COMPLETE) {
            logger.info("Load and replace the game");
            gameOptions.load(lodableGames.getSelectedValue());
        }
        if (stateChange.getStatusChange() == EGameStatusChange.GAME_LOADED) {

            uiFactory.invalidate();

            final MainGameScene mainGameScene1 = uiFactory.getMainGameScene(getRoot().getWidth(), getRoot().getHeight());
            mainGameScene1.initializeGameView(null);
            Platform.runLater(() ->
                getBackEvent().handle(null)
            );
        }
    }


}
