package ch.sahits.game.openpatrician.display.scene;

import ch.sahits.game.graphic.data.polygon.Polygons;
import ch.sahits.game.graphic.loaders.IPolygonLoader;
import ch.sahits.game.openpatrician.display.service.UIFactory;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import ch.sahits.game.openpatrician.utilities.plugin.PluginLoader;
import javafx.geometry.Insets;
import javafx.scene.control.Label;
import javafx.scene.effect.PerspectiveTransform;
import javafx.scene.layout.AnchorPane;
import javafx.scene.layout.Pane;
import javafx.scene.layout.VBox;
import javafx.scene.paint.Color;
import javafx.scene.shape.Polygon;
import javafx.scene.shape.Rectangle;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.util.List;

/**
 * The startup scene is the first scene displayed and allowing for:
 * <ul>
 * <li>New game which will replace this scene with {@link NewGameScene}</li>
 * <li>Load game which will replace this scene with {@link LoadGameScene}</li>
 * <li>Edit the game options which will replace this scene with {@link GameOptionsScene}</li>
 * <li>Display the game credits</li>
 * <li>Display the Hanse history</li>
 * </ul>
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Jul 14, 2013
 *
 */
@Prototype
@ClassCategory(EClassCategory.PROTOTYPE_BEAN)
public class StartupScene extends BaseStartupScene {

	@Autowired
	private IPolygonLoader polygonLoader;
	@Autowired
	private Locale locale;

	@Autowired
	private MessageSource messageSource;

    private boolean withPolygonShading = false;
	@Autowired
	private UIFactory uiFactory;
	@Autowired
	private PluginLoader pluginLoader;

	private Pane labelShapes;
	private VBox verticalLayout;
	public StartupScene() {
		super(new Pane());
        if (System.getProperty("javafx.polygon.shading") != null) {
            withPolygonShading = true;
        }

	}
	@Override
	protected String getStyleSheetFilename() {
		return "startscene.css";
	}

	@PostConstruct
	private final void setUpScene() {
		Pane root = createBaseSceneLayout();
        
        
        labelShapes = new Pane();

        final Label newGameLbl = new Label(messageSource.getMessage("ch.sahits.game.openpatrician.display.scene.StartupScene.newGame", new Object[]{}, locale.getCurrentLocal()));
        Polygon untransformed = createPolygonShape("newGame");
        untransformed.hoverProperty().addListener(new LabelHoverChangeListener(newGameLbl));
        untransformed.setOnMouseClicked(event -> {
                    pluginLoader.loadPlugins();
					changeScene(uiFactory.getNewGameScene());
				}
		);
        labelShapes.getChildren().add(untransformed);
        
        Label loadGameLbl = new Label(messageSource.getMessage("ch.sahits.game.openpatrician.display.scene.StartupScene.loadGame", new Object[]{}, locale.getCurrentLocal()));
        untransformed = createPolygonShape("loadGame");
        untransformed.hoverProperty().addListener(new LabelHoverChangeListener(loadGameLbl));
		untransformed.setOnMouseClicked(event -> {
					pluginLoader.loadPlugins();
					changeScene(uiFactory.getLoadGameScene());
				}
		);
		labelShapes.getChildren().add(untransformed);

		Label optionsLbl = new Label(messageSource.getMessage("ch.sahits.game.openpatrician.display.scene.StartupScene.options", new Object[]{}, locale.getCurrentLocal()));
		untransformed = createPolygonShape("options");
		untransformed.hoverProperty().addListener(new LabelHoverChangeListener(optionsLbl));
		labelShapes.getChildren().add(untransformed);
		Label creditsLbl = new Label(messageSource.getMessage("ch.sahits.game.openpatrician.display.scene.StartupScene.credits", new Object[]{}, locale.getCurrentLocal()));
		untransformed = createPolygonShape("credits");
		untransformed.hoverProperty().addListener(new LabelHoverChangeListener(creditsLbl));
		untransformed.setOnMouseClicked(event ->
				changeScene(uiFactory.getCreditsScene())
		);
		labelShapes.getChildren().add(untransformed);
		Label historyLbl = new Label(messageSource.getMessage("ch.sahits.game.openpatrician.display.scene.StartupScene.history", new Object[]{}, locale.getCurrentLocal()));
		untransformed = createPolygonShape("history");
		untransformed.hoverProperty().addListener(new LabelHoverChangeListener(historyLbl));
		untransformed.setOnMouseClicked(event ->
				changeScene(uiFactory.getHanseaticHistoryScene())
		);
		labelShapes.getChildren().add(untransformed);
		PerspectiveTransform perspectiveTransform = createPerspectiveTransform();
		verticalLayout = new VBox(30, newGameLbl, loadGameLbl, optionsLbl, creditsLbl, historyLbl);
		verticalLayout.setPadding(new Insets(10));   // avoid toggling effect when hover
		verticalLayout.setEffect(perspectiveTransform);

		// Add the scroll, vertical layout and label shapes to a Pane that is attached to the bottom right side.
		final Rectangle ph = new Rectangle(root.getWidth(), root.getHeight(), Color.TRANSPARENT);
		ph.widthProperty().bind(root.widthProperty());
		ph.heightProperty().bind(root.heightProperty());
		AnchorPane ap = new AnchorPane();
		ap.getChildren().add(paperScroll);
		AnchorPane.setBottomAnchor(paperScroll, 0.0);
		AnchorPane.setRightAnchor(paperScroll, 0.0);

		ap.getChildren().addAll(ph);

		root.getChildren().addAll(ap, verticalLayout, labelShapes);
		heightChange(0, root.getHeight());
		widthChange(0, root.getWidth());
	}

	/**
	 * Create the polygon using the polygon loader.
	 * @param polygonName name of the polygon
	 * @return Polygon definition
	 */
	private Polygon createPolygonShape(String polygonName) {
		List<Polygons.PolygonsForScene.Polygon> startupPolygons = polygonLoader.getPolygon("/polygonDefinition.xml", "startup");
        
		Polygon untransformed = polygonLoader.toJavaFXPolygon(polygonLoader.find(startupPolygons, polygonName));
        if (withPolygonShading) {
            untransformed.setFill(Color.DARKGREEN);
            untransformed.setOpacity(0.5);
        } else {
            untransformed.setFill(Color.TRANSPARENT);
        }
		return untransformed;
	}

	@Override
	public void heightChange(double oldHeight, double newHeigth) {
		double top = Math.max(newHeigth - untransformedHeight - labelShapes.getLayoutY() - 7, 0);
		labelShapes.translateYProperty().set(top);
		top = Math.max(newHeigth - untransformedHeight - verticalLayout.getLayoutY() + 15, 0);
		verticalLayout.translateYProperty().set(top);
	}
	

	@Override
	public void widthChange(double oldWidth, double newWidth) {
		double left = Math.max(newWidth - untransformedWidth - labelShapes.getLayoutX()-36, 0);
		labelShapes.translateXProperty().set(left);
		left = Math.max(newWidth - untransformedWidth - verticalLayout.getLayoutX() + 10, 0);
		verticalLayout.translateXProperty().set(left);
	}

	
	
}
