package ch.sahits.game.openpatrician.display.dialog.sea;

import ch.sahits.game.graphic.image.impl.SelectiveCachableXMLImageLoader;
import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.clientserverinterface.model.factory.ShipFactory;
import ch.sahits.game.openpatrician.clientserverinterface.service.ConvoyService;
import ch.sahits.game.openpatrician.clientserverinterface.service.OutriggerService;
import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.display.ClientViewState;
import ch.sahits.game.openpatrician.display.dialog.CloseButtonDialog;
import ch.sahits.game.openpatrician.clientserverinterface.model.event.ConvoyDisolveEvent;
import ch.sahits.game.openpatrician.clientserverinterface.model.event.ConvoyEvent;
import ch.sahits.game.openpatrician.event.data.ShipEntersPortEvent;
import ch.sahits.game.openpatrician.javafx.control.DecoratedText;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.openpatrician.javafx.service.DecoratedTextFactory;
import ch.sahits.game.openpatrician.model.DisplayMessage;
import ch.sahits.game.openpatrician.model.EMessageCategory;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.event.TargetedEvent;
import ch.sahits.game.openpatrician.model.people.IShipOwner;
import ch.sahits.game.openpatrician.model.ship.ConvoyList;
import ch.sahits.game.openpatrician.model.ship.IConvoy;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import com.google.common.eventbus.AsyncEventBus;
import javafx.event.EventHandler;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.ColumnConstraints;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.HBox;
import javafx.scene.layout.VBox;
import javafx.scene.text.Text;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.util.HashMap;
import java.util.List;

/**
 * This dialog allows creating, joining and dissolving a convoy.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Jan 28, 2016
 */
@Slf4j
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class ConvoyDialog extends CloseButtonDialog {

    /** Reference to the city view model */
    private final ICityPlayerProxyJFX city;
    @Autowired
    private ClientViewState viewState;
    @Autowired
    private MessageSource messageSource;
    @Autowired
    private Locale locale;
    @Autowired
    private DecoratedTextFactory textFactory;
    @Autowired
    @Qualifier("xmlImageLoader")
    private SelectiveCachableXMLImageLoader imageLoader;
    @Autowired
    private OutriggerService outriggerService;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    private ShipService shipService;
    @Autowired
    private ConvoyList convoyList;
    @Autowired
    private ConvoyService convoyService;

    public ConvoyDialog(ICityPlayerProxyJFX city) {
        this.city = city;
    }

    @PostConstruct
    private void initializeDialog() {

        INavigableVessel vessel = viewState.getCurrentCityProxy().get().getActiveShip();
        if (vessel instanceof IShip) {
            initializeCreateConvoyDialog((IShip)vessel);
        } else {
            initializeDissolveConvoyDialog((IConvoy)vessel);
        }
    }

    private void initializeCreateConvoyDialog(IShip vessel) {
        setTitle(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.sea.ConvoyDialog.createTitle", new Object[]{}, locale.getCurrentLocal()));
        if (shipService.isOrlegCapable(vessel, viewState.getCurrentCityProxy().get().getCity())) { // can create a convoy
            String template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.sea.ConvoyDialog.fullfilledRequirements.part1", new Object[]{vessel.getName()}, locale.getCurrentLocal());
            DecoratedText text = textFactory.createDecoratedText(template, new HashMap<>());
            VBox box = new VBox(text);
            HBox hbox = new HBox(50);
            String s = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.sea.ConvoyDialog.createPrivateConvoy", new Object[]{}, locale.getCurrentLocal());
            OpenPatricianLargeWaxButton action = new OpenPatricianLargeWaxButton(s);
            action.getStyleClass().add("actionButton");
            action.setId("privateConvoy");
            action.setOnAction(createPrivateConvoy(vessel));
            hbox.getChildren().add(action);
            s = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.sea.ConvoyDialog.createPublicConvoy", new Object[]{}, locale.getCurrentLocal());
            action = new OpenPatricianLargeWaxButton(s);
            action.getStyleClass().add("actionButton");
            action.setId("publicConvoy");
            action.setOnAction(createPublicConvoy(vessel));
            hbox.getChildren().add(action);
            box.getChildren().add(hbox);
            List<IConvoy> presentConvoys = convoyList.findConvoy(city.getCity());
            if (presentConvoys.isEmpty()) {
                template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.sea.ConvoyDialog.fullfilledRequirements.part2", new Object[]{vessel.getName()}, locale.getCurrentLocal());
                text = textFactory.createDecoratedText(template, new HashMap<>());
                box.getChildren().add(text);
            } else {
                template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.sea.ConvoyDialog.fullfilledRequirements.part2_2", new Object[]{vessel.getName()}, locale.getCurrentLocal());
                text = textFactory.createDecoratedText(template, new HashMap<>());
                box.getChildren().add(text);
                addConvoyList(box, presentConvoys, vessel);
            }

            box.setLayoutX(50);
            box.setLayoutY(CLOSE_BTN_Y_POS - 500);
            getContent().addAll(box);
        } else {
            String template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.sea.ConvoyDialog.unfullfilledRequirements", new Object[]{vessel.getName()}, locale.getCurrentLocal());
            DecoratedText text = textFactory.createDecoratedText(template, new HashMap<>());
            VBox box = new VBox(text);

            GridPane pane1 = new GridPane();
            pane1.setLayoutX(2*FRAME_BORDER);
            pane1.setLayoutY(200);
            pane1.getColumnConstraints().addAll(new ColumnConstraints(100), new ColumnConstraints(50), new ColumnConstraints(50), new ColumnConstraints(50));
            Image sailorIcon = imageLoader.getImage("icons/64/sailor-icon", 32, 32);
            Text captain = new Text(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.sea.ConvoyDialog.captain", new Object[]{}, locale.getCurrentLocal()));
            captain.getStyleClass().add("dialogText");
            pane1.add(captain, 0, 0);

            int amountSailors = getAmountSailors();

            Text sailors = new Text(String.valueOf(amountSailors));
            sailors.getStyleClass().add("dialogText");
            HBox hbox = new HBox(sailors, new ImageView(sailorIcon));
            pane1.add(hbox, 1, 0);

            int minStrength = outriggerService.getRequiredWeaponStrength(city.getCity());
            template = messageSource.getMessage("ch.sahits.game.openpatrician.model.ship.impl.Ship.weaponStrengthEven", new Object[]{minStrength}, locale.getCurrentLocal());
            text = textFactory.createDecoratedText(template, new HashMap<>());
            pane1.add(text, 2, 0);


            box.getChildren().add(pane1);

            List<IConvoy> presentConvoys = convoyList.findConvoy(city.getCity());
            if (! presentConvoys.isEmpty()) {
                template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.sea.ConvoyDialog.joinConvoy", new Object[]{vessel.getName()}, locale.getCurrentLocal());
                text = textFactory.createDecoratedText(template, new HashMap<>());
                box.getChildren().add(text);
                addConvoyList(box, presentConvoys, vessel);
            }

            box.setLayoutX(50);
            box.setLayoutY(CLOSE_BTN_Y_POS - 500);

            getContent().addAll(box);
        }
    }

    private void addConvoyList(VBox box, List<IConvoy> presentConvoys, IShip ship) {
        VBox innerBox = new VBox();
        int i = 0;
        for (IConvoy convoy : presentConvoys) {
            final String key = "ch.sahits.game.javafx.service.NoticeBoardShipSelectionMenuProvider.ship";
            String loadedText = messageSource.getMessage(key,
                    new Object[]{convoy.getName(), convoy.getCapacity()}, locale.getCurrentLocal());
            DecoratedText dt = textFactory.createDecoratedText(loadedText, new HashMap<>());
            dt.setId("convoy"+i);
            innerBox.getChildren().add(dt);
            dt.setOnMouseReleased(evt -> {
                convoyService.join(convoy, ship);
                executeOnCloseButtonClicked();
            });
            i++;
        }
        innerBox.setLayoutX(50);
        box.getChildren().add(innerBox);
    }

    private EventHandler<MouseEvent> createPublicConvoy(IShip vessel) {
        return event -> {
            try {
                convoyService.create(vessel, true);
                executeOnCloseButtonClicked();
            } catch (RuntimeException e) {
                log.error("Failed to create public convoy", e);
            }
        };
    }

    private EventHandler<MouseEvent> createPrivateConvoy(IShip vessel) {
        return event -> {
            try {
                convoyService.create(vessel, false);
                executeOnCloseButtonClicked();
            } catch (RuntimeException e) {
                log.error("Failed to create private convoy", e);
            }
        };
    }


    private void initializeDissolveConvoyDialog(IConvoy vessel) {
        setTitle(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.sea.ConvoyDialog.dissolveTitle", new Object[]{}, locale.getCurrentLocal()));

        String template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.sea.ConvoyDialog.areYouSure", new Object[]{vessel.getName()}, locale.getCurrentLocal());
        DecoratedText text = textFactory.createDecoratedText(template, new HashMap<>());
        VBox box = new VBox(text);
        final int actionButtonX = (WIDTH - 124) / 2;

        String s = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.sea.ConvoyDialog.dissolve", new Object[]{}, locale.getCurrentLocal());
        final OpenPatricianLargeWaxButton action = new OpenPatricianLargeWaxButton(s);
        action.getStyleClass().add("actionButton");
        action.setId("dissolve");
        ICity city = viewState.getCurrentCityProxy().get().getCity();
        action.setOnAction(dissolve(vessel, city));
        action.setLayoutX(actionButtonX);
        action.setLayoutY(CLOSE_BTN_Y_POS - 24);

        box.setLayoutX(50);
        box.setLayoutY(CLOSE_BTN_Y_POS - 500);
        getContent().addAll(box, action);

    }

    private EventHandler<MouseEvent> dissolve(IConvoy vessel, ICity city) {
        return event -> {
            try {
                List<IShip> ships = convoyService.dissolve(vessel, city);
                ships.parallelStream().forEach(ship -> clientServerEventBus.post(new ShipEntersPortEvent(ship, city)));
                executeOnCloseButtonClicked();
            } catch (RuntimeException e) {
                log.error("Failed to dissolve convoy", e);
            }
        };
    }


    private int getAmountSailors() {
        return 20;
    }
}
