package ch.sahits.game.openpatrician.display.dialog.ship;

import ch.sahits.game.event.ViewChangeEvent;
import ch.sahits.game.graphic.image.IDataImageLoader;
import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.display.ClientViewState;
import ch.sahits.game.openpatrician.display.EViewState;
import ch.sahits.game.openpatrician.display.dialog.CloseButtonDialog;
import ch.sahits.game.openpatrician.display.dialog.service.DialogUtil;
import ch.sahits.game.openpatrician.display.event.data.FocusLocationEvent;
import ch.sahits.game.openpatrician.display.javafx.MainGameView;
import ch.sahits.game.openpatrician.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.openpatrician.event.EViewChangeEvent;
import ch.sahits.game.openpatrician.event.NoticeBoardUpdate;
import ch.sahits.game.openpatrician.event.data.SwitchCity;
import ch.sahits.game.openpatrician.javafx.control.NoticeBoard;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.service.ModelTranslations;
import ch.sahits.game.openpatrician.model.ship.EShipTravelState;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.ui.MapState;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import javafx.geometry.HPos;
import javafx.geometry.VPos;
import javafx.scene.control.Label;
import javafx.scene.control.ScrollPane;
import javafx.scene.control.ScrollPane.ScrollBarPolicy;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.layout.ColumnConstraints;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.RowConstraints;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.net.URL;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;

/**
 * Dialog displaying all ships of the player and if they are selectable allow selection.
 * @author Andi Hotz, (c) Sahits GmbH, 2018
 * Created on Jun 24, 2018
 */
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class ShipListDialog extends CloseButtonDialog {
    /** Reference to the proxy view model */
    private final ICityPlayerProxyJFX proxy;

    @Autowired
    private MessageSource messageSource;
    @Autowired
    private Locale locale;
    @Autowired
    private IDataImageLoader imageLoader;
    @Autowired
    private ModelTranslations translations;
    @Autowired
    private ApplicationContext context;
    @Autowired
    private ClientViewState viewState;
    @Autowired
    private MapState mapState;
    @Autowired
    private ShipService shipService;
    @Autowired
    private DialogUtil dialogUtil;

    private Image anchor;
    private Image sail;
    private Image repair;
    private Image coin0;
    private Image coin1;
    private Image coin2;
    private Image coin3;
    private Image coin4;
    private Image coin5;
    private Image convoy;
    private Image autotrade;

    public ShipListDialog(ICityPlayerProxyJFX proxy) {
        this.proxy = proxy;
        String cssFileName = "ScrollPane.css";
        URL resource = NoticeBoard.class.getResource(cssFileName);
        getStylesheets().add(resource.toExternalForm());
    }

    @PostConstruct
    private void initializeDialog() {
        RowConstraints rowConstraints = new RowConstraints(32);
        anchor = imageLoader.getImage("icons/32/anchor");
        sail = imageLoader.getImage("icons/32/sailing-icon");
        repair = imageLoader.getImage("icons/32/repair-icon");
        coin0 = imageLoader.getImage("icons/32/coins-0-icon");
        coin1 = imageLoader.getImage("icons/32/coins-1-icon");
        coin2 = imageLoader.getImage("icons/32/coins-2-icon");
        coin3 = imageLoader.getImage("icons/32/coins-3-icon");
        coin4 = imageLoader.getImage("icons/32/coins-4-icon");
        coin5 = imageLoader.getImage("icons/32/coins-5-icon");
        convoy = imageLoader.getImage("icons/32/icon_convoy");
        autotrade = imageLoader.getImage("icons/32/icon_autotrade");
        // Scroll pane with a grid pane inside it
        GridPane header = new GridPane();
        header.setId("header");
        header.setLayoutY(100);
        header.setLayoutX(INSET);

        String shipName = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ship.ShipListDialog.shipname",
                new Object[]{}, locale.getCurrentLocal());
        Label shipNameTitle = new Label(shipName);
        shipNameTitle.getStyleClass().addAll("dialogText");
        header.add(shipNameTitle, 0, 0);
        // constraints
        GridPane.setValignment(shipNameTitle, VPos.CENTER);
        header.getRowConstraints().add(rowConstraints);
        int iconWidth = 32;
        int col2Width = 70;
        int col3Width = 100;
        int col1Width = WRAPPING_WIDTH - col2Width - col3Width - iconWidth * 6;
        header.getColumnConstraints().add(new ColumnConstraints(col1Width));

        Label shipTypeTitle = new Label(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ship.ShipListDialog.shiptype",
                new Object[]{}, locale.getCurrentLocal()));
        shipTypeTitle.getStyleClass().addAll("dialogText");
        header.add(shipTypeTitle, 1, 0);
        // constraints
        GridPane.setValignment(shipTypeTitle, VPos.CENTER);
        header.getColumnConstraints().add(new ColumnConstraints(col2Width));


        Label shipStateTitle = new Label(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ship.ShipListDialog.shipstate",
                new Object[]{}, locale.getCurrentLocal()));
        shipStateTitle.getStyleClass().addAll("dialogText");
        header.add(shipStateTitle, 2, 0);
        // constraints
        GridPane.setValignment(shipStateTitle, VPos.CENTER);
        header.getColumnConstraints().add(new ColumnConstraints(col3Width));

        Image img = imageLoader.getImage("icons/32/health-icon");
        ImageView healthIcon = new ImageView(img);
        header.add(healthIcon, 3, 0);
        // constraints
        GridPane.setValignment(healthIcon, VPos.CENTER);
        GridPane.setHalignment(healthIcon, HPos.CENTER);
        header.getColumnConstraints().add(new ColumnConstraints(img.getWidth()));

        img = imageLoader.getImage("icons/32/cargo-capacity-icon");
        ImageView capacityIcon = new ImageView(img);
        header.add(capacityIcon, 4, 0);
        // constraints
        GridPane.setValignment(capacityIcon, VPos.CENTER);
        GridPane.setHalignment(capacityIcon, HPos.CENTER);
        header.getColumnConstraints().add(new ColumnConstraints(img.getWidth()));

        img = imageLoader.getImage("icons/32/trade-icon");
        ImageView valueIcon = new ImageView(img);
        header.add(valueIcon, 5, 0);
        // constraints
        GridPane.setValignment(valueIcon, VPos.CENTER);
        GridPane.setHalignment(valueIcon, HPos.CENTER);
        header.getColumnConstraints().add(new ColumnConstraints(img.getWidth() * 2));

        Label emptyLbl = new Label("");
        header.add(emptyLbl, 6, 0);
        // constraints
        header.getColumnConstraints().add(new ColumnConstraints(img.getWidth()*2));

        GridPane body = new GridPane();
        body.setId("body");
        ScrollPane scrollPane = new ScrollPane();
        scrollPane.setLayoutX(INSET - 1);
        scrollPane.setLayoutY(100 + iconWidth);
        scrollPane.setMaxWidth(WIDTH - INSET);
        scrollPane.setMinWidth(WIDTH - INSET);
        scrollPane.setMaxHeight(HEIGHT - 200);
        scrollPane.setMinHeight(HEIGHT - 200);
        scrollPane.setHbarPolicy(ScrollBarPolicy.NEVER);
        scrollPane.setVbarPolicy(ScrollBarPolicy.AS_NEEDED);
        scrollPane.setContent(body);

        ObservableList<ShipModelForShipList> shipList = createObservableList();

        int row = 0;
        for (ShipModelForShipList shipModelForShipList : shipList) {
            boolean selectable = isSelectableShip(shipModelForShipList);

            Label name = new Label(shipModelForShipList.getShipName());
            name.getStyleClass().add("smallDialogText");
            name.setId("name"+row);
            if (!selectable) {
                name.getStyleClass().add("inactive");
            } else {
                name.setOnMouseReleased(event -> {
                    IShip ship = shipModelForShipList.getShip();
                    proxy.activateShip(ship);
                    switch (shipModelForShipList.travelingState().get()) {
                        case TRAVEL_TO_CITY:
                        case TRAVEL_TO_DESTINATION: {
                            if (viewState.getState() != EViewState.MAP) {
                                // Switch to map view
                                mapState.setShowingLargeSeaMap(true);
                                ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.MAIN_VIEW_SEA_MAP);
                                clientEventBus.post(new ViewChangeEvent(MainGameView.class, proxy));
                            }
                            // Focus on ship
                            clientEventBus.post(new FocusLocationEvent(ship.getLocation()));
                            // Update notice board
                            ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.NOTICE_HIDE);
                            clientEventBus.post(new NoticeBoardUpdate(proxy));
                            executeOnCloseButtonClicked();
                            break;
                        }
                        case ANCHOR:
                        case REPAIR: {
                            Optional<ICity> optCity = shipService.findCity(ship);
                            Preconditions.checkArgument(optCity.isPresent(), "Ship "+ship.getUuid()+" with state "+ship.travelState().get()+" is not in a city.");
                            if (!proxy.getCity().equals(optCity.get()) || viewState.getState() == EViewState.MAP) {
                                clientEventBus.post(new SwitchCity(optCity.get()));
                            }
                            executeOnCloseButtonClicked();
                            break;
                        }
                    }
                });
            }
            body.add(name, 0, row);
            // constraints
            body.getRowConstraints().add(rowConstraints);
            GridPane.setValignment(name, VPos.CENTER);
            body.getColumnConstraints().add(new ColumnConstraints(col1Width));

            Label type = new Label(translations.getLocalDisplayName(shipModelForShipList.getShipType()));
            type.getStyleClass().add("smallDialogText");
            if (!selectable) {
                type.getStyleClass().add("inactive");
            }
            body.add(type, 1, row);
            // constraints
            GridPane.setValignment(type, VPos.CENTER);
            body.getColumnConstraints().add(new ColumnConstraints(col2Width));

            ImageView statusIcon = new ImageView(dialogUtil.getStatusIcon(shipModelForShipList.travelingState().get()));
            shipModelForShipList.travelingState().addListener((state, oldValue, newvalue) -> {
                statusIcon.setImage(dialogUtil.getStatusIcon(shipModelForShipList.travelingState().get()));
            });
            body.add(statusIcon, 2, row);
            // constraints
            GridPane.setValignment(statusIcon, VPos.CENTER);
            body.getColumnConstraints().add(new ColumnConstraints(statusIcon.getImage().getWidth()));

            Label destination = new Label(shipModelForShipList.destination().get());
            shipModelForShipList.destination().addListener((dest, odlValue, newValue) -> destination.setText(newValue));
            destination.getStyleClass().add("smallDialogText");
            if (!selectable) {
                destination.getStyleClass().add("inactive");
            }
            body.add(destination, 3, row);
            // constraints
            GridPane.setValignment(destination, VPos.CENTER);
            body.getColumnConstraints().add(new ColumnConstraints(col3Width - iconWidth));

            Label health = new Label(String.valueOf(shipModelForShipList.getHealth()));
            health.getStyleClass().add("smallDialogText");
            if (!selectable) {
                health.getStyleClass().add("inactive");
            }
            body.add(health, 4, row);
            // constraints
            GridPane.setValignment(destination, VPos.CENTER);
            body.getColumnConstraints().add(new ColumnConstraints(iconWidth));

            Label loadedCapacity = new Label(String.valueOf(shipModelForShipList.loadedWares().get()));

            loadedCapacity.getStyleClass().add("smallDialogText");
            if (!selectable) {
                loadedCapacity.getStyleClass().add("inactive");
            }
            body.add(loadedCapacity, 5, row);
            // constraints
            GridPane.setValignment(loadedCapacity, VPos.CENTER);
            GridPane.setHalignment(loadedCapacity, HPos.CENTER);
            body.getColumnConstraints().add(new ColumnConstraints(iconWidth));

            ImageView value = new ImageView(getCargoValue(shipModelForShipList));

            shipModelForShipList.loadedWares().addListener((loaded, oldValue, newValue) -> {
                    loadedCapacity.setText(String.valueOf(newValue));
                    value.setImage(getCargoValue(shipModelForShipList));
            });
            body.add(value, 6, row);
            // constraints
            GridPane.setValignment(value, VPos.CENTER);
            GridPane.setHalignment(value, HPos.CENTER);
            body.getColumnConstraints().add(new ColumnConstraints(value.getImage().getWidth()));

            ImageView convoyIcon;
            if (shipModelForShipList.getParent().isPresent()) {
                 convoyIcon = new ImageView(convoy);
            } else {
                convoyIcon = new ImageView(sail);
            }
            body.add(convoyIcon, 7, row);
            // constraints
            GridPane.setValignment(convoyIcon, VPos.CENTER);
            GridPane.setHalignment(convoyIcon, HPos.CENTER);
            body.getColumnConstraints().add(new ColumnConstraints(convoyIcon.getImage().getWidth()));

            if (shipModelForShipList.isAutotrading()) {
                ImageView autoTrading = new ImageView(autotrade);
                body.add(autoTrading, 8, row);
                // constraints
                GridPane.setValignment(convoyIcon, VPos.CENTER);
                GridPane.setHalignment(convoyIcon, HPos.CENTER);
                body.getColumnConstraints().add(new ColumnConstraints(autotrade.getWidth()));
            }
            row++;
        }

        getContent().addAll(header, scrollPane);
    }

    private boolean isSelectableShip(ShipModelForShipList shipModelForShipList) {
        return shipModelForShipList.isPlayersShip(proxy.getPlayer())
                && !shipModelForShipList.travelingState().get().equals(EShipTravelState.EXPEDITION)
                && shipModelForShipList.getShip().isAvailable();
    }

    private Image getCargoValue(ShipModelForShipList shipModelForShipList) {
        int value = shipModelForShipList.getValue();
        if (value < 1000) {
            return coin0;
        }
        if (value < 2000) {
            return coin1;
        }
        if (value < 4000) {
            return coin2;
        }
        if (value < 8000) {
            return coin3;
        }
        if (value < 16000) {
            return coin4;
        } else {
            return coin5;
        }
    }

    /**
     * Create a list of all the players ship and include the parent ship and grouping them correctly.
     * @return
     */
    @VisibleForTesting
    ObservableList<ShipModelForShipList> createObservableList() {
        IHumanPlayer player = proxy.getPlayer();
        List<IShip> ships = new ArrayList<>(player.getFleet());
        List<ShipModelForShipList> sorted = new ArrayList<>();
        while (!ships.isEmpty()) {
            IShip ship = ships.get(0);
            if (ship.parentShipProperty().get() == null) {
                sorted.add(context.getBean(ShipModelForShipList.class, ship));
                ships.remove(ship);
            } else {
                IShip parent = ship.parentShipProperty().get();
                sorted.add(context.getBean(ShipModelForShipList.class, parent));
                ships.remove(parent);
                for (Iterator<IShip> iterator = ships.iterator(); iterator.hasNext(); ) {
                    IShip s = iterator.next();
                    if (parent.equals(s.parentShipProperty().get())) {
                        sorted.add(context.getBean(ShipModelForShipList.class, s));
                        iterator.remove();
                    }
                }
            }
        }
        return FXCollections.observableArrayList(sorted);
    }
}
