package ch.sahits.game.openpatrician.display.event.task;

import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.clientserverinterface.event.BuyWares;
import ch.sahits.game.openpatrician.clientserverinterface.event.PostponedDisplayDialogMessage;
import ch.sahits.game.openpatrician.clientserverinterface.event.PostponedDisplayMessage;
import ch.sahits.game.openpatrician.clientserverinterface.model.event.MarriageBrokerAnnouncementState;
import ch.sahits.game.openpatrician.clientserverinterface.service.ClientServerTaskFactory;
import ch.sahits.game.openpatrician.model.EMessageCategory;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.event.TimedTask;
import ch.sahits.game.openpatrician.model.initialisation.StartNewGameBean;
import ch.sahits.game.openpatrician.model.people.IBuyer;
import ch.sahits.game.openpatrician.model.people.INonFreeSeaPirate;
import ch.sahits.game.openpatrician.model.people.IWarehouseTenant;
import ch.sahits.game.openpatrician.model.people.impl.BuyerState;
import ch.sahits.game.openpatrician.model.people.impl.WarehouseTenant;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ui.DialogTemplate;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.DependentInitialisation;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import javafx.geometry.Point2D;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.Optional;

/**
 * Factory for the tasks in this player.
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Feb 23, 2014
 */
@Service
@Lazy
@DependentInitialisation(StartNewGameBean.class)
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class ClientTaskFactory extends ClientServerTaskFactory {
    @Autowired
    private ApplicationContext context;

    /**
     * Retrieve the oponents order.
     * @param opponent player
     * @param client who gives the order
     * @param player the order executor
     * @param premium on successful execution
     * @return Order of the opponent
     */
    public OpponentsOrder getOpponentsOrder(IPlayer opponent,
                                            IPlayer client, IPlayer player, int premium) {
        Object[] args = new Object[]{opponent, client, player, premium};
        return (OpponentsOrder)  context.getBean("opponentsOrder", args);
    }
    /**
     * Retrieve the concurrent reconsiliation timer task.
     * @return reconciliation task
     */
    public TimedTask getConcurrentReconciliation() {
        return  context.getBean(ConcurrentReconciliation.class);
    }



    /**
     * Retrieve the timed task for buying wares.
     * @param cityPlayerProxy proxy instance
     * @param buyer client
     * @return task to buy wares
     */
    public TimedTask getBuyWares(ICityPlayerProxyJFX cityPlayerProxy, IBuyer buyer) {
        Object[] args = new Object[]{cityPlayerProxy.getPlayer(), cityPlayerProxy.getCity(), new BuyerState(buyer)};
        return (BuyWares)  context.getBean("buyWares", args);
    }

    /**
     * Retrieve the delayed caught timed task.
     * @param player that is caught
     * @param where city where the player is caught
     * @return delayed action of being caught
     */
    public IDelayedCaughtAction getDelayedCaughtAction(IPlayer player, ICity where) {
        Object[] args = new Object[]{player, where};
        return (DelayedCaughtAction)  context.getBean("delayedCaughtAction", args);
    }

    /**
     * Retrieve the task for breaking in into an opponents office.
     * @param opponent .
     * @return break in action
     */
    public IBreakInAction getBreakInAction(IPlayer opponent) {
        Object[] args = new Object[]{opponent};
        return (BreakInAction)  context.getBean("breakInAction", args);
    }

    /**
     * Retrieve the postponed display message task. Add the task to the task list.
     * @param executionTime when the message should pop up
     * @param category of the message
     * @param messageKey text key of the message
     * @param  messageArgs array of message arguments
     * @return postponed display message
     */
    public PostponedDisplayMessage getPostponedDisplayMessage(LocalDateTime executionTime, EMessageCategory category, String messageKey, Object...messageArgs) {
        return context.getBean(PostponedDisplayMessage.class, executionTime, category, messageKey, messageArgs);
    }
    /**
     * Retrieve the postponed display message task. The message is added to the task list.
     * @param executionTime when the message should pop up
     * @param category of the message
     * @param template dialog template used for the dialog
     * @return postponed display dialog message
     */
    public PostponedDisplayDialogMessage getPostponedDisplayDialogMessage(LocalDateTime executionTime, EMessageCategory category, DialogTemplate template) {
        Object[] args = new Object[]{executionTime, category, template};
        return (PostponedDisplayDialogMessage)  context.getBean("postponedDisplayDialogMessage", args);
    }

    /**
     * Create a timed task for a fine.
     * @param player the fined player
     * @param fine positive fine amount
     * @param messageKey message key of the fine message
     * @param deadline time of the event
     * @return timed event to be supplied to the timed task list.
     */
    public FinedTimedTask getFinedTask(IHumanPlayer player, int fine, String messageKey, LocalDateTime deadline) {
        return (FinedTimedTask) context.getBean("finedTimedTask", player, fine, messageKey, deadline);
    }

    /**
     * Create a timed task for the marriage offer.
     * @param player for whom the off is
     * @param state of the marriage announcemnt
     * @param deadLine of the task execution
     * @return timed task to be suppled to the task list
     */
    public MarriageOfferTimedTask getMarriageOfferTask(IHumanPlayer player, MarriageBrokerAnnouncementState state, LocalDateTime deadLine) {
        return (MarriageOfferTimedTask) context.getBean("marriageOfferTimedTask", player, state, deadLine);
    }

    /**
     * Create a timed task for the delayed traveling event triggered and handled by the map view.
     * @param destination coordinates on the screen
     * @param vessel that travels
     * @param city if the destination is a city
     * @return timed task to be supplied to the timed task list.
     */
    public TravelToTimedTask getTravelToDelayed(Point2D destination, INavigableVessel vessel, Optional<ICity> city) {
        return (TravelToTimedTask) context.getBean("travelToTimedTask", destination, vessel, city);
    }

    /**
     * Create a timed task for the payment of the warehouse tenant.
     * @param player to which the payment is being made.
     * @param tenant who does the payment
     * @param deadLine when the payment happens
     * @return timed task to be supplied to the task list.
     */
    public WarehouseTenantPayTimedTask getWarehouseTenantPayment(IPlayer player, IWarehouseTenant tenant, LocalDateTime deadLine) {
        return (WarehouseTenantPayTimedTask) context.getBean("warehouseTenantPayTimedTask", player, new WarehouseTenant(tenant), deadLine);
    }

    /**
     * Create a timed task for the leaving of the warehouse tenant.
     * @param office which is occupied by the tenant
     * @param tenant who is leaving
     * @param dealine date of the leaving
     * @return timed task to be supplied to the task list.
     */
    public WarehouseTentantLeaveTimedTask getWarehouseTenantLeaving(ITradingOffice office, IWarehouseTenant tenant, LocalDateTime dealine) {
        return (WarehouseTentantLeaveTimedTask) context.getBean("warehouseTentantLeaveTimedTask", office, new WarehouseTenant(tenant), dealine);
    }

    /**
     * Create a timed task to free a non free sea pirate.
     * @param seaPirate to be freed.
     * @param deadline when it will happen
     * @return timed task to be supplied to the task list.
     */
    public FreePirateTimedTask getFreePirateTask(INonFreeSeaPirate seaPirate, LocalDateTime deadline) {
        return (FreePirateTimedTask) context.getBean("freePirateTimedTask", seaPirate, deadline);
    }

    /**
     * Create a timed task for the reputation update upon a church feeding.
     * @param city where the feeding happend
     * @param player who sponsored the feeding
     * @param amountBeggers amount of feed beggars
     * @param deadline when it will happen
     * @return timed task to be supplied to the task list.
     */
    public ChurchFeedingReputationUpdateTimedTask getReputationUpdateTaskChurchFeeding(ICity city, IPlayer player, int amountBeggers, LocalDateTime deadline) {
        return (ChurchFeedingReputationUpdateTimedTask) context.getBean("churchFeedingReputationUpdateTimedTask", city, player, amountBeggers, deadline);
    }

    /**
     * Create a timed task for the marriage feast delivery check.
     * @param deadline when the check should happen.
     * @param player who is responsible for delivering the wares
     * @param city where the wedding is happening.
     * @return timed task to be supplied to the task list.
     */
    public MarriageFeastDeliveryTask getMarriageFeastDelivery(LocalDateTime deadline, IPlayer player, ICity city) {
        return context.getBean(MarriageFeastDeliveryTask.class, deadline, player, city);
    }
}
