package ch.sahits.game.openpatrician.display.javafx;

import ch.sahits.game.graphic.image.IDataImageLoader;
import ch.sahits.game.graphic.image.ImageUtil;
import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.clientserverinterface.model.PathInterpolatorMap;
import ch.sahits.game.openpatrician.clientserverinterface.service.IPathConverter;
import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.display.ClientViewState;
import ch.sahits.game.openpatrician.display.event.data.DelayedTravelToEvent;
import ch.sahits.game.openpatrician.display.event.data.FocusLocationEvent;
import ch.sahits.game.openpatrician.display.event.task.ClientTaskFactory;
import ch.sahits.game.openpatrician.model.DisplayInfoMessage;
import ch.sahits.game.openpatrician.engine.sea.AStarGraphProvider;
import ch.sahits.game.openpatrician.engine.sea.LocationTracker;
import ch.sahits.game.openpatrician.engine.sea.SeafaringService;
import ch.sahits.game.openpatrician.event.EGameStatusChange;
import ch.sahits.game.openpatrician.event.GameStateChange;
import ch.sahits.game.openpatrician.event.data.ShipArrivesAtDestinationEvent;
import ch.sahits.game.openpatrician.event.data.ShipEntersPortEvent;
import ch.sahits.game.openpatrician.event.data.ShipLeavingPort;
import ch.sahits.game.openpatrician.event.data.ShipNearingPortEvent;
import ch.sahits.game.openpatrician.event.data.ShipPositionUpdateEvent;
import ch.sahits.game.openpatrician.event.data.SwitchCity;
import ch.sahits.game.openpatrician.javafx.bindings.HorizontalScrollBinding;
import ch.sahits.game.openpatrician.javafx.control.CityIcons;
import ch.sahits.game.openpatrician.javafx.control.ShipIcon;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.event.TimedTask;
import ch.sahits.game.openpatrician.model.event.TimedUpdatableTaskList;
import ch.sahits.game.openpatrician.model.map.IMap;
import ch.sahits.game.openpatrician.model.people.IShipOwner;
import ch.sahits.game.openpatrician.model.sea.ITravellingVessels;
import ch.sahits.game.openpatrician.model.sea.TravellingVessel;
import ch.sahits.game.openpatrician.model.ship.EShipType;
import ch.sahits.game.openpatrician.model.ship.IConvoy;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.MultimapType;
import ch.sahits.game.openpatrician.utilities.annotation.UniquePrototype;
import ch.sahits.game.openpatrician.utilities.javafx.IJavaFXApplicationThreadExecution;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;
import javafx.animation.Animation;
import javafx.animation.FadeTransition;
import javafx.animation.KeyFrame;
import javafx.animation.Timeline;
import javafx.application.Platform;
import javafx.beans.property.DoubleProperty;
import javafx.beans.property.SimpleDoubleProperty;
import javafx.geometry.Point2D;
import javafx.scene.Node;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.input.MouseButton;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.Pane;
import javafx.scene.paint.Color;
import javafx.scene.shape.Circle;
import javafx.scene.shape.CubicCurveTo;
import javafx.scene.shape.MoveTo;
import javafx.scene.shape.Path;
import javafx.scene.shape.PathElement;
import javafx.scene.shape.Rectangle;
import javafx.scene.shape.Shape;
import javafx.util.Duration;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.TimerTask;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import static java.util.stream.Collectors.toList;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Dec 26, 2015
 */
@Slf4j
@ClassCategory({EClassCategory.JAVAFX, EClassCategory.UNRELEVANT_FOR_DESERIALISATION, EClassCategory.SINGLETON_BEAN})
@UniquePrototype
public class SeamapImageView extends BaseMainGameImageView {
    // TODO: andi 4/28/16 make this a configurable constant 
    private int visibleRange = 40;

    private final static int PATH_FADING_DELAY_MS = 5000;

    private final ImageView imgView;
    private Pane shipCanvas;  // draw the ships here
    private final Rectangle clip;
    private final Rectangle scrollLeft;
    private final Rectangle scrollRight;
    private final Timeline slideLeftAnimation;
    private final Timeline slideRightAnimation;

    private DoubleProperty scale = new SimpleDoubleProperty(1);
    @Autowired
    private IMap map;
    @Autowired
    private SeafaringService seafaringService;
    @Autowired
    private ClientViewState viewState;
    @Autowired
    private AStarGraphProvider aStarGraphService;
    @Autowired
    private IPathConverter pathConverter;
    @Autowired
    private ITravellingVessels vessels;
    @Autowired
    private PathInterpolatorMap interpolators;
    @Autowired
    @Qualifier("xmlImageLoader")
    private IDataImageLoader imageLoader;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    @Qualifier("syncServerClientEventBus")
    private EventBus syncServerClientEventBus;
    @Autowired
    @Qualifier("clientEventBus")
    protected AsyncEventBus clientEventBus;
    @Autowired
    private TimedUpdatableTaskList taskList;
    @Autowired
    private Date date;
    @Autowired
    private ShipService shipService;
    @Autowired
    private LocationTracker locationTracker;
    @Autowired
    private IJavaFXApplicationThreadExecution threadExecution;
    @Autowired
    @Qualifier("uiTimer")
    private ScheduledExecutorService uiTimer;
    @Autowired
    private ClientTaskFactory taskFactory;

    private boolean displayAllShips = false;

    private Image mapImage;
    private final Rectangle slaveClip;
    private Point2D focus;
    private HorizontalScrollBinding scrollBinding;
    /**
     * Mapping the visible vessels of other players for all players ships.
     */
    @MultimapType(key = INavigableVessel.class, value = INavigableVessel.class)
    private Multimap<INavigableVessel, INavigableVessel> visibleShips = Multimaps.synchronizedMultimap(ArrayListMultimap.create());


    public SeamapImageView(Image mapImage, double width, double height, Point2D focus, double scale) {
        super();
        imgView = new ImageView();
        shipCanvas = new Pane();
        shipCanvas.setId("shipCanvas");
        clip = new Rectangle(0, 0, width, height);
        Color slideColor = new Color(1, 1, 1, 0.1);
        slaveClip = new Rectangle(0, 0, width, height);
        scrollLeft = new Rectangle(0, 0, 50, height);
        scrollLeft.setFill(slideColor);
        scrollRight = new Rectangle(width - 50, 0, 50, height);
        scrollRight.setFill(slideColor);
        slideLeftAnimation = new Timeline();
        slideLeftAnimation.setCycleCount(Animation.INDEFINITE);
        KeyFrame kf1 = new KeyFrame(Duration.millis(50),
                event -> resetClipXPosition(clip.getX() - 5, clip.getWidth()));
        slideLeftAnimation.getKeyFrames().add(kf1);
        slideRightAnimation = new Timeline();
        slideRightAnimation.setCycleCount(Animation.INDEFINITE);
        KeyFrame kf2 = new KeyFrame(Duration.millis(50),
                event -> resetClipXPosition(clip.getX() + 5, clip.getWidth()));
        slideRightAnimation.getKeyFrames().add(kf2);

        this.mapImage = mapImage;
        this.focus = focus;
        this.scale.set(scale);
        scrollBinding = new HorizontalScrollBinding(clip, 0);
        log.info("Initialize Sea Map View with focus at {} and scale {}", focus, scale);
    }
    @PostConstruct
    private void init() {

        clip.xProperty().addListener(
                (observable, oldValue, newValue) -> {
                    slaveClip.setX(newValue.doubleValue());
                }
        );
        scrollBinding.addListener((observable, oldValue, newValue) -> {
            switch (newValue) {
                case CENTER:
                    // Ensure left and right scroll are present
                    if (!getChildren().contains(scrollLeft)) {
                        getChildren().add(scrollLeft);
                    }
                    if (!getChildren().contains(scrollRight)) {
                        getChildren().add(scrollRight);
                    }
                    break;
                case NO_SCROLL:
                    // Ensure no scrolls are present
                    getChildren().remove(scrollLeft);
                    getChildren().remove(scrollRight);
                    break;
                case ALL_RIGHT:
                    // Ensure only the left scroll is present
                    getChildren().remove(scrollRight);
                    if (!getChildren().contains(scrollLeft)) {
                        getChildren().add(scrollLeft);
                    }
                    break;
                case ALL_LEFT:
                    // Ensure only the right scroll is present
                    getChildren().remove(scrollLeft);
                    if (!getChildren().contains(scrollRight)) {
                        getChildren().add(scrollRight);
                    }
                    break;
                default:
                    throw new IllegalStateException("Unhandled state: "+newValue);
            }
        });
        clip.yProperty().addListener(
                (observable, oldValue, newValue) -> slaveClip.setY(newValue.doubleValue()));
        clip.widthProperty().addListener((observable, oldValue, newValue) -> {
            slaveClip.setWidth(newValue.doubleValue());
            scrollRight.setX(newValue.doubleValue() - scrollRight.getWidth());
        });
        clip.heightProperty().addListener((observable, oldValue, newValue) -> {
            slaveClip.setHeight(newValue.doubleValue());
            scrollLeft.setHeight(newValue.doubleValue());
            scrollRight.setHeight(newValue.doubleValue());
        });
        resetImage(mapImage, slaveClip.getWidth(), slaveClip.getHeight(), scale.doubleValue());
        imgView.setClip(clip);
        shipCanvas.setClip(slaveClip);
        shipCanvas.setMouseTransparent(true);

        focusOnPoint(focus);

        imgView.addEventHandler(MouseEvent.MOUSE_RELEASED, this::handleMouseClick);


        scrollLeft.setOnMouseEntered(evt -> {
            if (scrolledAllToTheRight(clip.getX())&& !getChildren().contains(scrollRight)) {
                getChildren().add(scrollRight);
            }
            slideLeftAnimation.play();
        });
        scrollLeft.setOnMouseExited(evt -> slideLeftAnimation.stop());
        scrollRight.setOnMouseEntered(evt -> {
            if (scrolledAllToTheLeft(clip.getX()) && !getChildren().contains(scrollLeft)) {
                getChildren().add(scrollLeft);
            }
            slideRightAnimation.play();
        });
        scrollRight.setOnMouseExited(evt -> slideRightAnimation.stop());

        if (!scrolledAllToTheLeft(clip.getX()) && !getChildren().contains(scrollLeft)) {
            getChildren().add(scrollLeft);
        }
        if (!scrolledAllToTheRight(clip.getX()) && !getChildren().contains(scrollRight)) {
            getChildren().add(scrollRight);
        }
        initializeVisibleShips();
        clientServerEventBus.register(this);
        syncServerClientEventBus.register(this);
        clientEventBus.register(this);
    }

    private void initializeVisibleShips() {
        final IHumanPlayer player = viewState.getPlayer();
        for (INavigableVessel ship : player.getSelectableVessels()) {
            addVisibleVessel(ship, player);
        }
    }

    /**
     * Check which vessels are visible and add them.
     * @param ship for which other visible vessels should be checked
     * @param player human player owning the interface.
     */
    @VisibleForTesting
    void addVisibleVessel(INavigableVessel ship, IHumanPlayer player) {
        List<INavigableVessel> visibleVessels = locationTracker.getShipsInSegments(ship.getLocation(), visibleRange);
        for (INavigableVessel visibleVessel : visibleVessels) {
            addSingleVisibleVessel(ship, player, visibleVessel);
        }
    }

    /**
     * Add the ship to the visible set and map if it is not in a city and not owned by the player.
     * @param ship owned by the human player
     * @param player human player owning the interface.
     * @param visibleVessel ship that is visible to <code>ship</code>
     */
    private void addSingleVisibleVessel(INavigableVessel ship, IHumanPlayer player, INavigableVessel visibleVessel) {
        if (!isInCity(visibleVessel.getLocation()) && !isInCity(ship.getLocation()) && !visibleVessel.getOwner().equals(player) && areVisibleToEachOther(visibleVessel, ship)) { // not players ship and not in a city
            synchronized (visibleShips) {
                if (displayAllShips || !visibleShips.containsKey(visibleVessel) || !visibleShips.containsEntry(visibleVessel, ship)) {
                    visibleShips.put(visibleVessel, ship);
                }
            }
            if (findShipIcon(visibleVessel) == null) {
                drawShipOnMap(visibleVessel);
            }
        }
    }

    @PreDestroy
    void unregister() {
        clientServerEventBus.unregister(this);
        syncServerClientEventBus.unregister(this);
    }

    /**
     * Check if the location matches any city.
     * @param location to check
     * @return true if the <code>location</code> is in any city.
     */
    private boolean isInCity(Point2D location) {
        return map.getCityCoordinates().stream().anyMatch(coord -> coord.equals(location));
    }


    private void handleMouseClick(MouseEvent evt) {
        boolean travelTo = evt.getButton().equals(MouseButton.PRIMARY);
        double unscaledX = Math.round(evt.getX()/scale.doubleValue());
        double unscaledY = Math.round(evt.getY()/scale.doubleValue());
        Optional<ICity> city = findCity(unscaledX, unscaledY);
        travelTo = shouldTravelToClickedPosition(travelTo, city);

        ICityPlayerProxyJFX cityProxy = viewState.getCurrentCityProxy().get();
        if (travelTo) {
            INavigableVessel vessel = cityProxy.getActiveShip();
            if (vessel instanceof IConvoy && ((IConvoy)vessel).isPublicConvoy()) {
                // TODO: 26.03.17 The coordinates may be wrong: what happens when the map is scrolled, until the travelTo method is called?
                TimedTask task = taskFactory.getTravelToDelayed(new Point2D(evt.getX(), evt.getY()), vessel, city);
                taskList.add(task);
            } else {
                if (shipService.checkNumberOfSailors(vessel)) {
                    travelToDestination(evt.getX(), evt.getY(), city, vessel);
                } else {
                    DisplayInfoMessage infoMessage = new DisplayInfoMessage("ch.sahits.game.openpatrician.display.model.DisplayInfoMessage.notEnoughSailors", new Object[]{vessel.getName()});
                    clientEventBus.post(infoMessage);
                }
            }
        } else if (city.isPresent()) {
            List<ITradingOffice> office = city.get().findBuilding(ITradingOffice.class, Optional.of(viewState.getPlayer()));
            List<IShip> ships = viewState.getPlayer().getFleet();
            if (!office.isEmpty()) {
                switchToCity(city.get());
            } else {
                for (IShip ship : ships) {
                    if (ship.getLocation().equals(city.get().getCoordinates())) {
                        if (cityProxy.getActiveShip() == null) {
                            cityProxy.arrive(ship);
                        }
                        switchToCity(city.get());
                        break;
                    }
                }
            }
        } else {
            double x = evt.getX();
            double y = evt.getY();
            IHumanPlayer player = viewState.getPlayer();
            List<ShipIcon> shipIcons = shipCanvas.getChildren().stream()
                    .filter(node -> node instanceof ShipIcon)
                    .map(ShipIcon.class::cast)
                    .filter(shipIcon -> shipIcon.getPlayer().equals(player))
                    .collect(toList());

            for (Node node : shipIcons) {
                if (node.getBoundsInParent().contains(x, y)) {
                    String id = node.getId();
                    List<INavigableVessel> fleet = player.getSelectableVessels();
                    Optional<INavigableVessel> vessel = shipService.findShipByUuid(fleet, id);
                    vessel.ifPresent(iNavigableVessel ->
                            cityProxy.activateShip(iNavigableVessel));
                }
            }
        }
    }

    /**
     * Travel to the new location if:
     * <ul>
     *     <li>The mouse button click happened with the primary mouse button</li>
     *     <li>There is a view state with an active vessel</li>
     *     <li>If the destination is a city it must reachable by the vessel (i.e. holks and cogs cannot travel to
     *         river cities</li>
     *     <li>Or if the destination was not a city.</li>
     * </ul>
     * @param isPrimaryButton true if the mouse click was the primary button.
     * @param city if the point clicked was a city, it is stored here otherwise it is empty
     * @return true if the active vessel should travel to a new location.
     */
    private boolean shouldTravelToClickedPosition(boolean isPrimaryButton, Optional<ICity> city) {
        boolean travelTo = isPrimaryButton;
        if (travelTo) {
            travelTo = viewState.getCurrentCityProxy().isPresent();
            if (travelTo) {
                if (city.isPresent() && viewState.getCurrentCityProxy().get().getCity().equals(city.get())) {
                    travelTo = false; // back to the city we are currently in
                } else {
                    INavigableVessel activeShip = viewState.getCurrentCityProxy().get().getActiveShip();
                    if (activeShip != null) {
                        if (city.isPresent() && city.get().isRiverCity()) {
                            EShipType type = shipService.getShipType(activeShip);
                            travelTo = type != EShipType.COG && type != EShipType.HOLK;
                        } else {
                            travelTo = true;
                        }
                    } else {
                        travelTo = false; // no ship active
                    }
                }
            }
        }
        return travelTo;
    }

    private void travelToDestination(double x, double y, Optional<ICity> destinationCity, INavigableVessel vessel) {
        double unscaledX = Math.round(x/scale.doubleValue());
        double unscaledY = Math.round(y/scale.doubleValue());
        Optional<ICity> sourceCity = findCity(vessel.getLocation().getX(), vessel.getLocation().getY());
        List<Point2D> path = new ArrayList<>();
        if (!shipService.canReachDestination(vessel, new Point2D(x, y))) {
            DisplayInfoMessage infoMessage = new DisplayInfoMessage("ch.sahits.game.openpatrician.display.model.DisplayInfoMessage.destinationNotReachable", new Object[]{vessel.getName(), destinationCity.get().getName()});
            clientEventBus.post(infoMessage);
            return;
        }
        if (shipService.isShipTooDamagedToSail(vessel)) {
            DisplayInfoMessage infoMessage = new DisplayInfoMessage("ch.sahits.game.openpatrician.display.model.DisplayInfoMessage.tooDamaged", new Object[]{vessel.getName()});
            clientEventBus.post(infoMessage);
            return;
        }
        if (destinationCity.isPresent()) {
            path = seafaringService.travelTo(vessel, destinationCity.get().getCoordinates());
        } else {
            final Point2D destination = new Point2D(unscaledX, unscaledY);
            if (aStarGraphService.isOnSea(destination)) {
                path = seafaringService.travelTo(vessel, destination);
            }
        }
        drawPath(vessel, path);
        synchronized (interpolators) {
            if (destinationCity.isPresent()) {
                interpolators.get(vessel).setDestinationCity(true);
            } else {
                interpolators.get(vessel).setDestinationCity(false);
            }
            if (sourceCity.isPresent()) {
                interpolators.get(vessel).setSourceCity(true);
            } else {
                interpolators.get(vessel).setSourceCity(false);
            }
        }

        // Traveled from anywhere
        if (!sourceCity.isPresent() && viewState.getCurrentCityProxy().isPresent()) {
            viewState.getCurrentCityProxy().get().leave(vessel);
        }
    }

    /**
     * Calculate a new path for the <code>vessel</code> and draw it on the map.
     * @param vessel that travels along the path
     * @param path series of points defining the path.
     */
    private void drawPath(INavigableVessel vessel, List<Point2D> path) {
        if (path != null) {
            clearLines();
            Optional<Path> p = pathConverter.createPath(vessel, path, scale.doubleValue());
            if (p.isPresent()) {
                p.get().setId(vessel.getUuid());
                drawPathOnMap(vessel, path, p.get());
            } else {
                log.warn("Path not available for {} of {} {}", vessel.getName(), vessel.getOwner().getName(), vessel.getOwner().getLastName());
            }
        }
    }

    /**
     * Draw the path onto the map considering the proper scaling.
     * @param vessel for which the path is drawn
     * @param path series of points that define the corners of the path.
     * @param p path that is drawn.
     */
    private void drawPathOnMap(INavigableVessel vessel, List<Point2D> path, Path p) {
        shipCanvas.getChildren().add(p);
        TimerTask task = new TimerTask() {  // TODO: andi 7/18/16 This duplicates the timer for fading in DisplayOverlayMessage: move to a factory.
            @Override
            public void run() {
                Platform.runLater(() -> {
                    FadeTransition ft = new FadeTransition(Duration.millis(2000), p);
                    ft.setFromValue(1.0);
                    ft.setToValue(0);

                    ft.setAutoReverse(false);
                    ft.setOnFinished(event -> shipCanvas.getChildren().remove(p));

                    ft.play();

                });
            }
        };

        uiTimer.schedule(task, PATH_FADING_DELAY_MS, TimeUnit.MILLISECONDS);
        addNewShipIconToCanvas(vessel, path);

    }

    /**
     * Draw a vessel on the map with default orientation (faceing to the west)
     * @param vessel to draw on the map
     */
    private void drawShipOnMap(INavigableVessel vessel) {
        List<Point2D> path = vessels.getTravellingVessel(vessel).getCalculatablePath();
        addNewShipIconToCanvas(vessel, path);
    }

    private void addNewShipIconToCanvas(INavigableVessel vessel, List<Point2D> path) {
        ShipIcon shipIcon = new ShipIcon(vessel, viewState.getPlayer(), imageLoader);
        shipIcon.scaleProperty().bind(scale);
        shipIcon.updatePosition();
        shipIcon.setTravelingEast(path.get(0).getX() < path.get(path.size() - 1).getX());
        removeVesselFromView(vessel);
        shipCanvas.getChildren().add(shipIcon);
    }

    private ShipIcon findShipIcon(INavigableVessel vessel) {
        String id = vessel.getUuid();
        for (Node node : shipCanvas.getChildrenUnmodifiable()) {
            if (node instanceof ShipIcon && id.equals(node.getId())) {
                return (ShipIcon) node;
            }
        }
        log.trace("The vessel "+vessel.getUuid()+": "+vessel.getName()+" of "+vessel.getOwner().getClass().getSimpleName()+" "+vessel.getOwner().getName()+" "+vessel.getOwner().getLastName()+" is not visible on the map");
        return null;
    }

    private void clearLines() {
        shipCanvas.getChildren().removeIf(node -> node instanceof Shape);
    }

    private void switchToCity(ICity city) {
        clientEventBus.post(new SwitchCity(city));
    }

    private Optional<ICity> findCity(double unscaledX, double unscaledY) {
        Point2D p = new Point2D(unscaledX, unscaledY);
        for (ICity city : map.getCities()) {
            double distance = p.distance(city.getCoordinates());
            if (distance <= ImageUtil.CITY_RADIUS) {
                return Optional.of(city);
            }
        }
        return Optional.empty();
    }

    private boolean scrolledAllToTheLeft(double x) {
        return x <= 0.0;
    }
    private boolean scrolledAllToTheRight(double x) {
        return x >= imgView.getImage().getWidth() - clip.getWidth();
    }


    private void focusOnPoint(Point2D focus) {
        double totalwidth = imgView.getImage().getWidth();
        double focusX = focus.getX();
        double clipWidth = clip.getWidth();
        double x = Math.min(focusX - clipWidth/2, totalwidth - clipWidth);
        resetClipXPosition(x, clipWidth);
        this.focus = focus;
    }

    private void resetClipXPosition(double x, double clipWidth) {
        if (x < 0) {
            x = 0;
        }
        if (x > imgView.getImage().getWidth() - clipWidth) {
            x = imgView.getImage().getWidth() - clipWidth;
        }
        imgView.setLayoutX(-x);
        shipCanvas.setLayoutX(-x);
        clip.setX(x);
        log.trace("Set clip position={}, view and ship canvas layout: {}", x, -x);
    }

    /**
     * Reset the image to accomodate the dimensions.
     * @param mapImage image of the map
     * @param width target width of the map
     * @param height target height of the map
     * @param scale new scale of the map
     */
    public void resetImage(Image mapImage, double width, double height, double scale) {
        double oldScale = this.scale.doubleValue();
        this.scale.set(scale);
        imgView.setImage(mapImage);
        scrollBinding.setTotalWidth(mapImage.getWidth());
        shipCanvas.setMaxWidth(mapImage.getWidth());
        shipCanvas.setMinWidth(mapImage.getWidth());
        shipCanvas.setMaxHeight(mapImage.getHeight());
        shipCanvas.setMinHeight(mapImage.getHeight());

        getChildren().removeAll(imgView, shipCanvas);
        double oldWidth = clip.getWidth();
        boolean heightChange = clip.getHeight() != height;

        double x = Math.max((width - mapImage.getWidth())/2,0);
        double y = Math.max((height - mapImage.getHeight())/2,0);
        if (!heightChange) {
            imgView.setLayoutX(x);
            shipCanvas.setLayoutX(x);
        }
        imgView.setLayoutY(y);
        shipCanvas.setLayoutY(y);

        // Ensure the proper portion of the map is shown
        double focuspoint = clip.getX() + oldWidth/2;
        double xx = Math.max(0, focuspoint - width/2);
        resetClipXPosition(xx, width);

        for (Iterator<Node> iterator = shipCanvas.getChildren().iterator(); iterator.hasNext(); ) {
            Node node = iterator.next();
            if (node instanceof Circle) {
              iterator.remove();
            }
            if (node instanceof CityIcons) {
                iterator.remove();
            }
        }
        drawCityInfo(mapImage, scale, xx);
        drawShipsInCities(mapImage, scale, xx);

        for (INavigableVessel vessel : vessels) {
            TravellingVessel traveling = vessels.getTravellingVessel(vessel);
            if (isVesselVisible(vessel)) {
                removeVesselFromView(vessel);
                drawShipOnMap(traveling.getVessel());
            }
        }

        redrawPath(oldScale);
        clip.setWidth(width);
        clip.setHeight(height);
        getChildren().add(0, shipCanvas);
        getChildren().add(0, imgView);
        scrollRight.setX(width - scrollRight.getWidth());
    }

    private void drawCityInfo(Image mapImage, double scale, double x) {
        for (ICity city : map.getCities()) {
            if (city.getCoordinates().getX() >= x && city.getCoordinates().getX() <= x + mapImage.getWidth()) {
                CityIcons cityIcons = new CityIcons(city.getCityState(), imageLoader);
                // draw below city
                double xPos = (city.getCoordinates().getX() - ImageUtil.CITY_RADIUS) * scale;
                double yPos = (city.getCoordinates().getY() + ImageUtil.CITY_RADIUS + 5) * scale;
                cityIcons.setScaleX(scale);
                cityIcons.setScaleY(scale);
                cityIcons.setLayoutX(xPos);
                cityIcons.setLayoutY(yPos);
                shipCanvas.getChildren().add(0, cityIcons);
            }
        }
    }


    private void drawShipsInCities(Image mapImage, double scale, double x) {
        for (ICity city : map.getCities()) {
            if (city.getCoordinates().getX() >= x && city.getCoordinates().getX() <= x + mapImage.getWidth()) {
                List<INavigableVessel> ships = viewState.getPlayer().findShips(city);
                if (!ships.isEmpty()) {
                    drawShipPresenceInCity(city);
                }
            }
        }
    }

    private void drawShipPresenceInCity(ICity city) {
        int radius = 8;
        int cityX = (int) Math.rint(city.getCoordinates().getX());
        int cityY = (int) Math.rint(city.getCoordinates().getY());
        double lScale = scale.doubleValue();
        Circle c = new Circle(cityX * lScale, cityY * lScale, radius * lScale, Color.WHITE);
        shipCanvas.getChildren().add(c);
    }

    @Subscribe
    public void handleShipLeavesCity(ShipLeavingPort event) {
        ICity city = event.getCity();
        final IShipOwner owner = event.getShip().getOwner();
        if (owner instanceof IHumanPlayer && owner.equals(viewState.getPlayer())) {
            List<INavigableVessel> ships = ((IHumanPlayer)owner).findShips(city);
            if (ships.isEmpty()) {
                double cityX = (int) Math.rint(city.getCoordinates().getX()) * scale.doubleValue();
                double cityY = (int) Math.rint(city.getCoordinates().getY()) * scale.doubleValue();
                for (Iterator<Node> iterator = shipCanvas.getChildren().iterator(); iterator.hasNext(); ) {
                    Node node = iterator.next();
                    if (node instanceof Circle) {
                        Circle c = (Circle) node;
                        if (c.getCenterX() == cityX && c.getCenterY() == cityY) {
                            Platform.runLater(iterator::remove);
                            break;
                        }
                    }
                }
            }
        }
    }

    /**
     * Set the focus to the new location.
     * @param event for the focus point
     */
    @Subscribe
    public void handleFocusEvent(FocusLocationEvent event) {
        threadExecution.execute(() -> focusOnPoint(event.getFocus()));
    }

    /**
     * Initialize the visible ships.
     * @param event game state change event
     */
    @Subscribe
    public void handelLoadedNewGame(GameStateChange event) {
        if (event.getStatusChange() == EGameStatusChange.GAME_LOADED) {
            synchronized (visibleShips) {
                visibleShips.clear();
                initializeVisibleShips();
            }
        }
    }

    private void redrawPath(double oldScale) {
        double scaleCorrection = 1/oldScale;
        double scale = this.scale.doubleValue() * scaleCorrection;
        log.debug("Rescale path from {} to {}, which is a correction of {}", oldScale, this.scale, scale);
        for (Node node : shipCanvas.getChildren()) {
            if (node instanceof Path) {
                for (PathElement pathElement : ((Path) node).getElements()) {
                    rescale(pathElement, scale);
                }
            }
        }
    }

    private void rescale(MoveTo pathElement, double scale) {
         double value = pathElement.getX() * scale;
        pathElement.setX(value);
        value = pathElement.getY() * scale;
        pathElement.setY(value);
    }

    private void rescale(CubicCurveTo pathElement, double scale) {
        double value = pathElement.getX() * scale;
        pathElement.setX(value);
        value = pathElement.getY() * scale;
        pathElement.setY(value);
        value = pathElement.getControlX1() * scale;
        pathElement.setControlX1(value);
        value = pathElement.getControlY1() * scale;
        pathElement.setControlY1(value);
        value = pathElement.getControlX2() * scale;
        pathElement.setControlX2(value);
        value = pathElement.getControlY2() * scale;
        pathElement.setControlY2(value);
    }
    private void rescale(PathElement pathElement, double scale) {
        if (pathElement instanceof MoveTo) {
            rescale((MoveTo)pathElement, scale);
        } else if (pathElement instanceof CubicCurveTo) {
            rescale((CubicCurveTo) pathElement, scale);
        } else {
            throw new IllegalStateException("Rescaling for " + pathElement.getClass().getName() + " is not implemented");
        }
    }
    @Subscribe
    public void handleShipPositionUpdate(ShipPositionUpdateEvent event) {
        INavigableVessel vessel = event.getShip();
        TravellingVessel ship = vessels.getTravellingVessel(vessel);
        if (vessel.getOwner().equals(viewState.getPlayer())) {
            Preconditions.checkNotNull(ship, "The travelling vessel instance for " + vessel.getName() + " ("+vessel.getUuid()+") of " + vessel.getOwner().getName() + " " + vessel.getOwner().getLastName() + " could not be found");
        }
        if (ship != null) {
            threadExecution.execute(() -> {
                if (ship != null) { // The ship is traveling
                    ShipIcon view = findShipIcon(vessel);
                    if (view != null) { // The ship is visible
                        view.updatePosition();
                    } else {
                        log.trace("Failed to find vessel with uuid {}", vessel.getUuid());
                    }
                }
                handleVisibilityShips(vessel);
            });
        } else {
            log.debug("Ship ship update for {} ({}) as ship is not a traveling vessel", vessel.getName(), vessel.getUuid());
        }
    }

    private void handleVisibilityShips(INavigableVessel vessel) {
        IHumanPlayer player = viewState.getPlayer();
        // Track ships no longer visible
        Set<INavigableVessel> noLongerVisibleVessels = new HashSet<>();
        if (vessel.getOwner().equals(player)) {
            // Check that all ships that were visible before the update are still visible
            // if a ship is no longer visible, remove the vessel from the map (value).
            synchronized (visibleShips) {
                Set<INavigableVessel> vessels = new HashSet<>(visibleShips.keySet());
                for (INavigableVessel visibleVessel : vessels) {  // iterate over a view
                    Collection<INavigableVessel> visibleVessels = visibleShips.get(visibleVessel);
                    if (visibleVessels != null && visibleVessels.contains(vessel)) {
                        // visibleVessel was visible to vessel before the update
                        checkVisiblityAndUpdate(vessel, noLongerVisibleVessels, visibleVessel);
                    }
                }
            }
            // Check if there are new ships visible
            addVisibleVessel(vessel, player);
        } else { // The ship does not belong to the player
            // Check that the ship is still visible to one vessel
            // if a ship is no longer visible, remove the vessel from the map (key).
            synchronized (visibleShips) {
                if (visibleShips.containsKey(vessel)) {
                    // crate a copy to avoid concurrent modification issues, as it does not matter if we miss a visiblility
                    // change in this step, we will pick it up later.
                    Collection<INavigableVessel> seenByVessels = new ArrayList<>(visibleShips.get(vessel));
                    for (INavigableVessel visiblePlayerVessel : seenByVessels) {
                        checkVisiblityAndUpdate(visiblePlayerVessel, noLongerVisibleVessels, vessel);
                    }
                }
            }
            List<INavigableVessel> playersVessels = player.getSelectableVessels();
            for (INavigableVessel playersVessel : playersVessels) {
                if (areVisibleToEachOther(playersVessel, vessel)) {
                    if (visibleShips.containsKey(vessel)) {
                        if (!visibleShips.get(vessel).contains(playersVessel)) {
                            addSingleVisibleVessel(playersVessel, player, vessel);
                        }
                    } else {
                        addSingleVisibleVessel(playersVessel, player, vessel);
                    }
                }
            }
            // Check if there are any of the players ship visible
        }
        // remove no longer visible ships from the view
        removeNoLongerVisibleVessels(noLongerVisibleVessels);
    }

    private void removeNoLongerVisibleVessels(Set<INavigableVessel> noLongerVisibleVessels) {
        for (INavigableVessel v : noLongerVisibleVessels) {
            removeVesselFromView(v);
        }
    }

    private void removeVesselFromView(INavigableVessel v) {
        ShipIcon oldView = findShipIcon(v);
        if (oldView != null) {
            shipCanvas.getChildren().remove(oldView);
        }
    }

    /**
     * Check if the two vessels are visible. If they are not add the <code>visibleVessel</code> to the
     * Set <code>noLongerVisibleVessels</code> and remove the combination from the visible ships multimap.
     * @param vessel of the human player owning this view
     * @param noLongerVisibleVessels set of vessels that are no longer visible
     * @param visibleVessel of another player.
     */
    private void checkVisiblityAndUpdate(INavigableVessel vessel, Set<INavigableVessel> noLongerVisibleVessels, INavigableVessel visibleVessel) {
        if (!areVisibleToEachOther(visibleVessel, vessel)) {
            // They can no longer see each other
            synchronized (visibleShips) {
                visibleShips.remove(visibleVessel, vessel);
            }
            // this removes the last value the key is removed as well
            noLongerVisibleVessels.add(visibleVessel);
        }
    }

    /**
     * Check if the two vessels are visible to each other.
     * @param vessel1 first vessel
     * @param vessel2 second vessel
     * @return true if both vessels are within the visible range.
     */
    private boolean areVisibleToEachOther(INavigableVessel vessel1, INavigableVessel vessel2) {
        if (displayAllShips) {
            return true;
        }
        double distance = vessel1.getLocation().distance(vessel2.getLocation());
        return distance <= visibleRange;
    }
    @Subscribe
    public void handleShipReachesDestination(ShipArrivesAtDestinationEvent event) {
        INavigableVessel vessel = event.getShip();
        TravellingVessel ship = vessels.getTravellingVessel(vessel);
        Path drawablePath = null;
        if (ship != null) {
            drawablePath = ship.getDrwawablePath();
        } else {
           Optional<Node> optPath = shipCanvas.getChildren().stream().filter(node -> node instanceof Path && node.getId().equals(vessel.getUuid())).findFirst();
           if (optPath.isPresent()) {
               drawablePath = (Path)optPath.get();
           }
        }
        if (drawablePath != null) {
            final Path path = drawablePath;
            Platform.runLater(() -> shipCanvas.getChildren().remove(path));
        }
    }

    @Subscribe
    public void handleShipNearsPort(ShipNearingPortEvent event) {
        INavigableVessel vessel = event.getShip();
        TravellingVessel ship = vessels.getTravellingVessel(vessel);
        if (ship != null) {
            if (vessels.getTravellingVessel(vessel) != null) {
                threadExecution.execute(() -> removeVesselFromView(vessel));
                if (isPlayersVessel(vessel)) {
                    Set<INavigableVessel> noLongerVisible = removeVisibleVessel(vessel);
                    for (INavigableVessel v : noLongerVisible) {
                        log.debug("The ship {} ({}) is no longer visible", v.getName(), v.getUuid());
                    }
                    Platform.runLater(() -> removeNoLongerVisibleVessels(noLongerVisible));
                }
            }
        } else {
            log.debug("Ship ship nearing port for {} ({}) as ship is not a traveling vessel", vessel.getName(), vessel.getUuid());
        }
    }

    private boolean isPlayersVessel(INavigableVessel vessel) {
        return viewState.getPlayer().equals(vessel.getOwner());
    }

    @Subscribe
    public void handeShipReachesPort(ShipEntersPortEvent event) {
        ICity city = event.getCity();
        final IShipOwner owner = event.getShip().getOwner();
        Platform.runLater(() -> {
            if (owner instanceof IHumanPlayer && owner.equals(viewState.getPlayer())) {
                List<INavigableVessel> ships = ((IHumanPlayer)owner).findShips(city);
                if (ships.size() == 1) {
                    drawShipPresenceInCity(city);
                }
            }
        });
    }

    /**
     * Remove all ship icons from the sea map.
     */
    public void removeShipIcons() {
        shipCanvas.getChildren().removeIf(node -> node instanceof ShipIcon);
    }

    private Set<INavigableVessel> removeVisibleVessel(INavigableVessel vessel) {
        Set<INavigableVessel> noLongerVisibile = new HashSet<>();
        if (displayAllShips) {
            return noLongerVisibile;
        }
        Set<INavigableVessel> keys = new HashSet<>(visibleShips.keySet());
        for (INavigableVessel key : keys) {
            if (key.equals(vessel)) {
                Collection<INavigableVessel> vessels = visibleShips.removeAll(key);
                noLongerVisibile.addAll(vessels);
            } else {
                visibleShips.remove(key, vessel);
                if (visibleShips.get(key).isEmpty()) {
                    noLongerVisibile.addAll(visibleShips.removeAll(key));
                    noLongerVisibile.add(key);
                }
            }
        }
        return noLongerVisibile;
    }

    /**
     * Determin if the vessel is visible on the map by it's own.
     * @param vessel to check visiblity
     * @return true if it is a player vessel nearing a city.
     */
    private boolean isVesselVisible(INavigableVessel vessel) {
        if (displayAllShips) {
            return true;
        }
        if (isPlayersVessel(vessel)) {
           // only true if the ship is not in the city
            return !shipService.isNearingCity(vessel);
        }
        return false;
    }

    /**
     * Delegate calls for handling delayed to travel to actions.
     * @param event for delayed traveling
     */
    @Subscribe
    public void handleDelayedTravelAction(DelayedTravelToEvent event) {
        removeVesselFromView(event.getVessel());
        travelToDestination(event.getDestination().getX(), event.getDestination().getY(), event.getCity(), event.getVessel());
    }

    /**
     * Activate the cheat to display all ships on the map.
     */
    public void toggleVisibilityAllShips() {
        displayAllShips = !displayAllShips;
    }
}
