package ch.sahits.game.openpatrician.display.model;

import ch.sahits.game.openpatrician.model.product.ComputablePriceV2;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.SimpleIntegerProperty;

/**
 * Class to store the required amount in to facilitate the binding.
 * Based on the amount in storage and in the city together with the required amount bindings
 * for the buy amount and buy price is calculated, along with the possibility that not enough wares can be purchased.
 * @author Andi Hotz, (c) Sahits GmbH, 2017
 * Created on Sep 16, 2017
 */
@ClassCategory(EClassCategory.MODEL)
public class RequiredWareCityStorage {
    private IntegerProperty required;
    private final IntegerProperty inStorage;
    private final IntegerProperty inCity;
    private IntegerBinding buyAmount;
    private IntegerBinding buyPrice;
    private BooleanBinding canBuy;

    public RequiredWareCityStorage(int needed,
                                   IntegerProperty inStorageAmount,
                                   IntegerProperty inCityAmount,
                                   IWare ware,
                                   ComputablePriceV2 computablePrice) {
        this.required = new SimpleIntegerProperty(needed);
        this.inStorage = inStorageAmount;
        this.inCity = inCityAmount;
        buyAmount = new IntegerBinding() {
            {
                super.bind(required, inStorage);
            }
            @Override
            protected int computeValue() {
                return Math.max(0, required.get() - inStorage.get());
            }
        };
        canBuy = new BooleanBinding() {
            {
                super.bind(required, inCity, inStorage);
            }
            @Override
            protected boolean computeValue() {
                return required.get() < inStorage.get() + inCity.get();
            }
        };
        buyPrice = new IntegerBinding() {
            {
                super.bind(buyAmount, inCity);
            }
            @Override
            protected int computeValue() {
                if (buyAmount.get() > 0) {
                    return computablePrice.buyPrice(ware, inCity, buyAmount);
                } else {
                    return 0;
                }
            }
        };
    }

    public IntegerProperty requiredProperty() {
        return required;
    }

    public void setRequired(int required) {
        this.required.setValue(required);
    }

    public IntegerProperty inStorageProperty() {
        return inStorage;
    }

    public IntegerProperty inCityProperty() {
        return inCity;
    }

    public Number getBuyAmount() {
        return buyAmount.get();
    }

    public IntegerBinding buyAmountProperty() {
        return buyAmount;
    }

    public Number getBuyPrice() {
        return buyPrice.get();
    }

    public IntegerBinding buyPriceProperty() {
        return buyPrice;
    }

    public Boolean getCanBuy() {
        return canBuy.get();
    }

    public BooleanBinding canBuyProperty() {
        return canBuy;
    }
}
