package ch.sahits.game.openpatrician.display.dialog;

import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.display.dialog.service.ITransferableJFX;
import ch.sahits.game.openpatrician.display.dialog.service.impl.TransferableState;
import ch.sahits.game.openpatrician.display.javafx.action.BuyJFXProxyAction;
import ch.sahits.game.openpatrician.display.javafx.action.SellJFXProxyAction;
import ch.sahits.game.openpatrician.display.javafx.action.UserActions;
import ch.sahits.game.openpatrician.javafx.control.BaleAmount;
import ch.sahits.game.openpatrician.javafx.control.BarrelAmount;
import ch.sahits.game.openpatrician.javafx.control.ChangeAmountButton;
import ch.sahits.game.openpatrician.javafx.control.CoinPrice;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianSmallWaxButton;
import ch.sahits.game.openpatrician.javafx.model.ControlTableCell;
import ch.sahits.game.openpatrician.javafx.model.EDialogType;
import ch.sahits.game.openpatrician.javafx.model.StaticTextTableCell;
import ch.sahits.game.openpatrician.javafx.model.Table;
import ch.sahits.game.openpatrician.javafx.model.TableHeader;
import ch.sahits.game.openpatrician.javafx.model.TableRow;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.product.AmountablePrice;
import ch.sahits.game.openpatrician.model.product.ComputablePriceV2;
import ch.sahits.game.openpatrician.model.product.ETransferAmount;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.service.ModelTranslations;
import ch.sahits.game.openpatrician.model.service.TransferUtil;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.binding.NumberBinding;
import javafx.beans.binding.StringBinding;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.ReadOnlyIntegerProperty;
import javafx.beans.property.SimpleBooleanProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.beans.value.ObservableValue;
import javafx.geometry.HPos;
import javafx.scene.Group;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.util.Optional;

@Slf4j
public abstract class BaseTradeDialog extends TabelViewDialog {
    private ObjectProperty<ETransferAmount> movableAmount = new SimpleObjectProperty<>(ETransferAmount.ONE);

	/** Reference to the city view model */
	protected final ICityPlayerProxyJFX city;
    @Autowired
    private TransferUtil transferUtil;
	@Autowired
	protected Locale locale;
	@Autowired
	private ModelTranslations translator;
	@Autowired
	protected MessageSource messageSource;
	@Autowired
	private ComputablePriceV2 computablePrice;
	@Autowired
	private UserActions actionFactory;
	@Autowired
	private ApplicationContext context;

	private EDialogType type;

	public BaseTradeDialog(ICityPlayerProxyJFX city, EDialogType type) {
		super();
		this.city = city;
		this.type = type;
	}
	/**
	 * Provide a group for the sub title.
	 * @param city proxy for which to create the subtitle
	 * @return group representing the subtitle
	 */
	protected abstract Group createSubTitle(ICityPlayerProxyJFX city);
	@PostConstruct
	private void initializeModelAndDialog() {
		Group subTitle = createSubTitle(city);
		subTitle.setLayoutX(2*FRAME_BORDER);
		subTitle.setLayoutY(80);
		getContent().add(subTitle);

		setTitle(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.BaseTradeDialog.title", new Object[]{}, locale.getCurrentLocal()));

		OpenPatricianSmallWaxButton btn1 = new OpenPatricianSmallWaxButton("1");
		btn1.setOnAction(arg0 -> {
			try {
				movableAmount.setValue(ETransferAmount.ONE);
			} catch (RuntimeException e) {
				log.error("Failed to change moveable amount", e);
			}
		});
		OpenPatricianSmallWaxButton btn5 = new OpenPatricianSmallWaxButton("5");
		btn5.setOnAction(arg0 -> {
			try {
				movableAmount.setValue(ETransferAmount.FIVE);
			} catch (RuntimeException e) {
				log.error("Failed to change moveable amount", e);
			}
		});
		String max = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.BaseTradeDialog.max", new Object[]{}, locale.getCurrentLocal());
		OpenPatricianSmallWaxButton btnMax = new OpenPatricianSmallWaxButton(max);
		btnMax.setOnAction(arg0 -> {
			try {
				movableAmount.setValue(ETransferAmount.MAX);
			} catch (RuntimeException e) {
				log.error("Failed to change moveable amount", e);
			}
		});
		addButtomControlButtons(btn1, btn5, btnMax);
		Table model = createModel(city);
		setModel(model);
	}


	private Table createModel(ICityPlayerProxyJFX cityProxy) {
    	Table model = new Table();

    	TableHeader header = new TableHeader(6);
    	header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.BaseTradeDialog.ware", new Object[]{}, locale.getCurrentLocal())));
    	header.add(new StaticTextTableCell(getTradeFromDestination()));
    	header.add(new StaticTextTableCell(messageSource.getMessage(getFirstMoveActionHeaderTextKey(), new Object[]{}, locale.getCurrentLocal())));
    	header.add(new StaticTextTableCell(messageSource.getMessage(getSecondMoveActionHeaderTextKey(), new Object[]{}, locale.getCurrentLocal())));
    	header.add(new StaticTextTableCell(getTradeToDestination()));
    	header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.BaseTradeDialog.avg-price", new Object[]{}, locale.getCurrentLocal())));
        header.setAligenment(0, HPos.RIGHT);
        header.setAligenment(1, HPos.RIGHT);
        header.setAligenment(2, HPos.CENTER);
        header.setAligenment(3, HPos.CENTER);
        header.setAligenment(4, HPos.RIGHT);
        header.setAligenment(5, HPos.RIGHT);
        model.setHeader(header);
        model.setAligenment(0, HPos.RIGHT);
        model.setAligenment(1, HPos.RIGHT);
        model.setAligenment(2, HPos.LEFT);
        model.setAligenment(3, HPos.LEFT);
        model.setAligenment(4, HPos.RIGHT);
        model.setAligenment(5, HPos.RIGHT);
    	model.setColumnWidth(100, 60, 72, 72, 60, 70);
    	ICity city = cityProxy.getCity();
    	Optional<ITradingOffice> optOffice = cityProxy.getPlayer().findTradingOffice(city);
    	INavigableVessel ship = cityProxy.getActiveShip();
       	for (final EWare ware : EWare.values()) {
    			TableRow row = new TableRow();
    			row.add(new StaticTextTableCell(translator.getLocalDisplayName(ware)));
				ReadOnlyIntegerProperty amountAvailableProp = getAvailableAmountProperty(city, ware, optOffice);
    			IntegerBinding amountToTransfer = getAmountProperty(city.getWare(ware).amountProperty());
    			if (ware.isBarrelSizedWare()) {
    				BarrelAmount barrelAmount = new BarrelAmount();
    				barrelAmount.amountProperty().bind(amountAvailableProp.asString());
    				row.add(new ControlTableCell(barrelAmount));
    			} else {
    				BaleAmount baleAmount = new BaleAmount();
    				baleAmount.amountProperty().bind(amountAvailableProp.asString());
    				row.add(new ControlTableCell(baleAmount));
    			}
    			ChangeAmountButton buyBtn = new ChangeAmountButton();
    			buyBtn.textProperty().bind(buyPrice(ware, amountAvailableProp, amountToTransfer));
    			buyBtn.setOneClickAction(() -> new BuyJFXProxyAction(ware, createTransferable(), actionFactory).run());
    			row.add(new ControlTableCell(buyBtn));
    			ChangeAmountButton sellBtn = new ChangeAmountButton();
    			sellBtn.textProperty().bind(sellPrice(ware, amountAvailableProp, amountToTransfer));
    			sellBtn.disableProperty().bind(createObservableShipLoad(ship, ware));
    			sellBtn.setOneClickAction(() -> new SellJFXProxyAction(ware, createTransferable(), actionFactory).run());
    			row.add(new ControlTableCell(sellBtn));
				ReadOnlyIntegerProperty storedAmountProperty = getStoredAmountProperty(ware, ship, optOffice);
    			if (ware.isBarrelSizedWare()) {
    				BarrelAmount barrelAmount = new BarrelAmount();
    				barrelAmount.amountProperty().bind(storedAmountProperty.asString());
    				row.add(new ControlTableCell(barrelAmount));
    			} else {
       				BaleAmount baleAmount = new BaleAmount();
    				baleAmount.amountProperty().bind(storedAmountProperty.asString());
    				row.add(new ControlTableCell(baleAmount));
 	    		}
    			CoinPrice coinPrice = new CoinPrice();
    			AmountablePrice<IWare> amountable = getStoredAmountablePrice(ware, ship, optOffice);
    			final NumberBinding avgPriceProperty = amountable.avgPriceProperty();
    			StringBinding avgPriceStringBinding = new StringBinding() {
    				{
    					super.bind(avgPriceProperty);
    				}
					
					@Override
					protected String computeValue() {
						return String.valueOf((int)Math.rint(avgPriceProperty.doubleValue()));
					}
				}; 
    			coinPrice.amountProperty().bind(avgPriceStringBinding);
    			row.add(new ControlTableCell(coinPrice));
    			model.add(row);
    		}
		return model;
	}

	/**
	 * Observable value to indicate if the ship has capacity for one more item. Default implementation
	 * returns always false.
	 * @param ship on which the ware is observed
	 * @param ware which is observed
	 * @return boolean observable
	 */
	protected ObservableValue<? extends Boolean> createObservableShipLoad(INavigableVessel ship, EWare ware) {
		return new SimpleBooleanProperty(false);
	}

	private ITransferableJFX createTransferable() {
		return TransferableState.builder()
				.city(city.getCity())
				.vessel(city.getActiveShip())
				.player(city.getPlayer())
				.dialogType(getDialogType())
				.movableAmount(movableAmount.get())
				.build();
	}

	/**
	 * Text key for the table header of the second action button, defaults to sell.
	 * @return text key for the second move action header
	 */
	protected String getSecondMoveActionHeaderTextKey() {
		return "ch.sahits.game.openpatrician.display.dialog.BaseTradeDialog.sell";
	}

	/**
	 * Text key for the table header of the first action button, defaults to buy.
	 * @return text key for the first move action header
     */
	protected String getFirstMoveActionHeaderTextKey() {
		return "ch.sahits.game.openpatrician.display.dialog.BaseTradeDialog.buy";
	}

	/**
	 * Trade from the city.
	 * @return localized string for the trading from destination
	 */
	protected String getTradeFromDestination() {
		return messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.BaseTradeDialog.city", new Object[]{}, locale.getCurrentLocal());
	}
	/**
	 * Trade to the ship.
	 * @return localized string for trading to the ship.
	 */
	protected String getTradeToDestination() {
		return messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.BaseTradeDialog.ship", new Object[]{}, locale.getCurrentLocal());
	}
    /**
     * Retrieve the stored amount
     * @param ware Reference to the ware
     * @param ship Reference to the ship
     * @param office Reference to the trading office
     * @return read only integer property representing the sored amount.
     */
	private ReadOnlyIntegerProperty getStoredAmountProperty(IWare ware, INavigableVessel ship, Optional<ITradingOffice> office){
		AmountablePrice<IWare> amounable = getStoredAmountablePrice(ware, ship,
				office);
		return amounable.amountProperty(); // differ between ship and convoi
	}
	/**
	 * Retrieve the amountable price of a ware from the ship or the warehouse
	 * @param ware for which the amountable price is retrieved
	 * @param ship vessel on which to look for the wares
	 * @param office trading office to look for the wares
	 * @return amountable price for the ware.
	 */
	private AmountablePrice<IWare> getStoredAmountablePrice(IWare ware, INavigableVessel ship,
			Optional<ITradingOffice> office) {
		AmountablePrice<IWare> amounable;
		switch (type) {
		case PORT_CITY_TO_SHIP:
		case PORT_STORAGE_TO_SHIP: // stored here means stored on the ship
			amounable = ship.getWare(ware);
			break;
		case PORT_CITY_TO_STORAGE:
			amounable = office.map(iTradingOffice -> iTradingOffice.getWare(ware))
					.orElseGet(() -> (AmountablePrice<IWare>) context.getBean("amountablePrice", 0, 0));
			break;
		default:
			// This should only occur for weapons which have their own dialog
			throw new IllegalStateException("Dialog type "+type+" not handled");
		}
		return amounable;
	}
	private ReadOnlyIntegerProperty getAvailableAmountProperty(ICity city, IWare ware, Optional<ITradingOffice> office) {
		AmountablePrice<IWare> available;
		switch (type) {
		case PORT_CITY_TO_SHIP:
		case PORT_CITY_TO_STORAGE:
			available = city.getWare(ware);
			break;
		default:
			available = office.map(iTradingOffice -> iTradingOffice.getWare(ware))
					.orElseGet(() -> (AmountablePrice<IWare>) context.getBean("amountablePrice", 0, 0));
			break;
		}
		return available.amountProperty();
	}
	/**
	 * Create the binding for the buy price for the indicated ware.
	 * @param ware to be bought
	 * @param availableAmount available amount of the ware
	 * @param amountToBuy amount to be bought
	 * @return buy price for the amount
	 */
	private StringBinding buyPrice(final IWare ware, final ReadOnlyIntegerProperty availableAmount, final IntegerBinding amountToBuy) {
		return new StringBinding() {
			{
				super.bind(availableAmount, amountToBuy);
			}
			
			@Override
			protected String computeValue() {
				switch (type) {
				case PORT_CITY_TO_SHIP:
				case PORT_CITY_TO_STORAGE:
					if (availableAmount.get() > 0){
						return String.valueOf(computablePrice.buyPrice(ware, availableAmount, amountToBuy));
					} else {
						return "0"; // cannot buy anything if nothing is there
					}
				default:
					return "<";
				}
			}
		};
	}
	/**
	 * Create a binding for the sell price for the indicated ware.
	 * @param ware to be sold
	 * @param availableAmount available amount
	 * @param amountToSell amount to be sold
	 * @return binding representing the sell price
	 */
	private StringBinding  sellPrice(final IWare ware, final ReadOnlyIntegerProperty availableAmount,final IntegerBinding amountToSell) {
		return new StringBinding() {
			{
				super.bind(availableAmount, amountToSell);
			}
			@Override
			protected String computeValue() {
				switch (type) {
				case PORT_CITY_TO_SHIP:
				case PORT_CITY_TO_STORAGE:
					if (movableAmount.get()==ETransferAmount.MAX){
						return String.valueOf(ware.getMaxValueSell());
					} else if (amountToSell.get() > 0){
						return String.valueOf(computablePrice.sellPrice(ware, availableAmount, amountToSell));
					} else {
						return String.valueOf(ware.getMaxValueSell());
					}
				default:
					return ">";
				}
			}
		};
	}
	



	/**
	 * Bind the amount that can be transferred on the maximal available amount
	 * as well as the selected transfer amount.
	 * @param maxAmount maximum amount
	 * @return integer binding representing the available amount.
	 */
	private IntegerBinding getAmountProperty(final ReadOnlyIntegerProperty maxAmount) {
		return new IntegerBinding() {
			{
				super.bind(movableAmount, maxAmount);
			}
			
			@Override
			protected int computeValue() {
				return transferUtil.calculateAvailableAmount(movableAmount, maxAmount.get());
			}
		};
	}

	@Override
	public EDialogType getDialogType() {
		return type;
	}


}
