package ch.sahits.game.openpatrician.display.dialog.events;

import ch.sahits.game.graphic.image.impl.SelectiveCachableXMLImageLoader;
import ch.sahits.game.openpatrician.clientserverinterface.model.event.MarriageAnnouncementState;
import ch.sahits.game.openpatrician.clientserverinterface.model.event.MarriageOfferState;
import ch.sahits.game.openpatrician.clientserverinterface.model.task.CelebrationTask;
import ch.sahits.game.openpatrician.display.dialog.DefaultSimpleDialog;
import ch.sahits.game.openpatrician.display.event.task.ClientTaskFactory;
import ch.sahits.game.openpatrician.engine.event.EventEngineState;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.openpatrician.javafx.control.PlaceHolder;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.clientserverinterface.service.DateService;
import ch.sahits.game.openpatrician.model.DisplayStateMessage;
import ch.sahits.game.openpatrician.model.EMessageCategory;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.event.TargetedEvent;
import ch.sahits.game.openpatrician.model.event.TimedUpdatableTaskList;
import ch.sahits.game.openpatrician.model.service.ModelTranslations;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import com.google.common.eventbus.AsyncEventBus;
import javafx.event.EventHandler;
import javafx.geometry.Pos;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.BorderPane;
import javafx.scene.layout.Pane;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;
import java.time.LocalDateTime;

/**
 * Dialog for the marriage offer.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Dec 07, 2016
 */
@Slf4j
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN})
public class MarriageOfferDialog extends DefaultSimpleDialog {

    private final MarriageOfferState state;
    @Autowired
    private ModelTranslations modelTranslations;
    @Autowired
    @Qualifier("xmlImageLoader")
    private SelectiveCachableXMLImageLoader imgLoader;
    @Autowired
    private Date date;
    @Autowired
    private DateService dateService;
    @Autowired
    private TimedUpdatableTaskList taskList;
    @Autowired
    private ClientTaskFactory taskFactory;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    private EventEngineState eventState;

    public MarriageOfferDialog(MarriageOfferState state) {
        super(true);
        this.state = state;
    }

    @PostConstruct
    private void initializeDialog() {
        String letterDate = modelTranslations.toDisplayString(state.getDate());
        String tillDate = modelTranslations.toDisplayString(state.getTillDate());
        addDecoratedText("letter.location.date", "dateLocation", Pos.CENTER_RIGHT, state.getLocation(), letterDate);
        addVerticalSpacer(20);
        if (state.getPlayer().getPersonalData().isMale()) {
            addDecoratedText("letter.salutation.male", "salutationMale", state.getPlayer().getLastName());
        } else {
            addDecoratedText("letter.salutation.female", "salutationFemale", state.getPlayer().getLastName());
        }
        addVerticalSpacer(20);
        if (state.getPlayer().getPersonalData().isMale()) {
            addDecoratedText("ch.sahits.game.openpatrician.display.dialog.event.MarriageOfferDialog.content.male", "contentMale", state.getSpouse().getName(), state.getSpouse().getLastName(), state.getSpouse().getBirthPlace().getName());
        } else {
            addDecoratedText("ch.sahits.game.openpatrician.display.dialog.event.MarriageOfferDialog.content.female", "contentFemale", state.getSpouse().getName(), state.getSpouse().getLastName(), state.getSpouse().getBirthPlace().getName());
        }
        BorderPane centered = new BorderPane();
        centered.setPrefWidth(WRAPPING_WIDTH);
        Pane pane = new Pane();
        centered.setCenter(pane);

        Image img = imgLoader.getImage("images/portraitBackground");
        ImageView background = new ImageView(img);
        background.setFitWidth(200);
        background.setPreserveRatio(true);
        String portraitResouce = state.getSpouse().getPortraitResourceName();
        ImageView portrait = new ImageView(imgLoader.loadImage(portraitResouce));
        portrait.setFitWidth(200);
        portrait.setPreserveRatio(true);
        img = imgLoader.getImage("images/retro-gold-frame-oval");
        ImageView frame = new ImageView(img);
        frame.setFitWidth(200);
        frame.setPreserveRatio(true);
        pane.getChildren().addAll(background, portrait, frame);
        PlaceHolder ph = new PlaceHolder((WRAPPING_WIDTH - 200)/2, 1);
        centered.setLeft(ph);
        addNode(centered, "portrait");
        addDecoratedText("letter.feedback.date", "deadline", Pos.CENTER, tillDate);
        addVerticalSpacer(20);
        addDecoratedText("letter.greetings", "grettings", Pos.CENTER);
        addDecoratedText("letter.fullname", "signature", Pos.CENTER_RIGHT, state.getFromFirstName(), state.getFromLastName());
        if (!dateService.isPast(state.getTillDate())) {
            OpenPatricianLargeWaxButton button = addActionButton("accept.offer", "acceptBtn");
            if (state.isAccepted()) {
                button.setDisable(true);
            } else {
                button.setOnAction(createAcceptOffer());
            }
        }
    }

    private EventHandler<MouseEvent> createAcceptOffer() {
        return mouseEvent -> {
            try {
                // post DisplayStateMessage
                LocalDateTime weddingDate = date.getCurrentDate().plusDays(21);
                MarriageAnnouncementState announcementState = MarriageAnnouncementState.builder()
                        .date(date.getCurrentDate())
                        .player(state.getPlayer())
                        .spouse(state.getSpouse())
                        .weddingDate(weddingDate)
                        .build();
                DisplayStateMessage message = new DisplayStateMessage(EMessageCategory.PERSONAL, "ch.sahits.game.openpatrician.engine.event.EventEngine.message.marriage.title", announcementState);
                clientServerEventBus.post(new TargetedEvent((IHumanPlayer) state.getPlayer(), message)); // TODO: 12.03.17 why do we have a targeted event here? we can sending it directly, as the user is the target.
                clientEventBus.post(announcementState);
                // Schedule the celebration
                CelebrationTask task = taskFactory.getMarriageCelebrationEvent(weddingDate, state.getPlayer(), state.getPlayer().getHometown(), state.getSpouse());
                taskList.add(task);
                state.setAccepted(true);
                executeOnCloseButtonClicked();
            } catch (RuntimeException e) {
                log.error("Failed to order marriage celebrations", e);
            }
        };
    }
}
