package ch.sahits.game.openpatrician.display.dialog.loaner;

import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.clientserverinterface.service.LoanerService;
import ch.sahits.game.openpatrician.display.ClientViewState;
import ch.sahits.game.openpatrician.display.dialog.DefaultSimpleDialog;
import ch.sahits.game.openpatrician.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.openpatrician.event.EViewChangeEvent;
import ch.sahits.game.openpatrician.event.NoticeBoardUpdate;
import ch.sahits.game.openpatrician.javafx.control.DecoratedText;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.openpatrician.model.ICompany;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.impl.IDebt;
import ch.sahits.game.openpatrician.model.city.impl.LoanerState;
import ch.sahits.game.openpatrician.model.service.ModelTranslations;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.ListType;
import ch.sahits.game.openpatrician.utilities.annotation.ObjectPropertyType;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.event.EventHandler;
import javafx.scene.Node;
import javafx.scene.input.MouseEvent;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.PostConstruct;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Jan 06, 2015
 */
@Slf4j
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class LoanerPaybackDialog extends DefaultSimpleDialog {

    public static final String DEFAULT_NON_SELECTED_STYLE = "-fx-fill: dimgray; -fx-text-fill: dimgray;";
    public static final String SELECTED_STYLE = "-fx-font-weight: bolder; -fx-fill: black;  -fx-text-fill: black;";


    @Autowired
    private ClientViewState viewState;
    @Autowired
    private ModelTranslations modelTranslations;
    @Autowired
    private LoanerService loanerService;

    private LoanerState loaner;
    private final ICityPlayerProxyJFX city;
    @ObjectPropertyType(IDebt.class)
    private ObjectProperty<IDebt> selectedDebt = new SimpleObjectProperty<>(null);
    @ListType(Node.class)
    private List<Node> loans = new ArrayList<>();

    public LoanerPaybackDialog(ICityPlayerProxyJFX city) {
        super(false);
        this.city = city;
    }
    @PostConstruct
    private void initializeDialog() {
        final ICity city1 = city.getCity();
        loaner = (LoanerState) loanerService.findLoaner(city1);
        setTitle(messageSource.getMessage("ch.sahits.game.openpatrician.display.notice.NoticeBoardFactory.repayLoan", new Object[]{}, locale.getCurrentLocal()));
        addVerticalSpacer(20);
        List<IDebt> debts = loanerService.findDebts(loaner, city.getPlayer());
        if (debts.isEmpty()) {
            addDecoratedText("ch.sahits.game.openpatrician.display.dialog.LoanerPaybackDialog.debtFree", "debtFree");
        } else {
            addDecoratedText("ch.sahits.game.openpatrician.display.dialog.LoanerPaybackDialog.subHeader", "introText");
            for (final IDebt debt : debts) {
                int payback = loanerService.calculatePaybacksum(debt);
                String cityName = city1.getName();
                final LocalDateTime dueDate1 = debt.getDueDate();
                String dueDate = modelTranslations.toDisplayString(dueDate1);
                final DecoratedText entry = addDecoratedText("ch.sahits.game.openpatrician.display.dialog.LoanerPaybackDialog.payback", "payback", payback, cityName, dueDate);
                entry.setOnMouseReleased(mouseEvent -> {
                    selectedDebt.setValue(debt);
                    deactivateAll();
                    entry.applyStyle(SELECTED_STYLE);
                });
                entry.applyStyle(DEFAULT_NON_SELECTED_STYLE);
                loans.add(entry);
            } // end for debts
            final OpenPatricianLargeWaxButton acceptBtn = addActionButton("ch.sahits.game.openpatrician.display.dialog.LoanerPaybackDialog.paybackBtn", "paybackBtn");
            acceptBtn.setDisable(true);
            selectedDebt.addListener((observableValue, oldDebt, newDebt) -> {
                if (newDebt != null) {
                    acceptBtn.setDisable(false);
                }
            });
            acceptBtn.setOnAction(createAcceptHandler());
        } // end else there are debts
    }
    private void deactivateAll() {
        for (Node node : loans) {
            if (node instanceof DecoratedText) {
                ((DecoratedText)node).applyStyle(DEFAULT_NON_SELECTED_STYLE);
            }
        }
    }


    private EventHandler<MouseEvent> createAcceptHandler() {
        return mouseEvent -> {
            try {
                final ICompany company = city.getPlayer().getCompany();
                long cash = company.getCash();
                final IDebt debt = selectedDebt.get();
                int debtTotal = loanerService.calculatePaybacksum(debt);
                if (cash < debtTotal) {
                    clearContent();
                    addDecoratedText("ch.sahits.game.openpatrician.display.dialog.LoanerPaybackDialog.noRepay", "noRepay");
                } else {
                    debt.getCreditor().receiveSum(debtTotal);
                    company.updateCash(-debtTotal);
                    loaner.getDebts().remove(debt);
                    executeOnCloseButtonClicked();
                }
            } catch (RuntimeException e) {
                log.error("Failed to pay back loan", e);
            }
        };
    }
    /**
     * Update the notice board and close the dialog.
     */
    @Override
    public void executeOnCloseButtonClicked() {
        ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.NOTICE_LOANER);
        clientEventBus.post(new NoticeBoardUpdate(proxy));
        super.executeOnCloseButtonClicked();
    }
}
