package ch.sahits.game.openpatrician.display.dialog.ship;

import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.sea.TravellingVessel;
import ch.sahits.game.openpatrician.model.sea.TravellingVessels;
import ch.sahits.game.openpatrician.model.ship.EShipTravelState;
import ch.sahits.game.openpatrician.model.ship.EShipType;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.binding.StringBinding;
import javafx.beans.property.ObjectProperty;
import javafx.geometry.Point2D;
import lombok.Getter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;

import java.util.Optional;

/**
 * UI model for an entry in the ship list dialog.
 * @author Andi Hotz, (c) Sahits GmbH, 2018
 * Created on Jul 12, 2018
 */
@Prototype
@ClassCategory({EClassCategory.PROTOTYPE_BEAN, EClassCategory.MODEL, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class ShipModelForShipList {
    @Getter
    private final IShip ship;

    @Autowired
    private ShipService shipService;

    @Autowired
    private TravellingVessels travellingVessels;
    @Autowired
    private MessageSource messageSource;
    @Autowired
    private Locale locale;

    public ShipModelForShipList(IShip ship) {
        this.ship = ship;
    }

    /**
     * If the ship belongs to a convoy retrieve the parent ship.
     * @return Optional parent reference, not present when a vessel has no parent (i.e. not a convoy member)
     */
    public Optional<ShipModelForShipList> getParent() {
        if (ship.parentShipProperty().getValue() != null) {
            return Optional.of(new ShipModelForShipList(ship.parentShipProperty().getValue()));
        } else {
            return Optional.empty();
        }
    }

    /**
     * Retrieve the name of the ship.
     * @return ship name.
     */
    public String getShipName() {
        return ship.getName();
    }

    /**
     * Retrieve the ship type.
     * @return ship type.
     */
    public EShipType getShipType() {
        return ship.getShipType();
    }

    /**
     * Property of the travel state.
     * @return observable of the travel state
     */
    public ObjectProperty<EShipTravelState> travelingState() {
        return ship.travelState();
    }

    /**
     * Destination city name of the travel. This may be an empty string.
     * @return string binding representing the destination
     */
    public StringBinding destination() {
        return new StringBinding() {
            {
                super.bind(ship.travelState());
            }
            @Override
            protected String computeValue() {
                switch (ship.travelState().get()) {
                    case ANCHOR:
                    case REPAIR: {
                        Optional<ICity> city = shipService.findCity(ship);
                        if (city.isPresent()) {
                            return city.get().getName();
                        }
                    }
                    case TRAVEL_TO_CITY: {
                        TravellingVessel travelling;
                        if (getParent().isPresent()) {
                            travelling = travellingVessels.getTravellingVessel(ship.parentShipProperty().get());
                        } else {
                            travelling = travellingVessels.getTravellingVessel(ship);
                        }
                        Point2D destination = travelling.getCalculatablePath().get(travelling.getCalculatablePath().size() - 1);
                        Optional<ICity> city = travellingVessels.findCity(destination);
                        if (city.isPresent()) {
                            return city.get().getName();
                        }
                    }
                    case EXPEDITION: {
                        return messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ship.ShipListDialog.expedition",
                                new Object[]{}, locale.getCurrentLocal());
                    }
                }

                return "";
            }
        };
    }

    /**
     * Retrieve the ships health.
     * @return ship health.
     */
    public int getHealth() {
        return 100-ship.getDamage();
    }

    /**
     * Binding of the loaded amount of wares in barrels.
     * @return loaded ware binding.
     */
    public IntegerBinding loadedWares() {
        return ship.getLoadBinding();
    }

    /**
     * Get the ship's value.
     * @return ship value.
     */
    public int getValue() {
        return ship.getCargoValue();
    }

    /**
     * Check if the represented ship belongs to the current player.
     * @param player to check against owner
     * @return true if player and ship owner match
     */
    public boolean isPlayersShip(IHumanPlayer player) {
        return ship.getOwner().equals(player);
    }

    /**
     * Check if the ship is on an auto trade route.
     * @return true if the ship is on a trade route.
     */
    public boolean isAutotrading() {
        return ship.getAutotrading().isPresent();
    }
    
}
