package ch.sahits.game.openpatrician.display.dialog.tavern;

import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.display.event.handler.impl.EventHandlerFactory;
import ch.sahits.game.openpatrician.display.event.task.ClientTaskFactory;
import ch.sahits.game.openpatrician.javafx.control.DecoratedText;
import ch.sahits.game.openpatrician.model.EMessageCategory;
import ch.sahits.game.openpatrician.model.people.ITransportTrader;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.ui.DialogTemplate;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import javafx.beans.binding.BooleanBinding;
import javafx.event.EventHandler;
import javafx.scene.input.MouseEvent;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.PostConstruct;
import java.time.LocalDateTime;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Random;

import static com.google.common.collect.Lists.newArrayList;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 *         Created on Dec 26, 2013
 */
@Slf4j
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class TavernTransportTraderDialog extends TavernBaseSideRoomPersonDialog {
    @Autowired
    private Random rnd;
    @Autowired
    private ClientTaskFactory taskFactory;
    @Autowired
    private EventHandlerFactory eventHandlerFactory;

    public TavernTransportTraderDialog(ICityPlayerProxyJFX city) {
        super(city, city.getCity().getCityState().getTavernState().getTransportTrader());
    }
    @PostConstruct
    private void initializeDialog() {
        setTitle(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.TavernTransportTraderDialog.title", new Object[]{}, locale.getCurrentLocal()));
        initializeButtons();
        ITransportTrader trader = (ITransportTrader) currentPerson.get();

        String loadedText = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.TavernTransportTraderDialog.dialog", new Object[]{city.getCity().getName(), trader.getDestination().getName()}, locale.getCurrentLocal());


        HashMap<String, Object> parameters = new HashMap<>();
        parameters.put("amount", trader.getNeededCapacity());
        DecoratedText dt = decoratedTextFactory.createDecoratedText(loadedText, parameters);


        getContent().addAll(dt);
    }

    @Override
    protected EventHandler<MouseEvent> createAcceptHandler() {
        return mouseEvent -> {
            try {
                List<IShip> ships = newArrayList(city.getPlayersShips()); // we need a copy
                Collections.shuffle(ships);
                ITransportTrader trader = (ITransportTrader) currentPerson.get();
                for (IShip ship : ships) {
                    final int neededCapacity = trader.getNeededCapacity();
                    if (ship.getCapacity() > neededCapacity * 10) {
                        ship.setOccupiedSpace(neededCapacity * 10);
                        int minutes = rnd.nextInt(600);
                        LocalDateTime postMessage = date.getCurrentDate().plusMinutes(minutes);
                        DialogTemplate template = DialogTemplate.builder()
                                .closable(true)
                                .titleKey("ch.sahits.game.openpatrician.display.dialog.TavernTransportTraderDialog.waresLodedTitle")
                                .messageKey("ch.sahits.game.openpatrician.display.dialog.TavernTransportTraderDialog.acceptMessage")
                                .messageArgs(new Object[]{trader.getDestination().getName(), ship.getName()})
                                .build();
                        taskFactory.getPostponedDisplayDialogMessage(postMessage, EMessageCategory.TRADE, template);
                        trader.leave();
                        eventHandlerFactory.getUnloadOnArrivalHandler(ship, trader.getDestination(), neededCapacity * 10);
                        break;
                    }
                }
                executeOnCloseButtonClicked();
            } catch (RuntimeException e) {
                log.error("Failed to accept transportation offer", e);
            }
        };
    }

    @Override
    protected BooleanBinding disableAcceptButton() {
        return new BooleanBinding() {
            {
                super.bind(city.getPlayersShips());
                // todo: andi 23/12/13: should not consider ships that are automated in some way (these are the ships which's capacity may change)
            }
            @Override
            protected boolean computeValue() {
                ITransportTrader trader = (ITransportTrader) currentPerson.get();
                return tavernDialogUtil.noShipWithEnoughCapacityAvailable(city, trader.getNeededCapacity());
            }
        };
    }

}
