package ch.sahits.game.openpatrician.display.event.handler.impl;

import ch.sahits.game.openpatrician.clientserverinterface.service.DialogTemplateParameterSupplier;
import ch.sahits.game.openpatrician.clientserverinterface.service.EDialogTemplateType;
import ch.sahits.game.openpatrician.clientserverinterface.service.LinearDeadlinePremiumCalculator;
import ch.sahits.game.openpatrician.display.event.data.CourierPatrolStateInternal;
import ch.sahits.game.openpatrician.display.event.data.PatrolInternalState;
import ch.sahits.game.openpatrician.display.event.handler.IAttackListener;
import ch.sahits.game.openpatrician.javafx.control.TextSizingUtility;
import ch.sahits.game.openpatrician.javafx.model.EDialogType;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.event.TimedTask;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.ui.DialogTemplate;
import ch.sahits.game.openpatrician.model.ui.IDialogState;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Feb 16, 2014
 */
@Slf4j
@Service
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class EventHandlerFactory {
    @Autowired
    private ApplicationContext context;
    @Autowired
    private TextSizingUtility sizeing;

    /**
     * Retrieve the event handler for delivering packages
     * @param city to which should be delivered
     * @param ship that delivers
     * @param deadline until the delivery has to take place
     * @param premium to be payed out on delivery
     * @param amount of the occupied space in barrels.
     * @return event handler for delivering packages.
     */
    public DeliverPackages getDeliverPackagesHandler(ICity city, IShip ship, LocalDateTime deadline, int premium, int amount) {
        Object[] args = new Object[]{city, ship, deadline, premium,  amount};
        return (DeliverPackages)  context.getBean("deliverPackages", args);
    }

    public ShipArrivalForLoading getShipArrivalForLoadingHandler(ICity city, IPlayer player, LocalDateTime deadline, int requiredAmount, int premium) {
        Object[] args = new Object[]{city, player, deadline, requiredAmount, premium};
        return (ShipArrivalForLoading)  context.getBean("shipArrivalForLoading", args);
    }

    /**
     * Event handler for a passenger leaving the ship in a city
     * @param ship that is left
     * @param city destination
     * @param shipsOwner owner of the ship
     * @param deadline after which the passenger leaves without premium
     * @param dialogType enum type identifying the dialog that is to be created.
     * @param dialogParameterSupplier supplier function providing the arguments for the dialog template.
     * @param premiumCalculator supplier lambda function to calculate the premium to be payed upon arrival.
     * @return bean handling the passenger leaving the ship on arrival at the destination.
     */
    public PassengerLeavesShipOnArrival getPassengerLeavingOnShipArrivalHandler(IShip ship, ICity city, IPlayer shipsOwner, LocalDateTime deadline, EDialogTemplateType dialogType,
                                                                                DialogTemplateParameterSupplier dialogParameterSupplier, LinearDeadlinePremiumCalculator premiumCalculator) {
        Object[] args = new Object[]{ship, city, shipsOwner, deadline, dialogType, dialogParameterSupplier, premiumCalculator};
        return (PassengerLeavesShipOnArrival)  context.getBean("passengerLeavesShipOnArrival", args);
    }

    /**
     * Create the handler for unloading anonymous ware and register it with the event bus.
     * @param ship ship that has the ware loaded
     * @param city into which the ship should arrive
     * @param loadedAmount amount of anonymous ware loaded.
     * @return Event handler for arrivals
     */
    public UnloadOnArrival getUnloadOnArrivalHandler(IShip ship, ICity city, int loadedAmount) {
        return (UnloadOnArrival) context.getBean("unloadOnArrival", new Object[]{ship, city, loadedAmount});
    }

    /**
     * Retrieve the event handler for the next patrol station
     * @param ship that is on patrol
     * @param city next city
     * @param state patrol state
     * @param player for which to create the next courier patrol station
     * @param deadline until the patrol has to reach the city
     * @return next patrol station for the courier
     */
    public NextCourierPatrolStation getNextCourierPatrolStationHanlder(IShip ship, ICity city, CourierPatrolStateInternal state, IPlayer player, LocalDateTime deadline) {
        Object[] args = new Object[]{ship, city, state, player, deadline};
        return (NextCourierPatrolStation)  context.getBean("nextCourierPatrolStation", args);
    }

    /**
     * Retrieve the event handler for a new view change runnable.
     * @param selected notice item that is selected.
     * @return new instance of a view changable runnable.
     */
    public ViewChangeActionRunnable getViewChangeActionRunnable(EDialogType selected) {
        Object[] args = new Object[]{selected};
        return (ViewChangeActionRunnable)  context.getBean("viewChangeActionRunnable", args);
    }

    /**
     * Retrieve the event handler for a new dialog based on a template
     * @param template dialog template to inject into the view changable
     * @return new view changable based on a dialog template
     */
    public ViewChangeDialogTemplateRunnable getViewChangeActionRunnable(DialogTemplate template) {
        Object[] args = new Object[]{template};
        return (ViewChangeDialogTemplateRunnable)  context.getBean("viewChangeDialogTemplateRunnable", args);
    }


    /**
     * Retrieve the event handler for a new dialog based on a state
     * @param state dialog state for the vhiew change dialog runnable
     * @return new instance of a view change runable based on a dialog state
     */
    public Runnable getViewChangeActionRunnable(IDialogState state) {
        Object[] args = new Object[]{state};
        return (ViewChangeDialogStateRunnable)  context.getBean("viewChangeDialogStateRunnable", args);
    }

    /**
     * Get the attack listener for a standard patrol.
     * @param ship doing the patrol
     * @param state patrol state
     * @return new instance of an attack listener
     */
    public IAttackListener getAttackListener(IShip ship, PatrolInternalState state) {
        Object[] args = new Object[]{ship, state};
        return (AttackListener)  context.getBean("attackListener", args);
    }

    /**
     * Retrieve the next patrol station event listener
     * @param ship doing the patrrol
     * @param city next city
     * @param state current state
     * @param player for which to create the next patrol station
     * @param deadline until when the city has to be reached
     * @param atackListener listener for attacks
     * @return new instance of the next station on a patrol
     */
    public NextPatrolStation getNextPatrolStation(INavigableVessel ship, ICity city, PatrolInternalState state, IPlayer player, LocalDateTime deadline, IAttackListener atackListener) {
        Object[] args = new Object[]{ship, city, state, player, deadline, atackListener};
        log.info("Retrieve next patrol station for {}", state);
        return (NextPatrolStation)  context.getBean("nextPatrolStation", args);
    }


    /**
     * Retrieve the concurrent ship order
     * @param opponent player
     * @param player the order executor
     * @param offerdTime time to execute the task
     * @param premium on successful execution
     * @param reconciliation timer task for the reconcialiation
     * @return new instance of an order to sing a concurrents ship.
     */
    public SinkConcurrentsShipOrder getSinkConcurrentsShipOrder(IPlayer opponent,
                                                                IPlayer player, int offerdTime, int premium, TimedTask reconciliation) {
        Object[] args = new Object[]{opponent, player, offerdTime, premium, reconciliation};
        return (SinkConcurrentsShipOrder)  context.getBean("sinkConcurrentsShipOrder", args);
    }


}
