package ch.sahits.game.openpatrician.display.event.handler.impl;

import ch.sahits.game.openpatrician.clientserverinterface.service.ClientServerFactory;
import ch.sahits.game.openpatrician.clientserverinterface.service.DialogTemplateFactory;
import ch.sahits.game.openpatrician.clientserverinterface.service.DialogTemplateParameterSupplier;
import ch.sahits.game.openpatrician.clientserverinterface.service.EDialogTemplateType;
import ch.sahits.game.openpatrician.display.event.data.CourierPatrolStateInternal;
import ch.sahits.game.openpatrician.display.event.task.ClientTaskFactory;
import ch.sahits.game.openpatrician.event.data.ShipEntersPortEvent;
import ch.sahits.game.openpatrician.event.handler.ShipEntersPortEventListener;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.DisplayTemplateMessage;
import ch.sahits.game.openpatrician.model.EMessageCategory;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.event.TargetedEvent;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.ui.DialogTemplate;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.time.LocalDateTime;
import java.util.Optional;
import java.util.Random;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Feb 16, 2014
 */
@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class NextCourierPatrolStation  extends ShipEntersPortEventListener {
    /** Next station may be null if it is the last */
    private final CourierPatrolStateInternal state;
    private final IPlayer player;
    private final LocalDateTime deadline;
    @Autowired
    private Date date;
    @Autowired
    @XStreamOmitField
    private ClientTaskFactory taskFactory;
    @Autowired
    private Random rnd;

    @Autowired
    @Qualifier("clientEventBus")
    @XStreamOmitField
    private AsyncEventBus clientEventBus;
    @Autowired
    @XStreamOmitField
    private ClientServerFactory clientServerFactory;
    @Autowired
    @XStreamOmitField
    private DialogTemplateFactory dialogTemplateFactory;
    @Autowired
    @XStreamOmitField
    private Locale locale;
    @Autowired
    @XStreamOmitField
    private MessageSource messageSource;

    /**
     * Create the next patrol station listener. This listener adds itself to the event queue and removes the first city from the
     * patrol state
     * @param ship for which to construct the next patrol station
     * @param city of the next patrol station
     * @param state internal patrol state
     * @param player patroling player
     * @param deadline until when the patrol has to reach the city.
     */
    public NextCourierPatrolStation(IShip ship, ICity city, CourierPatrolStateInternal state, IPlayer player, LocalDateTime deadline) {
        super(ship, city);
        this.state = state;
        this.player = player;
        this.deadline = deadline;
        state.removeFirstCity();
    }
    @PostConstruct
    private void initializeEventBus() {
        clientServerEventBus.register(this);
    }
    @PreDestroy
    private void unregisterEventBus() { clientServerEventBus.unregister(this);}

    @Override
    @Subscribe
    public void handleShipEntersPort(ShipEntersPortEvent event) {
        super.handleShipEntersPort(event);
        LocalDateTime now = date.getCurrentDate();
        if (now.isAfter(deadline)) {
            if (player instanceof IHumanPlayer) {
                DialogTemplateParameterSupplier parameterSupplier = new DialogTemplateParameterSupplier(new Object[]{getShip().getName(), event.getCity().getName(), clientServerFactory.dateSupplier()});
                DialogTemplate template = dialogTemplateFactory.createDialogTemplate(EDialogTemplateType.COURIER_SERVICE_FINISHED_UNSUCCESSFUL, parameterSupplier);
                DisplayTemplateMessage message = new DisplayTemplateMessage(EMessageCategory.PERSONAL, "ch.sahits.game.event.handler.impl.NextCourierPatrolStation.unreliable", template);
                TargetedEvent tagetDisplayMsg = new TargetedEvent((IHumanPlayer) player, message);
                clientServerEventBus.post(tagetDisplayMsg);
            }
            ((IShip)getShip()).leavePassenger();
            clientServerEventBus.unregister(this);
        }
    }

    @Override
    public boolean execute(ICity city) {
        Optional<ICity> nextCity = state.nextDestination();
        if (!nextCity.isPresent()){ // last leg
            int premium = state.getPremium();
            if (player instanceof IHumanPlayer) {
                player.getCompany().updateCash(premium);
            } else {
                player.getCompany().updateCashDirectly(premium);
            }
            ((IShip)getShip()).leavePassenger();
            if (player instanceof IHumanPlayer) {
                DialogTemplateParameterSupplier parameterSupplier = new DialogTemplateParameterSupplier(new Object[]{getShip().getName(), city.getName(), clientServerFactory.dateSupplier(), premium});
                DialogTemplate template = dialogTemplateFactory.createDialogTemplate(EDialogTemplateType.COURIER_SERVICE_FINISHED_SUCCESSFUL, parameterSupplier);
                DisplayTemplateMessage message = new DisplayTemplateMessage(EMessageCategory.TRADE, "ch.sahits.game.event.handler.impl.NextCourierPatrolStation.salary", template, premium);
                TargetedEvent tagetDisplayMsg = new TargetedEvent((IHumanPlayer) player, message);
                clientServerEventBus.post(tagetDisplayMsg);
            }

            unregisterEventBus();
        } else { // next leg
            ICity next = nextCity.get();
            state.removeFirstCity();
            int minutes = rnd.nextInt(600);

            LocalDateTime postMessage = date.getCurrentDate().plusMinutes(minutes);
            DialogTemplate template = DialogTemplate.builder()
                    .closable(true)
                    .titleKey("ch.sahits.game.event.handler.impl.NextCourierPatrolStation.nextStation.title")
                    .messageKey("ch.sahits.game.event.handler.impl.NextCourierPatrolStation.nextStation")
                    .messageArgs(new Object[]{getShip().getName(), next.getName()})
                    .build();
            taskFactory.getPostponedDisplayDialogMessage(postMessage, EMessageCategory.TRADE, template);
            LocalDateTime deadline = date.getCurrentDate().plusDays(6);
            new NextCourierPatrolStation(((IShip)getShip()), state.nextDestination().get(), state, player, deadline);
        }
        return true;
    }
}
