package ch.sahits.game.openpatrician.display.event.task;

import ch.sahits.game.openpatrician.clientserverinterface.model.event.CelebrationSuccess;
import ch.sahits.game.openpatrician.clientserverinterface.service.CelebrationService;
import ch.sahits.game.openpatrician.model.DisplayTemplateMessage;
import ch.sahits.game.openpatrician.model.EMessageCategory;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.EPopulationClass;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.event.TargetedEvent;
import ch.sahits.game.openpatrician.model.event.TimedTask;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.ui.DialogTemplate;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import com.google.common.eventbus.AsyncEventBus;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.Map;

/**
 * Timed task to deliver wares for a wedding.
 * @author Andi Hotz, (c) Sahits GmbH, 2018
 * Created on Oct 28, 2018
 */
@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class MarriageFeastDeliveryTask extends TimedTask {
    @Autowired
    @Qualifier("clientEventBus")
    @XStreamOmitField
    private AsyncEventBus clientEventBus;
    @Autowired
    @XStreamOmitField
    private CelebrationService celebrationService;

    private IPlayer player;

    private ICity city;

    public MarriageFeastDeliveryTask(LocalDateTime deadLine, IPlayer player, ICity city) {
        this.player = player;
        this.city = city;
        setExecutionTime(deadLine);
    }

    @Override
    public void run() {
        // Grab all food from trading office
        Map<EPopulationClass, Integer> nbGuests = celebrationService.calculateAttendees(city);
        Map<IWare, Integer> requiredAmount = celebrationService.calculateRequiredAmounts(nbGuests);
        ITradingOffice office = player.findTradingOffice(city).get();
        Map<IWare, Integer> removedAmount = new HashMap<>();
        Map<IWare, Integer> prices = new HashMap<>();
        requiredAmount.entrySet().stream().forEach(entry -> {
            int availableAmount = office.getWare(entry.getKey()).getAmount();
            prices.put(entry.getKey(), office.getWare(entry.getKey()).getAVGPrice());
            office.move(entry.getKey(), -availableAmount, 0);
            removedAmount.put(entry.getKey(), availableAmount);
        });
        // Check if it is enough for a feast
        int sum = 0;
        for (IWare ware : removedAmount.keySet()) {
            int price = prices.get(ware);
            sum += price * removedAmount.get(ware);
        }
        // decide on the price: at cost if there was to less, +10% if it was sufficient, +15% if there was something left.
        double bonusFactor;
        if (wereWaresMissing(requiredAmount, removedAmount)) {
            bonusFactor = 1;
        } else if (wereWaresExact(requiredAmount, removedAmount)) {
            bonusFactor = 1.1;
        } else {
            bonusFactor = 1.15;
        }
        // Pay
        long payout = Math.round(sum * bonusFactor);
        player.getCompany().updateCashDirectly(payout);
        if (bonusFactor > 1) {
            city.getReputation(player).update(200);
        }
        if (player instanceof IHumanPlayer) {
            String messageKey;
            if (bonusFactor > 1) {
                messageKey = "ch.sahits.game.openpatrician.display.event.task.MarriageFeastDeliveryTask.success";
            } else {
                messageKey = "ch.sahits.game.openpatrician.display.event.task.MarriageFeastDeliveryTask.flop";
            }
            DialogTemplate template = DialogTemplate.builder()
                    .closable(true)
                    .titleKey("ch.sahits.game.openpatrician.display.event.task.MarriageFeastDeliveryTask.title")
                    .messageKey(messageKey)
                    .messageArgs(new Object[]{payout})
                    .build();
            DisplayTemplateMessage message = new DisplayTemplateMessage(EMessageCategory.PERSONAL, "ch.sahits.game.openpatrician.display.event.task.MarriageFeastDeliveryTask.title", template);
            TargetedEvent tagetDisplayMsg = new TargetedEvent((IHumanPlayer) player, message);
            clientEventBus.post(tagetDisplayMsg);
        }
    }

    /**
     * Check if there was any ware that's amount was less than required.
     * @param requiredAmount
     * @param removedAmount
     * @return
     */
    private boolean wereWaresMissing(Map<IWare,Integer> requiredAmount, Map<IWare,Integer> removedAmount) {
        for (IWare ware : requiredAmount.keySet()) {
            if (requiredAmount.get(ware) > removedAmount.get(ware)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Check if the required amounts were exactly the provided amounts
     * @param requiredAmount
     * @param removedAmount
     * @return
     */
    private boolean wereWaresExact(Map<IWare,Integer> requiredAmount, Map<IWare,Integer> removedAmount) {
        for (IWare ware : requiredAmount.keySet()) {
            if (requiredAmount.get(ware) != removedAmount.get(ware)) {
                return false;
            }
        }
        return true;
    }

    @Override
    public String toString() {
        return "MarriageFeastDeliveryTask{" +
                ", execution at " + getExecutionTime() +
                ", player=" + player.getUuid() + " " + player.getName() + " " + player.getLastName() +
                ", city=" + city +
                '}';
    }
}
