package ch.sahits.game.openpatrician.display.javafx.control;

import ch.sahits.game.graphic.image.IDataImageLoader;
import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.display.ClientViewState;
import ch.sahits.game.openpatrician.display.SceneChangeable;
import ch.sahits.game.openpatrician.display.event.handler.impl.EventHandlerFactory;
import ch.sahits.game.openpatrician.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.openpatrician.display.service.UIFactory;
import ch.sahits.game.openpatrician.event.EViewChangeEvent;
import ch.sahits.game.openpatrician.event.NoticeBoardUpdate;
import ch.sahits.game.openpatrician.javafx.event.WrappedDialogType;
import ch.sahits.game.openpatrician.javafx.model.EDialogType;
import ch.sahits.game.openpatrician.model.ship.ConvoyList;
import ch.sahits.game.openpatrician.model.ship.IConvoy;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import com.google.common.eventbus.AsyncEventBus;
import javafx.application.Platform;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.property.ObjectProperty;
import javafx.scene.Group;
import javafx.scene.control.Label;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.layout.HBox;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationContext;

import javax.annotation.PostConstruct;

/**
 * Control representing the main menu.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Nov 1, 2013
 *
 */
@Slf4j
@ClassCategory({EClassCategory.SINGLETON_BEAN, EClassCategory.JAVAFX})
public class SubMenu extends Group implements ISubMenuControl {
	@Autowired
	@Qualifier("xmlImageLoader")
	private IDataImageLoader imageLoader;
	@Autowired
	private ClientViewState viewState;
	@Autowired
	private ShipService shipService;
	@Autowired
	private ConvoyList convoyList;
	@Autowired
	@Qualifier("clientEventBus")
	private AsyncEventBus clientEventBus;
	@Autowired
	private UIFactory uiFactory;
	@Autowired
	private IMainMenuControl mainMenu;
    @Autowired
    private ApplicationContext context;
    @Autowired
    private EventHandlerFactory eventHandlerFactory;
    
	private SceneChangeable sceneChangeable;
	private HBox iconBar;

	public SubMenu() {
		setManaged(false);
		getStylesheets().add(getClass().getResource(getClass().getSimpleName()+".css").toExternalForm());

	}
	@PostConstruct
	private void initializeControls() {
		mainMenu.registerSubNavigationCallback(this);
		sceneChangeable = uiFactory.getApplicationWindow();
		iconBar = new HBox(2);
		// Layout values are dependent on the label scale
		iconBar.setLayoutX(2);
		iconBar.setLayoutY(6);
		getChildren().addAll(iconBar);
	}

	private void clearIcons() {
		iconBar.getChildren().clear();
	}


	private void setupMessagesSubNavigation() {
		final Image officialImg = imageLoader.getImage("icons/32/crown-seal-icon");
		final Image tradeImg = imageLoader.getImage("icons/32/coin-seal-icon");
		final Image personalImg = imageLoader.getImage("icons/32/coatOfArms-seal-icon");
		final Image configImg = imageLoader.getImage("icons/32/seal-stamp-icon");
		ImageView officialGraphic = new ImageView(officialImg);
		ImageView tradeGraphic = new ImageView(tradeImg);
		ImageView personalGraphic = new ImageView(personalImg);
		ImageView configGraphic = new ImageView(configImg);
		Label officialLbl = new Label();
		officialLbl.setGraphic(officialGraphic);
		officialLbl.setOnMouseReleased(evt -> {
			ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.OFFICIAL_MESSAGES);
			NoticeBoardUpdate noticeBoardUpdate = new NoticeBoardUpdate(proxy);
			clientEventBus.post(noticeBoardUpdate);
		});
		Label tradeLbl = new Label();
		tradeLbl.setGraphic(tradeGraphic);
		tradeLbl.setOnMouseReleased(evt -> {
			ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.TRADE_MESSAGES);
			NoticeBoardUpdate noticeBoardUpdate = new NoticeBoardUpdate(proxy);
			clientEventBus.post(noticeBoardUpdate);
		});
		Label personalLbl = new Label();
		personalLbl.setGraphic(personalGraphic);
		personalLbl.setOnMouseReleased(evt -> {
			ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.PERSONAL_MESSAGES);
			NoticeBoardUpdate noticeBoardUpdate = new NoticeBoardUpdate(proxy);
			clientEventBus.post(noticeBoardUpdate);
		});
		Label configLbl = new Label();
		configLbl.setGraphic(configGraphic);
		final Runnable run = eventHandlerFactory.getViewChangeActionRunnable(EDialogType.MESSAGE_CONFIGURATION);
		configLbl.setOnMouseReleased(evt -> {
			run.run();
		});

		iconBar.getChildren().addAll(tradeLbl, personalLbl, officialLbl, configLbl);

	}

	private void setupShipSubNavigation() {
		final Image wareImg = imageLoader.getImage("icons/32/icon_barrel");
		final Image sailorImg = imageLoader.getImage("icons/32/icon_head");
		final Image weaponImg = imageLoader.getImage("icons/32/bombard-icon");
		final Image convoyImg = imageLoader.getImage("icons/32/icon_convoy");
		final Image leaveConvoyImg = imageLoader.getImage("icons/32/icon_leave_convoy");
		final Image joinConvoyImg = imageLoader.getImage("icons/32/icon_join_convoy");
		final Image autoTradeImg = imageLoader.getImage("icons/32/icon_autotrade");
		ImageView wareGraphic = new ImageView(wareImg);
		ImageView sailorGraphic = new ImageView(sailorImg);
		ImageView weaponGraphic = new ImageView(weaponImg);
		ImageView convoyGraphic = new ImageView(convoyImg);
		ImageView convoyJoinLeaveGraphic = new ImageView(joinConvoyImg);
		BooleanBinding inConvoyBinding = inConvoy();
		if (inConvoyBinding.get()) {
            convoyJoinLeaveGraphic.setImage(leaveConvoyImg);
        } else {
            convoyJoinLeaveGraphic.setImage(joinConvoyImg);
        }
		ImageView autoTradeGraphic = new ImageView(autoTradeImg);

		Label wareLbl = new Label();
		wareLbl.setGraphic(wareGraphic);
		wareLbl.setOnMouseReleased(evt -> {
			ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.NOTICE_SHIP_WARE_INFO);
			NoticeBoardUpdate noticeBoardUpdate = new NoticeBoardUpdate(proxy);
			clientEventBus.post(noticeBoardUpdate);
		});

		Label sailorLbl = new Label();
		sailorLbl.setGraphic(sailorGraphic);
		BooleanBinding disableShipInfo = new BooleanBinding() {
			{
				super.bind(viewState.getCurrentCityProxy().get().activeShipProperty());
			}
			@Override
			protected boolean computeValue() {
				return !(viewState.getCurrentCityProxy().get().getActiveShip() instanceof IShip);
			}
		};
		sailorLbl.disableProperty().bind(disableShipInfo);
		sailorLbl.setOnMouseReleased(evt -> {
			ICityPlayerProxyJFX proxy = viewState.getCurrentCityProxy().get();
			if (proxy.getActiveShip() instanceof IShip) {
				Object[] args = new Object[]{EDialogType.SHIP_INFO};
				final Runnable runnable1 = (Runnable) context.getBean("viewChangeActionRunnable", args);
				IShip ship = (IShip) proxy.getActiveShip();
				proxy.activateShip(ship);
				runnable1.run();
			}
		});

		Label weaponLbl = new Label();
		weaponLbl.setGraphic(weaponGraphic);
		BooleanBinding disableWeapon = new BooleanBinding() {
			{
				super.bind(viewState.getCurrentCityProxy().get().activeShipProperty());
			}
			@Override
			protected boolean computeValue() {
				return !(viewState.getCurrentCityProxy().get().getActiveShip() instanceof IShip);
			}
		};
		weaponLbl.disableProperty().bind(disableWeapon);
		weaponLbl.setOnMouseReleased(evt -> {
			ICityPlayerProxyJFX proxy = viewState.getCurrentCityProxy().get();
			if (proxy.getActiveShip() instanceof IShip) {
				Object[] args = new Object[]{EDialogType.SHIP_WEAPON_INFO};
				final Runnable runnable1 = (Runnable) context.getBean("viewChangeActionRunnable", args);
				IShip ship = (IShip) proxy.getActiveShip();
				proxy.activateShip(ship);
				runnable1.run();
			}
		});

		Label convoyLbl = new Label();
		convoyLbl.setGraphic(convoyGraphic);
		convoyLbl.disableProperty().bind(inConvoyBinding.not());

		Label convoyJoinLeaveLbl = new Label();
        convoyJoinLeaveLbl.setGraphic(convoyJoinLeaveGraphic);
        setupConvoyJoinLeaveLabel(leaveConvoyImg, joinConvoyImg, convoyJoinLeaveGraphic, convoyJoinLeaveLbl, inConvoyBinding.getValue());
        inConvoyBinding.addListener((observable, oldValue, newValue) ->
				setupConvoyJoinLeaveLabel(leaveConvoyImg, joinConvoyImg, convoyJoinLeaveGraphic, convoyJoinLeaveLbl, newValue));
		// Dependent on if ship is in convoy or not.

		Label autoTradeLbl = new Label();
		autoTradeLbl.setGraphic(autoTradeGraphic);
		autoTradeLbl.disableProperty().bind(isOrlegCapable().not());
		// bind disable to Orleg capability

		iconBar.getChildren().addAll(wareLbl, sailorLbl, weaponLbl, convoyLbl, convoyJoinLeaveLbl, autoTradeLbl);
	}

    private void setupConvoyJoinLeaveLabel(Image leaveConvoyImg, Image joinConvoyImg, ImageView convoyJoinLeaveGraphic, Label convoyJoinLeaveLbl, Boolean newValue) {
		ICityPlayerProxyJFX proxy = viewState.getCurrentCityProxy().get();
        INavigableVessel vessel = proxy.getActiveShip();
		if (newValue) { // part of convoy
            convoyJoinLeaveGraphic.setImage(leaveConvoyImg);
            IShip ship;
            if (vessel instanceof IShip) {  // Leave convoy
                ship = (IShip) vessel;
				IShip orleg = ship.parentShipProperty().get();
				IConvoy convoy =  convoyList.findConvoyOfOrleg(orleg);
                WrappedDialogType wrapper = new WrappedDialogType(EDialogType.LEAVE_CONVOY, convoy, ship);
                final Runnable run = (Runnable) context.getBean("viewChangeWrappedRunnable", wrapper);
                convoyJoinLeaveLbl.setOnMouseReleased(evt -> {
                    proxy.activateShip(vessel);
                    run.run();
                });
            } else { // Disolve convoy
                convoyJoinLeaveLbl.setOnMouseReleased(evt -> openConvoyDialog(proxy, vessel));
            }
        } else { // Not part of convoy
            convoyJoinLeaveGraphic.setImage(joinConvoyImg);
            convoyJoinLeaveLbl.setOnMouseReleased(evt -> openConvoyDialog(proxy, vessel));
        }
    }

    private void openConvoyDialog(ICityPlayerProxyJFX proxy, INavigableVessel vessel) {
        Object[] args = new Object[]{EDialogType.SHIP_CONVOY};
        final Runnable runnable1 = (Runnable) context.getBean("viewChangeActionRunnable", args);
        proxy.activateShip(vessel);
        runnable1.run();
    }

	@Override
	public void closeSubMenuNavigation() {
		 Platform.runLater(() -> clearIcons());
	}

	@Override
	public void displayShipSubNavigation() {
		clearIcons();
		setupShipSubNavigation();
		viewState.getCurrentCityProxy().get().activeShipProperty().addListener(evt ->
			Platform.runLater(() -> {
				clearIcons();
				setupShipSubNavigation();
			})
		);
	}

	@Override
	public void displayMessageSubNavigation() {
		clearIcons();
		setupMessagesSubNavigation();
	}

	private BooleanBinding inConvoy() {
		return new BooleanBinding() {
			{
				if (viewState.getCurrentCityProxy().isPresent()) {
					ObjectProperty<INavigableVessel> activeVessel = viewState.getCurrentCityProxy().get().activeShipProperty();
					super.bind(activeVessel);
					if (activeVessel.get() != null)
						activeVessel.addListener((observable, oldValue, newValue) -> {
							if (oldValue instanceof IShip) {
								super.unbind(((IShip) oldValue).parentShipProperty());
							}
							if (newValue instanceof IShip) {
								super.bind(((IShip) newValue).parentShipProperty());
							}
						});
				}
			}
			@Override
			protected boolean computeValue() {
				INavigableVessel activeShip = viewState.getCurrentCityProxy().get().getActiveShip();
				if (activeShip == null) {
					return false;
				}
				if (activeShip instanceof IShip) {
					return ((IShip) activeShip).parentShipProperty().get() != null;
				}
				return false;
			}
		};
	}

	private BooleanBinding isOrlegCapable() {
		return new BooleanBinding() {
			{
				if (viewState.getCurrentCityProxy().isPresent()) {
					ObjectProperty<INavigableVessel> activeVessel = viewState.getCurrentCityProxy().get().activeShipProperty();
					super.bind(activeVessel);
				}
			}
			@Override
			protected boolean computeValue() {
				INavigableVessel activeShip = viewState.getCurrentCityProxy().get().getActiveShip();
				if (activeShip == null) {
					return false;
				}
				if (activeShip instanceof IShip) {
					return shipService.isOrlegCapable((IShip) activeShip, viewState.getCurrentCityProxy().get().getCity());
				}
				return false;
			}
		};
	}
}
