package ch.sahits.game.openpatrician.display.javafx.control;

import ch.sahits.game.graphic.image.IDataImageLoader;
import ch.sahits.game.openpatrician.display.ClientViewState;
import ch.sahits.game.openpatrician.display.EViewState;
import ch.sahits.game.openpatrician.display.dialog.service.DialogFactory;
import ch.sahits.game.openpatrician.display.javafx.IDialogContoller;
import ch.sahits.game.openpatrician.event.NoticeBoardClose;
import ch.sahits.game.openpatrician.javafx.model.EDialogType;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import com.google.common.eventbus.AsyncEventBus;
import javafx.beans.property.DoubleProperty;
import javafx.beans.property.SimpleDoubleProperty;
import javafx.beans.property.SimpleStringProperty;
import javafx.beans.property.StringProperty;
import javafx.event.EventHandler;
import javafx.scene.Group;
import javafx.scene.image.ImageView;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.StackPane;
import javafx.scene.paint.Color;
import javafx.scene.shape.Rectangle;
import javafx.scene.text.Font;
import javafx.scene.text.Text;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationContext;

import javax.annotation.PostConstruct;

/**
 * Status control displaying the city name.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Nov 2, 2013
 *
 */
@Slf4j
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class ViewStatus extends Group {

	private StringProperty city;
	private DoubleProperty width;
	private Text cityName;
	@Autowired
	private ClientViewState clientViewState;
    @Autowired
    private DialogFactory dialogFactory;
	@Autowired
	private IDataImageLoader imageLoader;

	@Autowired
	@Qualifier("clientEventBus")
	protected AsyncEventBus clientEventBus;
	@Autowired
	private ApplicationContext context;


    private IDialogContoller dialogContoller; // this instance exists not at startup
	public ViewStatus() {
		setManaged(false);
        getStylesheets().add(getClass().getResource(getClass().getSimpleName() + ".css").toExternalForm());
		city = new SimpleStringProperty(this, "city", "");
		width = new SimpleDoubleProperty(this, "controlWidth", 0);

	}
    @PostConstruct
    private void init() {
		cityName = new Text();
		cityName.setLayoutY(10);
		cityName.setFill(Color.WHITE);
		cityName.getStyleClass().add("cityName");
		cityName.textProperty().bind(city);
		cityName.setOnMouseReleased(createHistoryDisplayHandler());  // handler to close any open dialog and open a history dialog

		Rectangle placeholder = new Rectangle(width.doubleValue(), 50);
		placeholder.setFill(Color.TRANSPARENT);
		placeholder.widthProperty().bind(width);

		StackPane centering = new StackPane();
		centering.prefWidthProperty().bind(width);
		centering.getChildren().add(cityName);

		ImageView shipListIcon = new ImageView(imageLoader.getImage("icons/64/icon_ship_selection"));
		shipListIcon.setScaleX(0.75);
		shipListIcon.setScaleY(0.75);
		double iconWidth = shipListIcon.getImage().getWidth();
		shipListIcon.setLayoutY(-iconWidth * 0.2); // Move up, so the scaled image is aligned
		shipListIcon.setLayoutX(this.width.doubleValue() - iconWidth - 5);
		shipListIcon.layoutXProperty().bind(this.width.subtract(iconWidth + 5));

		shipListIcon.setOnMouseReleased(evt -> {
			Object[] args = new Object[]{EDialogType.SHIP_LIST};
			final Runnable runnable1 = (Runnable) context.getBean("viewChangeActionRunnable", args);
			runnable1.run();
		});

		getChildren().addAll(placeholder, centering, shipListIcon);

		clientViewState.stateProperty().addListener((observable, oldValue, newValue) -> {
			if (clientViewState.getState() != EViewState.MAP) {
				cityName.textProperty().bind(city);
			} else {
				cityName.textProperty().unbind();
				cityName.setText("");
			}
		});
    }
    public void setDialogContoller(IDialogContoller dialogContoller) {
        this.dialogContoller = dialogContoller;
    }
	private EventHandler<? super MouseEvent> createHistoryDisplayHandler() {
		return mouseEvent -> {
			try {
				if (clientViewState.getCurrentCityProxy().isPresent()) {
					clientEventBus.post(new NoticeBoardClose());
					dialogContoller.replaceDialog(EDialogType.CITY_HISTORY);
				}
			} catch (RuntimeException e) {
				log.error("Failed to display history", e);
			}
		};
	}

    private String getHistoryResourceName(String name) {
        return "history."+name;
    }


	public void setCity(String city) {
		this.city.set(city);
	}
	public String getCity() {
		return this.city.get();
	}


	public DoubleProperty widthProperty() {
		return width;
	}
	
	public void setWidth(double width) {
		this.width.set(width);
	}
	public double getWidth() {
		return width.doubleValue();
	}
	public void setFont(Font f) {
		cityName.setFont(f);
	}
}
