package ch.sahits.game.openpatrician.display.model;

import ch.sahits.game.graphic.image.IDataImageLoader;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.paint.Color;
import javafx.scene.shape.ArcTo;
import javafx.scene.shape.CubicCurveTo;
import javafx.scene.shape.MoveTo;
import javafx.scene.shape.Path;
import javafx.scene.shape.Rectangle;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.PostConstruct;

/**
 * Configuration for the barrel depending on the barrel size.
 * @author Andi Hotz, (c) Sahits GmbH, 2018
 * Created on Jul 28, 2018
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class BarrelConfiguration {
    @Autowired
    private IDataImageLoader imageLoader;

    private int iconSize = 64;
    private int barrelTop = 4;
    private int barrelBottom = 63;
    private int totalHeigh;

    @PostConstruct
    private void init() {
        double scaleFactor = iconSize / 64.0;
        barrelTop *= scaleFactor;
        barrelBottom *= scaleFactor;
        totalHeigh = barrelBottom - barrelTop;
    }

    /**
     * Get the image view in the correct size.
     * @return ImageView for a barrel.
     */
    public ImageView getBarrelView() {
        Image img = imageLoader.getImage("icons/"+ iconSize+"/barrel");
        ImageView barrel = new ImageView(img);
        barrel.setId("barrelImage");
        return barrel;
    }

    /**
     * Create the path outlining the barrel.
     * @return path shape of the barrel image view.
     */
    public Path createBarrelShape() {
        Path path = new Path();
        double scaleFrom128 = 1;
        if (iconSize == 64) {
            scaleFrom128 = 0.5;
        }

        MoveTo start = new MoveTo(19 * scaleFrom128, 13 * scaleFrom128);
        path.getElements().add(start);

        ArcTo arcToTop = new ArcTo();
        arcToTop.setX(104 * scaleFrom128);
        arcToTop.setY(13 * scaleFrom128);
        arcToTop.setRadiusX(42.5 * scaleFrom128);
        arcToTop.setRadiusY(-5 * scaleFrom128);
        arcToTop.setSweepFlag(true);
        path.getElements().add(arcToTop);

        CubicCurveTo cubicToRight = new CubicCurveTo();
        cubicToRight.setX(61.5 * scaleFrom128);
        cubicToRight.setY(126 * scaleFrom128);
        double c1x = 115 * scaleFrom128;
        double c1y = 57 * scaleFrom128;
        double c2x = 109 * scaleFrom128;
        double c2y = 126 * scaleFrom128;
        cubicToRight.setControlX1(c1x);
        cubicToRight.setControlY1(c1y);
        cubicToRight.setControlX2(c2x);
        cubicToRight.setControlY2(c2y);
        path.getElements().add(cubicToRight);

        CubicCurveTo cubicToLeft = new CubicCurveTo();
        cubicToLeft.setX(start.getX());
        cubicToLeft.setY(start.getY());
        cubicToLeft.xProperty().bind(start.xProperty());
        cubicToLeft.yProperty().bind(start.yProperty());
        cubicToLeft.setControlX1(26 * scaleFrom128);
        cubicToLeft.setControlY1(c2y);
        cubicToLeft.setControlX2(8 * scaleFrom128);
        cubicToLeft.setControlY2(c1y);

        path.getElements().add(cubicToLeft);
        path.setFill(new Color(0, 0, 0, 0));

        return path;
    }

    /**
     * Calculate the rectangle covering the barrel shape that represents the amount of loadable space
     * that cannot be loaded with cargo. This is the space occupied by weapons and passengers.
     * @param ship for which to calculate.
     * @return rectangular shape representing the unloadable space of the barrel
     */
    public Rectangle getUnloadableShape(IShip ship) {
        double y = getUnloadableTop(ship);
        double height = getHeight(getUnloadablePercentage(ship));
        Rectangle rect = new Rectangle(0, y, iconSize, height);
        rect.setId("unloadableRect");
        rect.setFill(Color.BLUE);
        return rect;
    }

    /**
     * Calculate the rectangle covering the barrel shape that represents the amount of loaded wares.
     * @param ship for which to calculate.
     * @return rectangular shape representing the loaded ware.
     */
    public Rectangle getCargoShape(IShip ship) {
         double y = getCargoTop(ship);
         double height = getHeight(getCargoPercentage(ship));
         Rectangle rect = new Rectangle(0, y, iconSize, height);
         rect.setId("cargoRect");
         rect.setFill(Color.BLUE);
         return rect;
    }

    private double getUnloadablePercentage(IShip ship) {
        double loadedCargoAmount = ship.getLoadBinding().get();
        double totalLoaded = ship.getLoadableSpace() - ship.getCapacity();
        double loadedWeaponsPassengerEtc = totalLoaded - loadedCargoAmount;
        return loadedWeaponsPassengerEtc / ship.getLoadableSpace();

    }

    private double getUnloadableTop(IShip ship) {
        double height = getHeight(getUnloadablePercentage(ship));
        return barrelBottom -height;
    }

    private double getCargoPercentage(IShip ship) {
        double loadedCargoAmount = ship.getLoadBinding().get();
        return loadedCargoAmount / ship.getLoadableSpace();
    }

    private double getCargoTop(IShip ship) {
        double height = getHeight(getCargoPercentage(ship));
        return getUnloadableTop(ship) - height;
    }

    private double getHeight(double heighPercentage) {
        return heighPercentage * totalHeigh;
    }


}
