package ch.sahits.game.openpatrician.display.dialog.cityhall;

import ch.sahits.game.graphic.image.impl.SelectiveCachableXMLImageLoader;
import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.clientserverinterface.service.ModelStateAccessor;
import ch.sahits.game.openpatrician.clientserverinterface.service.OutriggerService;
import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.display.ClientViewState;
import ch.sahits.game.openpatrician.display.dialog.CloseButtonDialog;
import ch.sahits.game.openpatrician.display.dialog.service.DialogUtil;
import ch.sahits.game.openpatrician.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.openpatrician.event.EViewChangeEvent;
import ch.sahits.game.openpatrician.event.NoticeBoardUpdate;
import ch.sahits.game.openpatrician.javafx.control.DecoratedText;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.openpatrician.javafx.service.DecoratedTextFactory;
import ch.sahits.game.openpatrician.javafx.service.JavaFXUtils;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.cityhall.ICityHall;
import ch.sahits.game.openpatrician.model.city.cityhall.IOutriggerContract;
import ch.sahits.game.openpatrician.model.city.cityhall.impl.OutriggerContract;
import ch.sahits.game.openpatrician.model.service.ModelTranslations;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.ListType;
import ch.sahits.game.openpatrician.utilities.annotation.ObjectPropertyType;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.collections.FXCollections;
import javafx.collections.ListChangeListener;
import javafx.collections.ObservableList;
import javafx.collections.WeakListChangeListener;
import javafx.css.CssMetaData;
import javafx.css.SimpleStyleableObjectProperty;
import javafx.css.StyleConverter;
import javafx.css.Styleable;
import javafx.css.StyleableObjectProperty;
import javafx.css.StyleableProperty;
import javafx.event.EventHandler;
import javafx.scene.control.Control;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.GridPane;
import javafx.scene.text.Font;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Optional;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Mar 21, 2015
 */
@Slf4j
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class OutriggerNoticeDialog extends CloseButtonDialog {
    @ObjectPropertyType(IShip.class)
    private final ObjectProperty<IShip> currentShip;
    @Autowired
    private ClientViewState viewState;
    @Autowired
    private ModelStateAccessor cityHallAccessor;
    @Autowired
    private Locale locale;
    @Autowired
    private MessageSource messageSource;
    @Autowired
    private DecoratedTextFactory textFactory;
    @Autowired
    private OutriggerService outriggerService;
    @Autowired
    private ShipService shipService;
    @Autowired
    private ModelTranslations modelTranslator;
    @Autowired
    @Qualifier("xmlImageLoader")
    private SelectiveCachableXMLImageLoader imageLoader;
    @Autowired
    private JavaFXUtils fxUtils;
    @Autowired
    private DialogUtil dialogUtil;
    @ListType(IShip.class)
    private ObservableList<IShip> capableShips;

    private ListChangeListener<IShip> availableShipChangeListener;


    private final IntegerProperty currentIndex = new SimpleIntegerProperty(0);
    private BooleanBinding capableShipPresent;
    private final ICityPlayerProxyJFX city;

    public OutriggerNoticeDialog(ICityPlayerProxyJFX city) {
        super();
        this.city = city;
        getStylesheets().add(this.getClass().getResource("/styles/base.css").toExternalForm());
        getStyleClass().add("dialog");
        capableShipPresent = new BooleanBinding() {
            {
                super.bind(city.getPlayersShips());
            }
            @Override
            protected boolean computeValue() {
                return !getCapableShips().isEmpty();
            }
        };
        capableShips = FXCollections.observableArrayList();
        availableShipChangeListener = c -> capableShips.setAll(getCapableShips());
        city.getPlayersShips().addListener(new WeakListChangeListener<>(availableShipChangeListener));
        //noinspection Convert2Lambda
        capableShips.addListener(new ListChangeListener<>() {
            @Override
            public void onChanged(Change<? extends IShip> c) {
                if (capableShips.size() <= currentIndex.get()) {
                    currentIndex.setValue(capableShips.size() - 1);
                }
            }
        });
        this.currentShip = new SimpleObjectProperty<>(this, "currentShip", null);
        if (city.getActiveShip() instanceof IShip) {
            currentShip.setValue((IShip) city.getActiveShip());
        }
    }

    private List<IShip> getCapableShips() {
        int minWeaponStrength = outriggerService.getRequiredWeaponStrength(city.getCity());
        List<IShip> ships = new ArrayList<>();
        for (IShip ship : city.getPlayersShips()) {
            if (ship.getCaptian().isPresent() &&
                    ship.getNumberOfSailors() >= 20 &&
                    ship.getDamage() < 50 &&
                    shipService.calculateShipsWeaponsStrength(ship) >= minWeaponStrength) {
                ships.add(ship);
            }
        }
        return ships;

    }

    @PostConstruct
    private void initializeDialog() {
        capableShips.setAll(getCapableShips());
        ICityHall cityHall = cityHallAccessor.getCityHall(city.getCity());
        setTitle(messageSource.getMessage("ch.sahits.game.openpatrician.display.notice.NoticeBoardFactory.cityHallOutriggerNotice", new Object[]{}, locale.getCurrentLocal()));
        final int actionButtonX = (WIDTH - 124) / 2;
        if (cityHall.getOutriggerContract().isPresent()) {
            String template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.cityhall.OutriggerNoticeDialog.outriggerPresent", new Object[]{city.getCity().getName()}, locale.getCurrentLocal());
            DecoratedText text = textFactory.createDecoratedText(template, new HashMap<>());
            getContent().add(text);
            DecoratedText prevText = text;
            IOutriggerContract contract = cityHall.getOutriggerContract().get();
            IShip ship = contract.getOutrigger();
            IPlayer owner = (IPlayer) ship.getOwner();
            Object[] args = new Object[]{modelTranslator.getLocalDisplayNameWithArticle(ship.getShipType(), true), ship.getName(), modelTranslator.getLocalDisplayName(owner.getCareerLevel()), modelTranslator.getLocalDisplayName(owner.getRank()), owner.getName()+ " "+owner.getLastName()};
            template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.cityhall.OutriggerNoticeDialog.outriggerOwner", args, locale.getCurrentLocal());
            text = textFactory.createDecoratedText(template, new HashMap<>());
            text.layoutYProperty().bind(prevText.heightProperty().add(prevText.layoutYProperty()).add(24));
            getContent().add(text);
            GridPane grid = dialogUtil.createShipSelection3LinesForShips(FXCollections.singletonObservableList(contract.getOutrigger()), currentShip);
            grid.getStyleClass().add("tableFont");
            grid.setLayoutX(100);
            grid.setLayoutY(CLOSE_BTN_Y_POS - 400);
            getContent().add(grid);
            template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.cityhall.OutriggerNoticeDialog.weeklyRefund", new Object[]{ship.getName(), contract.getWeeklyRefund()}, locale.getCurrentLocal());
            text = textFactory.createDecoratedText(template, new HashMap<>());
            text.setLayoutY(CLOSE_BTN_Y_POS - 200);
            getContent().add(text);

            final OpenPatricianLargeWaxButton action = new OpenPatricianLargeWaxButton(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.cityhall.OutriggerNoticeDialog.discharge", new Object[]{}, locale.getCurrentLocal()));
            action.getStyleClass().add("actionButton");
            action.setOnAction(getDischargeAction(contract));
            action.setLayoutX(actionButtonX);
            action.setLayoutY(CLOSE_BTN_Y_POS - 24);
            getContent().addAll(action);

        } else {
            int minStrength = outriggerService.getRequiredWeaponStrength(city.getCity());
            int refund = outriggerService.getWeeklyRefund(city.getCity());
            String key = null;
            if (minStrength % 2 != 0) {
                if (minStrength == 1) {
                    key = "ch.sahits.game.openpatrician.display.dialog.cityhall.OutriggerNoticeDialog.outriggerRequestOne";
                } else {
                    key = "ch.sahits.game.openpatrician.display.dialog.cityhall.OutriggerNoticeDialog.outriggerRequestUneven";
                }
            } else {
                key = "ch.sahits.game.openpatrician.display.dialog.cityhall.OutriggerNoticeDialog.outriggerRequestEven";
            }
            String template = messageSource.getMessage(key, new Object[]{city.getCity().getName(), minStrength/2, refund}, locale.getCurrentLocal());
            DecoratedText text = textFactory.createDecoratedText(template, new HashMap<>());
            getContent().add(text);
            if (!capableShipPresent.get()) {
                template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.cityhall.OutriggerNoticeDialog.noShip", new Object[]{}, locale.getCurrentLocal());
                text = textFactory.createDecoratedText(template, new HashMap<>());
                text.setLayoutY(CLOSE_BTN_Y_POS - 400);
                getContent().add(text);
            } else {
                GridPane grid = dialogUtil.createShipSelection3LinesForShips(capableShips, currentShip);
                grid.getStyleClass().add("tableFont");
                grid.setLayoutX(100);
                grid.setLayoutY(CLOSE_BTN_Y_POS - 400);
                getContent().add(grid);
            }

            final OpenPatricianLargeWaxButton action = new OpenPatricianLargeWaxButton(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.TavernBaseSideRoomPersonDialog.accept", new Object[]{}, locale.getCurrentLocal()));
            action.getStyleClass().add("actionButton");
            action.setOnAction(getHireAction(minStrength, refund));
            action.setLayoutX(actionButtonX);
            action.setLayoutY(CLOSE_BTN_Y_POS - 24);
            action.setDisable(!capableShipPresent.get());
            capableShipPresent.addListener((observable, oldFlag, newFlag) -> action.setDisable(!newFlag));

            getContent().addAll(action);
        }
    }

    private EventHandler<MouseEvent> getDischargeAction(final IOutriggerContract contract) {
        return (mouseEvent) -> {
            try {
                IShip ship = contract.getOutrigger();
                ship.setAvailable(true);
                city.getPlayersShips().add(ship);
                cityHallAccessor.getCityHall(city.getCity()).setOutriggerContract(Optional.empty());
                executeOnCloseButtonClicked();
            } catch (RuntimeException e) {
                log.error("Failed to switch outrigger", e);
            }
        };
    }

    private EventHandler<MouseEvent> getHireAction(int minStrength, int refund) {
        return (mouseEvent) -> {
            try {
                IShip ship = currentShip.get();
                OutriggerContract contract = new OutriggerContract(ship, minStrength, refund);
                ship.setAvailable(false);
                city.leave(ship);
                Optional<IOutriggerContract> optContract = Optional.of(contract);
                cityHallAccessor.getCityHall(city.getCity()).setOutriggerContract(optContract);
                executeOnCloseButtonClicked();
            } catch (RuntimeException e) {
                log.error("Failed to switch outrigger", e);
            }
        };
    }

    /**
     * Move the selection to the next index
     * @return .
     */
    private EventHandler<MouseEvent> createNextAction() {
        return (mouseEvent) -> {
            try {
                if (currentIndex.get() == capableShips.size() - 1) {
                    currentIndex.set(0);
                } else {
                    currentIndex.set(currentIndex.get() + 1);
                }
            } catch (RuntimeException e) {
                log.error("Failed to switch outrigger", e);
            }
        };
    }

    /**
     * move the selection to the previous index.
     * @return  .
     */
    private EventHandler<MouseEvent> createPreviousAction() {
        return (mouseEvent)  -> {
            try {
                if (currentIndex.get() == 0) {
                    currentIndex.set(capableShips.size() - 1);
                } else {
                    currentIndex.set(currentIndex.get() - 1);
                }
            } catch (RuntimeException e) {
                log.error("Failed to switch to previous ship", e);
            }
        };
    }

    /**
     * Update the notice board and close the dialog.
     */
    @Override
    public void executeOnCloseButtonClicked() {
        ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.NOTICE_CITY_HALL_BOARD);
        clientEventBus.post(new NoticeBoardUpdate(proxy));
        super.executeOnCloseButtonClicked();
    }

    private StyleableObjectProperty<Font> font;

    public Font getFont() {
        return font == null ? Font.getDefault() : font.get();
    }
    public void setFont(Font font) {
        this.font.set(font);
    }
    public StyleableObjectProperty<Font> fontProperty() {
        if (font == null) {
            font = new SimpleStyleableObjectProperty<>(StyleableProperties.FONT, OutriggerNoticeDialog.this, "font", Font.getDefault());
        }
        return font;
    }

    private static class StyleableProperties {
        private static final CssMetaData< OutriggerNoticeDialog, Font> FONT =
                new CssMetaData<>("-fx-font",
                        StyleConverter.getFontConverter(), Font.getDefault()) {
                    @Override
                    public boolean isSettable(OutriggerNoticeDialog control) {
                        return control.font == null || !control.font.isBound();
                    }

                    @Override
                    public StyleableProperty<Font> getStyleableProperty(OutriggerNoticeDialog control) {
                        return control.fontProperty();
                    }
                };
        private static final List<CssMetaData<? extends Styleable, ?>> STYLEABLES;
        static {
            final List<CssMetaData<? extends Styleable, ?>> styleables =
                    new ArrayList<>(Control.getClassCssMetaData());
            Collections.addAll(styleables,
                    FONT
            );
            STYLEABLES = Collections.unmodifiableList(styleables);
        }
    }
    public List<CssMetaData<? extends Styleable, ?>> getControlCssMetaData() {
        return getClassCssMetaData();
    }
    public static List<CssMetaData<? extends Styleable, ?>> getClassCssMetaData() {
        return StyleableProperties.STYLEABLES;
    }
}
