package ch.sahits.game.openpatrician.display.dialog.ship;

import ch.sahits.game.graphic.image.IDataImageLoader;
import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.display.dialog.CloseButtonDialog;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.people.ICaptain;
import ch.sahits.game.openpatrician.model.people.impl.CaptainsState;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.weapon.EWeapon;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import ch.sahits.game.openpatrician.utilities.spring.DependentPropertyInitializer;
import ch.sahits.game.openpatrician.utilities.spring.DependentValue;
import javafx.beans.binding.StringBinding;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.geometry.HPos;
import javafx.geometry.Pos;
import javafx.scene.control.Label;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.layout.ColumnConstraints;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.HBox;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;

/**
 * Dialog displaying the ship info of the currently selected ship.
 * @author Andi Hotz, (c) Sahits GmbH, 2018
 * Created on Jun 05, 2018
 */
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class ShipInfoDialog extends CloseButtonDialog {
    /** Reference to the proxy view model */
    private final ICityPlayerProxyJFX proxy;

    @DependentValue("sailor.cost.per.day")
    private double dailySailorCost = 0.5;


    @Autowired
    private MessageSource messageSource;
    @Autowired
    private Locale locale;
    @Autowired
    private IDataImageLoader imageLoader;
    @Autowired
    private CaptainsState captainsState;
    @Autowired
    private Date date;
    @Autowired
    private DependentPropertyInitializer propertyInitializer;

    public ShipInfoDialog(ICityPlayerProxyJFX proxy) {
        this.proxy = proxy;
    }

    @PostConstruct
    private void initializeDialog() {
        IShip activeShip = (IShip) proxy.getActiveShip();
        ObjectProperty<IShip> shipProperty = new SimpleObjectProperty<>((IShip) activeShip);
        setTitle(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ship.ShipInfoDialog.title",
                new Object[]{activeShip.getName()}, locale.getCurrentLocal()));

        GridPane sailorsPane = new GridPane();
        sailorsPane.setId("shipInfo");
        sailorsPane.setLayoutY(100);
        sailorsPane.setLayoutX(INSET);
        ColumnConstraints col1Constraints = new ColumnConstraints();
        col1Constraints.setPercentWidth(50);
        ColumnConstraints col2Constraints = new ColumnConstraints();
        col2Constraints.setPercentWidth(50);
        sailorsPane.getColumnConstraints().addAll(col1Constraints, col2Constraints);
        HBox sailorsBox = new HBox(10);
        sailorsBox.setId("sailorsBox");
        sailorsBox.setAlignment(Pos.CENTER);
        Image img = imageLoader.getImage("icons/128/sailor-icon");
        ImageView headView = new ImageView(img);
        headView.setScaleX(0.5);
        headView.setScaleY(0.5);
        Label sailorNumbers = new Label();
        StringBinding sailorNumberBinding = new StringBinding() {
            {
                super.bind(activeShip.numberOfSailorsProperty());
            }
            @Override
            protected String computeValue() {
                return activeShip.getNumberOfSailors() + " ("+activeShip.getMinNumberOfSailors()+" - "+ activeShip.getMaxNumberOfSailors()+")";
            }
        };
        sailorNumbers.setText(sailorNumberBinding.get());
        sailorNumbers.getStyleClass().addAll("dialogText");
        sailorNumbers.textProperty().bind(sailorNumberBinding);
        sailorsBox.getChildren().addAll(headView, sailorNumbers);

        sailorsPane.add(sailorsBox, 0, 0);

        HBox salaryBox = new HBox(10);
        salaryBox.setId("salaryBox");
        salaryBox.setAlignment(Pos.CENTER);
        img = imageLoader.getImage("icons/128/tradeskill_icon");
        ImageView coinsView = new ImageView(img);
        coinsView.setScaleX(0.5);
        coinsView.setScaleY(0.5);
        Label sailorSalaryLbl = new Label();
        sailorSalaryLbl.setId("sailorSalaryLbl");
        StringBinding sailorSalaryBinding = new StringBinding() {
            {
                super.bind(activeShip.numberOfSailorsProperty());
            }
            @Override
            protected String computeValue() {
                INavigableVessel ship = proxy.getActiveShip();
                double sailorSalary = ship.getNumberOfSailors() * dailySailorCost;
                if (ship.getCaptian().isPresent()) {
                    ICaptain captain = ship.getCaptian().get();
                    sailorSalary += captain.getSalary();
                }
                sailorSalary *= 7;
                return messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ship.ShipInfoDialog.sailorSalaery",
                        new Object[]{sailorSalary}, locale.getCurrentLocal());
            }
        };
        sailorSalaryLbl.setText(sailorSalaryBinding.get());
        sailorSalaryLbl.getStyleClass().addAll("dialogText");
        sailorSalaryLbl.textProperty().bind(sailorSalaryBinding);
        salaryBox.getChildren().addAll(coinsView, sailorSalaryLbl);

        sailorsPane.add(salaryBox, 1, 0);

        HBox dismissSailorBox = new HBox(10);
        dismissSailorBox.setId("dismissSailorBox");
        dismissSailorBox.setAlignment(Pos.CENTER);
        img = imageLoader.getImage("icons/plusIcon");
        ImageView plusIcon = new ImageView(img);
        plusIcon.setId("plusIcon");
        IntegerProperty dismissNbSailors = new SimpleIntegerProperty(0);
        plusIcon.setOnMouseReleased((e) -> {
            IShip ship = shipProperty.get();
            int value = Math.min(ship.getNumberOfSailors(), dismissNbSailors.get() + 1);
            dismissNbSailors.setValue(value);
        });
        Label dismissNbSailorsLbl = new Label();
        dismissNbSailorsLbl.setId("dismissNbSailorsLbl");
        dismissNbSailorsLbl.getStyleClass().add("dialogText");
        dismissNbSailorsLbl.setText("0");
        dismissNbSailorsLbl.textProperty().bind(dismissNbSailors.asString());
        img = imageLoader.getImage("icons/minusIcon");
        ImageView minusIcon = new ImageView(img);
        minusIcon.setId("minusIcon");
        minusIcon.setOnMouseReleased((e) -> {
            int value = Math.max(0, dismissNbSailors.get() - 1);
            dismissNbSailors.setValue(value);});
        dismissSailorBox.getChildren().addAll(plusIcon, dismissNbSailorsLbl, minusIcon);

        sailorsPane.add(dismissSailorBox, 0, 1);

        OpenPatricianLargeWaxButton dismissBtn = new OpenPatricianLargeWaxButton(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ship.ShipInfoDialog.dismissSailors", new Object[]{}, locale.getCurrentLocal()));
        dismissBtn.setId("dismissBtn");
        dismissBtn.getStyleClass().add("actionButton");
        dismissBtn.setOnAction(event -> {
            IShip ship = shipProperty.get();
            ship.setNumberOfSailors(ship.getNumberOfSailors() - dismissNbSailors.get());
            dismissNbSailors.setValue(0);
        });
        sailorsPane.add(dismissBtn, 1, 1);



        HBox handWeaponBox = new HBox(10);
        handWeaponBox.setId("handWeaponBox");
        handWeaponBox.setAlignment(Pos.CENTER);
        img = imageLoader.getImage("icons/128/cutlass_icon");
        ImageView cutlassView = new ImageView(img);
        cutlassView.setScaleX(0.5);
        cutlassView.setScaleY(0.5);
        Label weaponsLbl = new Label(String.valueOf(proxy.getActiveShip().getWeaponAmount(EWeapon.HAND_WEAPON)));
        weaponsLbl.getStyleClass().addAll("dialogText");
        handWeaponBox.getChildren().addAll(cutlassView, weaponsLbl);

        sailorsPane.add(handWeaponBox, 0, 2, 2, 1);
        GridPane.setHalignment(handWeaponBox, HPos.CENTER);

        if (proxy.getActiveShip().getCaptian().isPresent()){
            ICaptain captain = proxy.getActiveShip().getCaptian().get();
            String s = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ship.ShipInfoDialog.captainName", new Object[]{captain.getName(), captain.getAge()}, locale.getCurrentLocal());
            Label captainName = new Label(s);
            captainName.getStyleClass().addAll("dialogText");
            captainName.setId("captainName");
            sailorsPane.add(captainName, 0, 3, 2, 1);
            GridPane.setHalignment(captainName, HPos.CENTER);

            HBox captainCapabilityBox = new HBox(10);
            captainCapabilityBox.setId("captainCapabilityBox");
            captainCapabilityBox.setAlignment(Pos.CENTER);
            img = imageLoader.getImage("icons/64/trade-icon");
            ImageView tradeView = new ImageView(img);
            tradeView.setScaleX(0.5);
            tradeView.setScaleY(0.5);
            Label tradeLbl = new Label(String.valueOf(captain.getTradingSkillLevel()));
            tradeLbl.getStyleClass().addAll("dialogText");
            img = imageLoader.getImage("icons/64/compass-icon");
            ImageView navigationView = new ImageView(img);
            navigationView.setScaleX(0.5);
            navigationView.setScaleY(0.5);
            Label navigationLbl = new Label(String.valueOf(captain.getNavigationSkillLevel()));
            navigationLbl.getStyleClass().addAll("dialogText");
            img = imageLoader.getImage("icons/64/fightskill_icon");
            ImageView fightView = new ImageView(img);
            fightView.setScaleX(0.5);
            fightView.setScaleY(0.5);
            Label fightLbl = new Label(String.valueOf(captain.getFightSkillLevel()));
            fightLbl.getStyleClass().addAll("dialogText");
            captainCapabilityBox.getChildren().addAll(tradeView, tradeLbl, navigationView, navigationLbl, fightView, fightLbl);
            sailorsPane.add(captainCapabilityBox, 0, 4, 2, 1);
            GridPane.setHalignment(captainCapabilityBox, HPos.CENTER);
            OpenPatricianLargeWaxButton dismissCaptain = new OpenPatricianLargeWaxButton(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ship.ShipInfoDialog.dismissCaptain", new Object[]{}, locale.getCurrentLocal()));
            dismissCaptain.setId("dismissCaptain");
            dismissCaptain.getStyleClass().add("actionButton");
            dismissCaptain.setOnAction(event -> {
                        IShip ship = shipProperty.get();
                        ship.setCaptain(null);
                        captainsState.addCaptain(proxy.getCity(), date.getCurrentDate(), captain);
                        captainCapabilityBox.setVisible(false);
                        captainName.setVisible(false);
                        dismissCaptain.setVisible(false);
                        sailorSalaryBinding.invalidate();
                    });
            sailorsPane.add(dismissCaptain, 0, 5);
            GridPane.setHalignment(dismissCaptain, HPos.CENTER);
        }

        OpenPatricianLargeWaxButton togglePirate = new OpenPatricianLargeWaxButton(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ship.ShipInfoDialog.switchPirate", new Object[]{}, locale.getCurrentLocal()));
        togglePirate.setId("togglePirate");
        togglePirate.getStyleClass().add("actionButton");
        togglePirate.setOnAction(event -> {
            IShip ship = shipProperty.get();
            ship.togglePirateFlag();
            executeOnCloseButtonClicked();
        });
        sailorsPane.add(togglePirate, 1, 5);
        GridPane.setHalignment(togglePirate, HPos.CENTER);

        getContent().addAll(sailorsPane);
    }

}
