package ch.sahits.game.openpatrician.display.dialog.shipyard;

import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.display.dialog.service.DialogUtil;
import ch.sahits.game.openpatrician.event.data.ShipyardOrderRepair;
import ch.sahits.game.openpatrician.javafx.control.BarrelAmount;
import ch.sahits.game.openpatrician.javafx.control.DecoratedText;
import ch.sahits.game.openpatrician.javafx.model.ControlTableCell;
import ch.sahits.game.openpatrician.javafx.model.StaticTextTableCell;
import ch.sahits.game.openpatrician.javafx.model.Table;
import ch.sahits.game.openpatrician.javafx.model.TableHeader;
import ch.sahits.game.openpatrician.javafx.model.TableRow;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.ObjectPropertyType;
import com.google.common.eventbus.AsyncEventBus;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.property.BooleanProperty;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleBooleanProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.event.EventHandler;
import javafx.scene.Group;
import javafx.scene.Node;
import javafx.scene.input.MouseEvent;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.util.HashMap;
import java.util.Iterator;

/**
 * Dialog for repairing a ship
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 *         Created on Dec 16, 2013
 */
@Slf4j
@Component("shipRepairDialog")
@Scope(value = "prototype")
@ClassCategory({EClassCategory.DIALOG, EClassCategory.SINGLETON_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class ShipRepairDialogV2 extends BaseShipyardDialog {

    @ObjectPropertyType(IShip.class)
    private ObjectProperty<IShip> currentShip;
    @Autowired
    private DialogUtil dialogUtil;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    private ShipService shipService;
    @Autowired
    private ApplicationContext context;

    public ShipRepairDialogV2(final ICityPlayerProxyJFX city) {
        super(city);
        this.currentShip = new SimpleObjectProperty<>(this, "currentShip", null);
        if (city.getActiveShip() instanceof IShip) {
            IShip ship = (IShip) city.getActiveShip();
            currentShip.setValue(ship);
        }
        if (currentShip.get() != null) {
            for (int i = 0; i < shipTypes.length; i++) {
                if (shipTypes[i] == currentShip.get().getShipType()) {
                    currentShipTypeIndexProperty().set(i);
                    break;
                }
            }
        } else {
            currentShipTypeIndexProperty().set(0);
        }
    }
    @PostConstruct
    private void init() {
        setTitle(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ShipRepairDialogV2.title", new Object[]{}, locale.getCurrentLocal()));
        this.enablePreviousNext = dialogUtil.enableShipCatalogueForShips(city.getPlayersShips());
    }

    @Override
    protected void initializeRequirements() {
        // only requirement is money
    }

    @Override
    protected EventHandler<MouseEvent> getAction() {
        return mouseEvent -> {
            try {
                IShip ship = currentShip.get();
                ship.setAvailable(false);
                city.leave(ship);
                ShipyardOrderRepair event = new ShipyardOrderRepair(getCity().getCityState().getShipyardState(), currentShip.get(), city);
                clientServerEventBus.post(event);
                getPlayer().getCompany().updateCash(-calculateRepairCosts());
                executeOnCloseButtonClicked();
            } catch (RuntimeException e) {
                log.error("Failed to order repair of ship", e);
            }
        };
    }

    @Override
    protected String getActionText() {
        return messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ShipRepairDialogV2.repair", new Object[]{}, locale.getCurrentLocal());
    }

    @Override
    protected boolean hasShips() {
        if (city.getPlayersShips().size() > 0) {
            return city.getPlayersShips().stream().anyMatch(IShip::isAvailable);
        } else {
            return false;
        }
    }

    @Override
    protected Group createFooterText() {
        int costs =  calculateRepairCosts();
        Group g = new Group();
        String t = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ShipyardConstructionDialogV2.totalSum", new Object[]{}, locale.getCurrentLocal());
        HashMap<String, Object> parameters = new HashMap<>();
        parameters.put("price", costs);
        DecoratedText dt = decoratedTextFactory.createDecoratedText(t, parameters);
        dt.setId("totalPrice");
        g.getChildren().add(dt);
        g.setId("footerText");
        return g;
    }

    private int calculateRepairCosts() {
        return shipService.calculateRepairCosts(getCity(), currentShip.get());
    }

    @Override
    protected Table createMainTable() {
        Table model = new Table();
        // Add dummy header which is not rendered
        TableHeader header = new TableHeader(1);

        model.setHeader(header);
        model.setColumnWidth(200);
        TableRow row = new TableRow();

        row.add(new StaticTextTableCell(currentShip.get().getName()));
        model.add(row);
        return model;
    }

    @Override
    protected Table createTopTable() {
        Table model = new Table();
        // Add dummy header which is not rendered
        TableHeader header = new TableHeader(4);
        model.setHeader(header);
        model.setColumnWidth(70, 70, 70, 70);
        TableRow row = new TableRow();
        BarrelAmount capacityAmount = context.getBean(BarrelAmount.class);
        capacityAmount.setAmount(currentShip.get().getLoadableSpace());
        row.add(new ControlTableCell(capacityAmount));

        row.add(new StaticTextTableCell(String.valueOf(100 - currentShip.get().getDamage())));
        row.add(new StaticTextTableCell(String.valueOf(currentShip.get().getNumberOfSailors())));
        row.add(new StaticTextTableCell("1"));
        model.add(row);
        return model;
    }

    @Override
    protected BooleanBinding actionEnabledBinding() {
        return new BooleanBinding() {
            {
                super.bind(getPlayer().getCompany().cashProperty());
            }
            @Override
            protected boolean computeValue() {
                return calculateRepairCosts() < getPlayer().getCompany().getCash();
            }
        };
    }

    @Override
    protected EventHandler<MouseEvent> createNextAction() {
        return event -> {
            dialogUtil.createNextActionForShips(city.getPlayersShips(), currentShip).handle(event);
            selectionChanged();
        };
    }

    @Override
    protected EventHandler<MouseEvent> createPreviousAction() {
        return event -> {
            dialogUtil.createPreviousActionForShips(city.getPlayersShips(), currentShip).handle(event);
            selectionChanged();
        };
    }

    @Override
    protected BooleanProperty disableNavigation() {
        return new SimpleBooleanProperty(city.getPlayersShips().size() <= 1);
    }
    @Override
    protected String getTitleText() {
        return currentShip.get().getName();
    }

    @Override
    protected void selectionChanged() {
        super.selectionChanged();
        Group footerText = createFooterText();
        footerText.setLayoutX(50);
        footerText.setLayoutY(250 + 36 + 7 * 24);
        for (Iterator<Node> iterator = getContent().iterator(); iterator.hasNext(); ) {
            Node child =  iterator.next();
            if ("footerText".equals(child.getId())) {
                iterator.remove();
                break;
            }
        }
        getContent().add(footerText);
    }
}
