package ch.sahits.game.openpatrician.display.dialog.shipyard;

import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.clientserverinterface.model.WeaponSlotCount;
import ch.sahits.game.openpatrician.clientserverinterface.model.factory.ShipFactory;
import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.display.dialog.service.DialogUtil;
import ch.sahits.game.openpatrician.display.model.RequiredWareCityStorage;
import ch.sahits.game.openpatrician.event.data.ShipyardOrderRefit;
import ch.sahits.game.openpatrician.javafx.control.BaleAmountAlwaysVisible;
import ch.sahits.game.openpatrician.javafx.control.BarrelAmount;
import ch.sahits.game.openpatrician.javafx.control.BarrelAmountAlwaysVisible;
import ch.sahits.game.openpatrician.javafx.control.CoinPriceAlwaysVisible;
import ch.sahits.game.openpatrician.javafx.control.DecoratedText;
import ch.sahits.game.openpatrician.javafx.model.ControlTableCell;
import ch.sahits.game.openpatrician.javafx.model.ECellConstraint;
import ch.sahits.game.openpatrician.javafx.model.StaticTextTableCell;
import ch.sahits.game.openpatrician.javafx.model.Table;
import ch.sahits.game.openpatrician.javafx.model.TableHeader;
import ch.sahits.game.openpatrician.javafx.model.TableRow;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.service.ModelTranslations;
import ch.sahits.game.openpatrician.model.ship.EShipUpgrade;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.ship.IWeaponSlot;
import ch.sahits.game.openpatrician.model.ship.impl.ShipWeaponsLocationFactory;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.MapType;
import ch.sahits.game.openpatrician.utilities.annotation.ObjectPropertyType;
import com.google.common.base.Preconditions;
import com.google.common.eventbus.AsyncEventBus;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.property.BooleanProperty;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.ReadOnlyIntegerProperty;
import javafx.beans.property.SimpleBooleanProperty;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.event.EventHandler;
import javafx.geometry.HPos;
import javafx.scene.Group;
import javafx.scene.Node;
import javafx.scene.input.MouseEvent;
import javafx.scene.text.Text;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 *         Created on Dec 13, 2013
 */
@Component("shipyardUpgradeDialog")
@Scope(value = "prototype")
@ClassCategory({EClassCategory.DIALOG, EClassCategory.SINGLETON_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class ShipyardUpgradeDialogV2 extends BaseShipyardDialog {

    @ObjectPropertyType(IShip.class)
    private ObjectProperty<IShip> currentShip;
    @Autowired
    private ShipFactory shipFactory;
    @Autowired
    private ShipService shipService;
    @Autowired
    private ShipWeaponsLocationFactory weaponsLocationFactory;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    private ModelTranslations translator;
    @Autowired
    private DialogUtil dialogUtil;
    @Autowired
    private ApplicationContext context;

    private static final EWare[] MATERIALS = new EWare[]{EWare.WOOD,
                                                         EWare.IRON,
                                                         EWare.PITCH};

    @MapType(key = IWare.class, value = RequiredWareCityStorage.class)
    private HashMap<IWare, RequiredWareCityStorage> requirementsMap = new HashMap<>();

    public ShipyardUpgradeDialogV2(final ICityPlayerProxyJFX city) {
        super(city);
        this.currentShip = new SimpleObjectProperty<>(this, "currentShip", null);
        if (city.getActiveShip() instanceof IShip) {
            IShip ship = (IShip) city.getActiveShip();
            currentShip.setValue(ship);
        }

        this.enablePreviousNext = new BooleanBinding() {
            {
                super.bind(city.getPlayersShips());
            }
            @Override
            protected boolean computeValue() {
                return city.getPlayersShips().size() > 1;
            }
        };
        if (currentShip.get() != null) {
            for (int i = 0; i < shipTypes.length; i++) {
                if (shipTypes[i] == currentShip.get().getShipType()) {
                    currentShipTypeIndexProperty().set(i);
                    break;
                }
            }
        } else {
            currentShipTypeIndexProperty().set(0);
        }
        mainTableYPosition = 250 + 36 + 48;
    }

    /**
     * Initialize the requirementsMap.
     */
    @Override
    protected void initializeRequirements() {
        ICity city = getCity();
        Optional<ITradingOffice> optOffice = getPlayer().findTradingOffice(city);
        for (EWare material : MATERIALS) {
            int requiredAmount = shipFactory.getUpgradeAmount(currentShip.get().getShipType(), material);
            ReadOnlyIntegerProperty amountInCity = city.getWare(material).amountProperty();
            ReadOnlyIntegerProperty amountInStorage;
            if (optOffice.isPresent()) {
                ITradingOffice office = optOffice.get();
                amountInStorage = office.getWare(material).amountProperty();
            } else {
                amountInStorage = new SimpleIntegerProperty(0);
            }
            if (requirementsMap.containsKey(material)) {
                // update
                RequiredWareCityStorage requiredWare = requirementsMap.get(material);
                requiredWare.setRequired(requiredAmount);
                requiredWare.inStorageProperty().unbind();
                requiredWare.inStorageProperty().bind(amountInStorage);
                requiredWare.inCityProperty().unbind();
                requiredWare.inCityProperty().bind(amountInCity);
            } else {
                // initialize
                IntegerProperty inCity = new SimpleIntegerProperty();
                inCity.bind(amountInCity);
                IntegerProperty inStorage = new SimpleIntegerProperty();
                inStorage.bind(amountInStorage);
                RequiredWareCityStorage requiredWare = new RequiredWareCityStorage(requiredAmount, inStorage, inCity,
                        material, computablePrice);
                requirementsMap.put(material, requiredWare);
            }
        }
    }

    @PostConstruct
    private void intializeAdditionalText() {
        setTitle(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ShipyardUpgradeDialogV2.title", new Object[]{}, locale.getCurrentLocal()));
        final Text name = new Text();
        name.getStyleClass().add("dialogText");
        name.setLayoutX(50);
        name.setLayoutY(250 + 36 + 24);
        if (currentShip.get() != null) {
            name.setText(currentShip.get().getName());
            currentShip.addListener((observableValue, oldValue, newValue) -> name.setText(newValue.getName()));
            final Text level = new Text();
            level.getStyleClass().add("dialogText");
            level.setLayoutX(50);
            level.setLayoutY(250 + 36 + 48);
            level.setText(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ShipyardUpgradeDialogV2.upgradeLevel", new Object[]{currentShip.get().getShipUpgradeLevel().nextLevel().name()}, locale.getCurrentLocal()));
            currentShip.addListener((observableValue, oldValue, newValue) ->
                    level.setText(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ShipyardUpgradeDialogV2.upgradeLevel", new Object[]{newValue.getShipUpgradeLevel().nextLevel().name()}, locale.getCurrentLocal())));
            getContent().addAll(name, level);
        }
    }
    @Override
    protected boolean hasShips() {
        if (city.getPlayersShips().size() > 0) {
            return city.getPlayersShips().stream().anyMatch(IShip::isAvailable);
        } else {
            return false;
        }
    }


    @Override
    protected EventHandler<MouseEvent> getAction() {
        return mouseEvent -> {
            Optional<ITradingOffice> office = getPlayer().findTradingOffice(getCity());
            Preconditions.checkArgument(office.isPresent(), "Trading office must be present");
            for (EWare ware : MATERIALS) {
                int buyAmount = requirementsMap.get(ware).getBuyAmount().intValue();
                getCity().move(ware, -buyAmount, getPlayer());
                int needed = shipFactory.getUpgradeAmount(currentShip.get().getShipType(), ware);
                int fromWarhouse = needed - buyAmount;
                office.get().move(ware, -fromWarhouse, 0);
            }
            ShipyardOrderRefit event = new ShipyardOrderRefit(getCity().getCityState().getShipyardState(),currentShip.get(), currentShip.get().getShipUpgradeLevel().nextLevel(), city);
            clientServerEventBus.post(event);

            IShip ship = currentShip.get();
            ship.setAvailable(false);
            city.leave(ship);
            getPlayer().getCompany().updateCash(-calculateTotalUpgradeCosts());
            executeOnCloseButtonClicked();
        };
    }

    @Override
    protected String getActionText() {
        return messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ShipyardUpgradeDialogV2.refit", new Object[]{}, locale.getCurrentLocal());
    }

    @Override
    protected Group createFooterText() {
        IntegerBinding totalPriceBinding = new IntegerBinding() {
            {
                // Bind to amount in office and in city
                for (EWare ware : MATERIALS) {
                    RequiredWareCityStorage requiredWare = requirementsMap.get(ware);
                    super.bind(requiredWare.buyPriceProperty());
                }
            }
            @Override
            protected int computeValue() {
                return calculateTotalUpgradeCosts();
            }
        };
        Group g = new Group();
        String t = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ShipyardConstructionDialogV2.totalSum", new Object[]{}, locale.getCurrentLocal());
        HashMap<String, Object> parameters = new HashMap<>();
        parameters.put("price", totalPriceBinding.get());
        DecoratedText dt = decoratedTextFactory.createDecoratedText(t, parameters);
        dt.setId("totalSum");
        g.getChildren().add(dt);
        g.setId("footerText");
        return g;
    }

    private int calculateTotalUpgradeCosts() {
        int total = getCity().getCityState().getShipyardState().calculateRefitCosts(currentShip.get().getShipType(), 1);
        for (EWare ware : MATERIALS) {
            RequiredWareCityStorage requiredWare = requirementsMap.get(ware);
            total += requiredWare.getBuyPrice().intValue();
        }
        return total;
    }

    @Override
    protected Table createMainTable() {
        Table model = new Table();
        TableHeader header = new TableHeader(5);
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ShipyardConstructionDialogV2.materials", new Object[]{}, locale.getCurrentLocal())), ECellConstraint.COLSPAN2);
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ShipyardConstructionDialogV2.stored", new Object[]{}, locale.getCurrentLocal())));
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ShipyardConstructionDialogV2.buy", new Object[]{}, locale.getCurrentLocal())));
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ShipyardConstructionDialogV2.costs", new Object[]{}, locale.getCurrentLocal())));
        header.setAligenment(0, HPos.CENTER);
        header.setAligenment(2, HPos.CENTER);
        header.setAligenment(3, HPos.CENTER);
        header.setAligenment(4, HPos.CENTER);
        model.setHeader(header);
        model.setAligenment(0, HPos.RIGHT);
        model.setAligenment(1, HPos.RIGHT);
        model.setAligenment(2, HPos.RIGHT);
        model.setAligenment(3, HPos.RIGHT);
        model.setAligenment(4, HPos.RIGHT);

        model.setColumnWidth(70, 70, 70, 70, 100);
        for (final EWare ware : MATERIALS) {
            RequiredWareCityStorage requiredWare = requirementsMap.get(ware);

            TableRow row = new TableRow();
            row.add(new StaticTextTableCell(translator.getLocalDisplayName(ware)));

            final IntegerProperty needed = requiredWare.requiredProperty();
            if (ware.isBarrelSizedWare()) {
                BarrelAmountAlwaysVisible barrelAmount = context.getBean(BarrelAmountAlwaysVisible.class);
                barrelAmount.amountProperty().bind(needed.asString());
                row.add(new ControlTableCell(barrelAmount));
            } else {
                BaleAmountAlwaysVisible baleAmount = context.getBean(BaleAmountAlwaysVisible.class);
                baleAmount.amountProperty().bind(needed.asString());
                row.add(new ControlTableCell(baleAmount));
            }

            final ReadOnlyIntegerProperty stored = requiredWare.inStorageProperty();
            if (ware.isBarrelSizedWare()) {
                BarrelAmountAlwaysVisible barrelAmount = context.getBean(BarrelAmountAlwaysVisible.class);
                barrelAmount.amountProperty().bind(stored.asString());
                row.add(new ControlTableCell(barrelAmount));
            } else {
                BaleAmountAlwaysVisible baleAmount = context.getBean(BaleAmountAlwaysVisible.class);
                baleAmount.amountProperty().bind(stored.asString());
                row.add(new ControlTableCell(baleAmount));
            }

            final IntegerBinding buyAmount = requiredWare.buyAmountProperty();
            if (ware.isBarrelSizedWare()) {
                BarrelAmountAlwaysVisible barrelAmount = context.getBean(BarrelAmountAlwaysVisible.class);
                barrelAmount.amountProperty().bind(buyAmount.asString());
                row.add(new ControlTableCell(barrelAmount));
            } else {
                BaleAmountAlwaysVisible baleAmount = context.getBean(BaleAmountAlwaysVisible.class);
                baleAmount.amountProperty().bind(buyAmount.asString());
                row.add(new ControlTableCell(baleAmount));
            }

            IntegerBinding priceBinding = requiredWare.buyPriceProperty();
            CoinPriceAlwaysVisible price = context.getBean(CoinPriceAlwaysVisible.class);
            price.amountProperty().bind(priceBinding.asString());
            row.add(new ControlTableCell(price));

            model.add(row);
        }
        return model;
    }

    @Override
    protected Table createTopTable() {
        int capacity = shipFactory.calculateInitialCapacity(getCurrentShipType(), getCity().getCoordinates().getX());
        List<IWeaponSlot> shipWeapons = weaponsLocationFactory.getShipWeaponsLocation(getCurrentShipType(), EShipUpgrade.NONE);
        Table model = new Table();
        // Add dummy header which is not rendered
        TableHeader header = new TableHeader(5);
        model.setHeader(header);
        model.setColumnWidth(70, 70, 70, 70, 70);
        TableRow row = new TableRow();
        BarrelAmount capacityAmount = context.getBean(BarrelAmount.class);
        capacityAmount.setAmount(capacity);
        row.add(new ControlTableCell(capacityAmount));
        WeaponSlotCount slotCount = shipService.getWeaponSlotCount(shipWeapons);
        row.add(new StaticTextTableCell(String.valueOf(slotCount.getNbSmallSlots())));
        row.add(new StaticTextTableCell(String.valueOf(slotCount.getNbLargeSlots())));
        row.add(new StaticTextTableCell(String.valueOf(shipFactory.getShipSpeed(getCurrentShipType()))));
        row.add(new StaticTextTableCell(String.valueOf(shipFactory.getMinimalSailors(getCurrentShipType()))));
        model.add(row);
        return model;
    }

    @Override
    protected BooleanBinding actionEnabledBinding() {
        return new BooleanBinding() {
            {
                for (EWare ware : MATERIALS) {
                    RequiredWareCityStorage requiredWare = requirementsMap.get(ware);
                    super.bind(requiredWare.canBuyProperty());
                }
                super.bind(getPlayer().getCompany().cashProperty());

            }
            @Override
            protected boolean computeValue() {
                long cash = getPlayer().getCompany().getCash();
                long buildCosts = calculateTotalUpgradeCosts();
                if (cash < buildCosts) {
                    return false;
                }
                for (EWare ware : MATERIALS) {
                    RequiredWareCityStorage requiredWare = requirementsMap.get(ware);
                    if (!requiredWare.canBuyProperty().get()) {
                        return false;
                    }
                }
                return true;
            }
        };
    }
    @Override
    protected EventHandler<MouseEvent> createNextAction() {
        return event -> {
            dialogUtil.createNextActionForShips(city.getPlayersShips(), currentShip).handle(event);
            selectionChanged();
        };
    }

    @Override
    protected EventHandler<MouseEvent> createPreviousAction() {
        return event -> {
            dialogUtil.createPreviousActionForShips(city.getPlayersShips(), currentShip).handle(event);
            selectionChanged();
        };
    }

    @Override
    protected BooleanProperty disableNavigation() {
        return new SimpleBooleanProperty(city.getPlayersShips().size() <= 1);
    }

    @Override
    protected String getTitleText() {
        return currentShip.get().getName();
    }
    @Override
    protected void selectionChanged() {
        super.selectionChanged();
        initializeRequirements();
        Group footerText = createFooterText();
        footerText.setLayoutX(50);
        footerText.setLayoutY(250 + 36 + 7 * 24);
        for (Iterator<Node> iterator = getContent().iterator(); iterator.hasNext(); ) {
            Node child =  iterator.next();
            if ("footerText".equals(child.getId())) {
                iterator.remove();
                break;
            }
        }
        getContent().add(footerText);
    }
}
