package ch.sahits.game.openpatrician.display.dialog.tavern;

import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.clientserverinterface.service.ClientServerFactory;
import ch.sahits.game.openpatrician.clientserverinterface.service.DialogTemplateParameterSupplier;
import ch.sahits.game.openpatrician.clientserverinterface.service.EDialogTemplateType;
import ch.sahits.game.openpatrician.clientserverinterface.service.LinearDeadlinePremiumCalculator;
import ch.sahits.game.openpatrician.display.ClientViewState;
import ch.sahits.game.openpatrician.display.dialog.CloseButtonDialog;
import ch.sahits.game.openpatrician.display.dialog.service.TavernDialogUtil;
import ch.sahits.game.openpatrician.display.event.handler.impl.EventHandlerFactory;
import ch.sahits.game.openpatrician.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.openpatrician.event.EViewChangeEvent;
import ch.sahits.game.openpatrician.event.NoticeBoardUpdate;
import ch.sahits.game.openpatrician.javafx.control.DecoratedText;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.DisplayMessage;
import ch.sahits.game.openpatrician.model.EMessageCategory;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.impl.TavernState;
import ch.sahits.game.openpatrician.model.people.ITraveler;
import ch.sahits.game.openpatrician.model.service.ModelTranslations;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import javafx.beans.binding.BooleanBinding;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.time.LocalDateTime;
import java.util.HashMap;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 *         Created on Dec 26, 2013
 */
@Slf4j
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class TavernTravelerDialog extends CloseButtonDialog {
    @Autowired
    private TavernDialogUtil tavernUtil;
    @Autowired
    private Date date;
    @Autowired
    private ClientViewState viewState;
    @Autowired
    private EventHandlerFactory eventHandlerFactory;
    @Autowired
    private Locale locale;

    @Autowired
    private MessageSource messageSource;
    @Autowired
    private ModelTranslations modelTranslator;
    @Autowired
    private ClientServerFactory clientServerFactory;

    private final ICityPlayerProxyJFX city;
    private final TavernState tavernState;
    private final ITraveler traveler;

    public TavernTravelerDialog(ICityPlayerProxyJFX city) {
        super();
        this.city = city;
        tavernState = city.getCity().getCityState().getTavernState();
        traveler = tavernState.getTraveler();
    }
    @PostConstruct
    private void initializeDialog() {
        setTitle(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.TavernTravelerDialog.title", new Object[]{}, locale.getCurrentLocal()));
        IPlayer player = city.getPlayer();
        final LocalDateTime arrivalDate = traveler.getArrivalAtDestinationDate();
        String d = modelTranslator.toDisplayString(arrivalDate);
        String loadedText = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.TavernTravelerDialog.dialog",
                new Object[]{city.getPlayer().getLastName(), modelTranslator.getLocalDisplayName(player.getCareerLevel()), player.getName(), traveler.getDestination().getName(), d}, locale.getCurrentLocal());


        HashMap<String, Object> parameters = new HashMap<>();
        parameters.put("amount",traveler.getPremium());
        DecoratedText dt = decoratedTextFactory.createDecoratedText(loadedText, parameters);

        final OpenPatricianLargeWaxButton acceptButton = new OpenPatricianLargeWaxButton(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.TavernBaseSideRoomPersonDialog.accept", new Object[]{}, locale.getCurrentLocal()));
        acceptButton.getStyleClass().add("actionButton");
        int BUTTON_X = (WIDTH - 124) / 2;
        acceptButton.setLayoutX(BUTTON_X);
        int LOWER_BUTTON_Y = CLOSE_BTN_Y_POS - 24;
        acceptButton.setLayoutY(LOWER_BUTTON_Y);
        acceptButton.setOnAction(mouseEvent -> {
            try {
                IShip ship = (IShip) city.getActiveShip();
                if (ship != null) {
                    ship.addPassenger(traveler);
                    final LocalDateTime now = date.getCurrentDate();
                    final LocalDateTime deadline = now.plusMonths(1);
                    LinearDeadlinePremiumCalculator premiumCalculator = clientServerFactory.premiumCalculator(now, deadline, traveler.getPremium());
                    DialogTemplateParameterSupplier parameterSupplier = new DialogTemplateParameterSupplier(new Object[]{traveler.getDestination().getName(), ship.getName(), clientServerFactory.dateSupplier(), premiumCalculator});
                    eventHandlerFactory.getPassengerLeavingOnShipArrivalHandler(ship, traveler.getDestination(), city.getPlayer(), deadline, EDialogTemplateType.TRAVELER_LEAVES_SHIP, parameterSupplier, premiumCalculator);
                    clientEventBus.post(new DisplayMessage(EMessageCategory.TRADE, "ch.sahits.game.openpatrician.display.dialog.TavernBaseSideRoomPersonDialog.acceptMessage", ship.getName()));

                    executeOnCloseButtonClicked();
                    traveler.leave();
                }
            } catch (RuntimeException e) {
                log.error("Failed to accept traveler on ship", e);
            }

        });
        BooleanBinding disable = disableAcceptBinding();
        acceptButton.setDisable(disable.get());
        disable.addListener((observableValue, oldValue, newValue) -> acceptButton.setDisable(newValue));

        getContent().addAll(dt, acceptButton);
    }

    private BooleanBinding disableAcceptBinding() {
        return new BooleanBinding() {
            {
                super.bind(city.getPlayersShips());
                for (IShip ship : city.getPlayersShips()) {
                    super.bind(ship.passengerPresentProperty());
                }
            }
            @Override
            protected boolean computeValue() {
                if (city.getActiveShip() != null && city.getActiveShip() instanceof IShip) {
                    return ((IShip)city.getActiveShip()).getPassenger().isPresent(); // If passenger is on board, disable
                } else {
                    return false;
                }
            }
        };
    }
    /**
     * Update the notice board and close the dialog.
     */
    @Override
    public void executeOnCloseButtonClicked() {
        ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.NOTICE_TAVERN);
        clientEventBus.post(new NoticeBoardUpdate(proxy));
        super.executeOnCloseButtonClicked();
    }

    @Override
    public void close() {
        super.close();
        tavernUtil.stopTalkingToPerson(tavernState, city.getPlayer());
    }
}
