package ch.sahits.game.openpatrician.display.event.handler.impl;

import ch.sahits.game.openpatrician.event.data.ShipEntersPortEvent;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import com.google.common.eventbus.Subscribe;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.time.LocalDateTime;

/**
 * Event handler for a ship arrival for loading of wares for someone else.
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Feb 16, 2014
 */
@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class ShipArrivalForLoading extends WaitTimeForShipArrival {
    private final int requiredAmount;
    private final int premium;
    @Autowired
    @XStreamOmitField
    private EventHandlerFactory eventHandlerFactory;

    public ShipArrivalForLoading(ICity city, IPlayer player, LocalDateTime deadline, int requiredAmount, int premium) {
        super(city, player, deadline);
        this.requiredAmount = requiredAmount;
        this .premium = premium;
    }
    @PostConstruct
    private void initializeEventBus() {
        clientServerEventBus.register(this);
    }
    @PreDestroy
    private void unregisterEventBus() { clientServerEventBus.unregister(this);}
    @Override
    public boolean execute() {
        unregisterEventBus();
        if (hasShipEnoughCapacity(getShip(), requiredAmount)) {
            IShip s = (IShip) getShip();
            s.setOccupiedSpace(requiredAmount + s.getOccupiedSpace());
            LocalDateTime waitTenDays = getDate().getCurrentDate().plusDays(10);

            eventHandlerFactory.getDeliverPackagesHandler(getCity(), s, waitTenDays, premium, requiredAmount);
            return true;
        }
        return false;
    }


    /**
     * Check if the ship has enough capacity to load wares
     * @param ship for which to check capacity
     * @param requiredAmount in barrels
     * @return true if the ship has enough capacity for the required amount.
     */
    private static boolean hasShipEnoughCapacity(INavigableVessel ship, int requiredAmount) {
        return ship instanceof IShip && ship.getCapacity()>requiredAmount;
    }

    @Override
    @Subscribe
    public void handleShipEntersPort(ShipEntersPortEvent event) {
        super.handleShipEntersPort(event);
    }
}
