package ch.sahits.game.openpatrician.display.notice;

import ch.sahits.game.graphic.image.IDataImageLoader;
import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.display.dialog.service.DialogUtil;
import ch.sahits.game.openpatrician.display.dialog.ship.ShipModelForShipList;
import ch.sahits.game.openpatrician.display.model.BarrelConfiguration;
import ch.sahits.game.openpatrician.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.openpatrician.javafx.control.BaleIconView;
import ch.sahits.game.openpatrician.javafx.control.BarrelIconView;
import ch.sahits.game.openpatrician.javafx.control.CoinIconView;
import ch.sahits.game.openpatrician.javafx.model.ENoticeBoardType;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.product.AmountablePrice;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.service.ModelTranslations;
import ch.sahits.game.openpatrician.model.ship.EShipTravelState;
import ch.sahits.game.openpatrician.model.ship.IConvoy;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.binding.StringBinding;
import javafx.beans.property.SimpleObjectProperty;
import javafx.beans.value.ObservableValue;
import javafx.geometry.HPos;
import javafx.scene.control.Label;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.layout.ColumnConstraints;
import javafx.scene.layout.FlowPane;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.Pane;
import javafx.scene.layout.Region;
import javafx.scene.layout.VBox;
import javafx.scene.paint.Color;
import javafx.scene.shape.Path;
import javafx.scene.shape.Rectangle;
import javafx.scene.shape.Shape;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;

import static ch.sahits.game.openpatrician.javafx.control.NoticeBoard.DEFAULT_WIDTH;

/**
 * Service to provide content for the noticeboard in the form of nodes.
 * @author Andi Hotz, (c) Sahits GmbH, 2018
 * Created on Jul 17, 2018
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class NoticeBoardContentProvider {
    @Autowired
    private IDataImageLoader imageLoader;
    @Autowired
    private MessageSource messageSource;
    @Autowired
    private Locale locale;
    @Autowired
    private DialogUtil dialogUtil;
    @Autowired
    private ModelTranslations translations;
    @Autowired
    private Date date;
    @Autowired
    private BarrelConfiguration barrelConfiguration;
    @Autowired
    private ApplicationContext context;
    @Autowired
    private ShipService shipService;

    private Path barrelShape;

    @PostConstruct
    private void init() {
        barrelShape = barrelConfiguration.createBarrelShape();
    }

    /**
     * Create the header node for to be placed in the notice boards head.
     * @param type for which to create the header.
     * @param proxy of the current state.
     * @return Node representing the header.
     */
    public Region createHeader(ENoticeBoardType type, ViewChangeCityPlayerProxyJFX proxy) {
        switch (type) {
            case SHIP_WARE_INFO:
                return createShipWareInfoHeader(proxy);
            default:
                throw new IllegalStateException("For "+type+" no header is defined");
        }
    }


    /**
     * Create the content to be placed in the notice boards scroll pane.
     * @param type for which to create the content
     * @param proxy of the current state
     * @return Node representing the content.
     */
    public Region createContent(ENoticeBoardType type, ViewChangeCityPlayerProxyJFX proxy) {
        switch (type) {
            case SHIP_WARE_INFO:
                return createShipWareInfoContent(proxy);
            default:
                throw new IllegalStateException("For "+type+" no content is defined");
        }
    }

    /**
     * Create an observable to which an change listener can be subscribed that indicates,
     * that the header and content need updating.
     * @param type for which to create the content
     * @param proxy of the current state
     * @return Observable
     */
    public ObservableValue<?> createObservable(ENoticeBoardType type, ViewChangeCityPlayerProxyJFX proxy) {
        switch (type) {
            case SHIP_WARE_INFO:
                return shipWareObservable(proxy);
            default:
                throw new IllegalStateException("For "+type+" no observable is defined");
        }
    }

    // Header


    /**
     * Right side barrel, with capacity, loaded and ocupied by weapons and passangers.
     * Left Side:
     *  - Auto trade icon, ship/convoy Ship name
     *  - sailing duration/status icon/ destination
     *  - DialogUtil#createShipInfoOnThreeLines(javafx.beans.property.ObjectProperty)
     * @param proxy
     * @return
     */
    private Region createShipWareInfoHeader(ViewChangeCityPlayerProxyJFX proxy) {
        INavigableVessel vessel = proxy.getActiveShip();
        VBox headerLeft = new VBox();
        IShip ship;

        FlowPane firstLine = new FlowPane();
        if (vessel.getAutotrading().isPresent()) {
            Image autotrade = imageLoader.getImage("icons/32/icon_autotrade");
            ImageView icon = new ImageView(autotrade);
            icon.setId("autotrade");
            firstLine.getChildren().add(icon);
        }
        if (vessel instanceof IShip) {
            Image img = imageLoader.getImage("icons/32/sailing-icon");
            ImageView icon = new ImageView(img);
            icon.setId("singleShip");
            firstLine.getChildren().add(icon);
            ship = (IShip) vessel;
        } else {
            Image img = imageLoader.getImage("icons/32/icon_convoy");;
            ImageView icon = new ImageView(img);
            icon.setId("convoy");
            firstLine.getChildren().add(icon);
            ship = ((IConvoy) vessel).getOrlegShip();
        }
        EShipTravelState travelState = ship.travelState().get();
        Label shipName = new Label(vessel.getName());
        shipName.setId("shipName");
        shipName.getStyleClass().add("noticeBoardText");
        firstLine.getChildren().add(shipName);
        headerLeft.getChildren().add(firstLine);

        FlowPane secondLine = new FlowPane();
        switch (travelState) {
            case TRAVEL_TO_CITY:
            case TRAVEL_TO_DESTINATION:
                int duration = shipService.calculateDaysTillArrival(ship);
                String s = messageSource.getMessage("ch.sahits.game.openpatrician.display.notice.NoticeBoardContentProvider.remainingTravelDuration",
                        new Object[]{duration}, locale.getCurrentLocal());
                Label remaining = new Label(s);
                remaining.setId("remaining");
                remaining.getStyleClass().add("noticeBoardText");
                secondLine.getChildren().add(remaining);
                break;
        }
        ImageView statusIcon = new ImageView(dialogUtil.getStatusIcon(travelState));
        statusIcon.setId("statusIcon");
        secondLine.getChildren().add(statusIcon);
        ShipModelForShipList model = context.getBean(ShipModelForShipList.class, ship);
        StringBinding destinationBinding = model.destination();
        String s = destinationBinding.get();
        Label destination = new Label(s);
        destination.setId("destination");
        destination.getStyleClass().add("noticeBoardText");
        destination.textProperty().bind(destinationBinding);
        secondLine.getChildren().add(destination);
        headerLeft.getChildren().add(secondLine);

        headerLeft.getChildren().add(dialogUtil.createShipInfoOnThreeLines(new SimpleObjectProperty<>(ship), false, 200));

        Pane headerRight = new Pane();
        ImageView barrel = barrelConfiguration.getBarrelView();
        headerRight.getChildren().add(barrel);
        int barrelBottom = 63;
        int cargoCapacity = vessel.getCapacity(); // what is free
        double size = vessel.getLoadableSpace();
        int loadedWares = vessel.getLoadBinding().get();
        double unloadable = size - loadedWares - cargoCapacity;
        Shape unloadableShape = null;
        if (unloadable > 0) {
            Rectangle unloadableRect = barrelConfiguration.getUnloadableShape(ship);
            unloadableShape = Shape.intersect(unloadableRect, barrelShape);
            unloadableShape.setFill(new Color(0, 0, 1, 0.65));
            unloadableShape.setId("unloadableShape");
            headerRight.getChildren().add(unloadableShape);

            Label weaponSpace = new Label(String.valueOf((int)unloadable));
            weaponSpace.setId("weaponSpace");
            weaponSpace.getStyleClass().add("noticeBoardText");
            weaponSpace.setLayoutX(55);
            weaponSpace.setLayoutY(58);
            headerRight.getChildren().add(weaponSpace);
        }
        if (loadedWares > 0) {
            Rectangle waresRect = barrelConfiguration.getCargoShape(ship);
            Shape intersect = Shape.intersect(waresRect, barrelShape);
            Shape wareShape;
            if (unloadableShape != null) {
                wareShape = Shape.subtract(intersect, unloadableShape);
            } else {
                wareShape = intersect;
            }
            wareShape.setFill(new Color(0, 0.39062, 0, 0.65));
            wareShape.setId("wareShape");
            headerRight.getChildren().add(wareShape);

            Label wareSpace = new Label(String.valueOf(loadedWares));
            wareSpace.setId("wareSpace");
            wareSpace.getStyleClass().add("noticeBoardText");
            wareSpace.setLayoutX(55);
            wareSpace.setLayoutY(barrelBottom / 2);
            headerRight.getChildren().add(wareSpace);
        }
        Label totalCapacity = new Label(String.valueOf((int)size));
        totalCapacity.setId("totalCargoCapacity");
        totalCapacity.getStyleClass().add("noticeBoardText");
        totalCapacity.setLayoutX(55);
        headerRight.getChildren().add(totalCapacity);
        headerRight.setMaxWidth(75);
        headerRight.setMinWidth(75);
        headerRight.setLayoutX(DEFAULT_WIDTH - 75);

        Pane header = new Pane(headerLeft, headerRight);
        header.setId("header");
        header.setMaxWidth(DEFAULT_WIDTH);
        header.setMinWidth(DEFAULT_WIDTH);

        return header;
    }

    // Content creation
    private Region createShipWareInfoContent(ViewChangeCityPlayerProxyJFX proxy) {
        GridPane content = new GridPane();
        content.setId("shipWareInfo");
        INavigableVessel vessel = proxy.getActiveShip();
        int row = 0;
        for (IWare ware : vessel.getLoadedWares()) {
            AmountablePrice<IWare> amountable = vessel.getWare(ware);
            int amount = amountable.getAmount();
            if (amount <= 0) {
                continue;
            }
            int avgValue = amountable.getAVGPrice();
            String wareName = translations.getLocalDisplayName(ware);

            Label amountLbl = new Label(String.valueOf(amount));
            amountLbl.setId("amount"+row);
            amountLbl.getStyleClass().add("noticeBoardText");
            content.add(amountLbl, 0, row);
            GridPane.setHalignment(amountLbl, HPos.RIGHT);

            if (ware.isBarrelSizedWare()) {
                BarrelIconView barrel = context.getBean(BarrelIconView.class);
                barrel.setId("barrelIcon"+row);
                content.add(barrel, 1, row);
                GridPane.setHalignment(barrel, HPos.LEFT);
            } else {
                BaleIconView bale = context.getBean(BaleIconView.class);
                bale.setId("baleIcon"+row);
                content.add(bale, 1, row);
                GridPane.setHalignment(bale, HPos.LEFT);
            }

            Label wareNameLbl = new Label(wareName);
            wareNameLbl.setId("wareName"+row);
            wareNameLbl.getStyleClass().add("noticeBoardText");
            content.add(wareNameLbl, 2, row);
            GridPane.setHalignment(wareNameLbl, HPos.LEFT);

            Label priceLbl = new Label(String.valueOf(avgValue));
            priceLbl.setId("price"+row);
            priceLbl.getStyleClass().add("noticeBoardText");
            content.add(priceLbl, 3, row);
            GridPane.setHalignment(priceLbl, HPos.RIGHT);

            CoinIconView coin = context.getBean(CoinIconView.class);
            coin.setId("coinIcon"+row);
            content.add(coin, 4, row);
            GridPane.setHalignment(coin, HPos.LEFT);

            content.getColumnConstraints().addAll(new ColumnConstraints(40),
                                                  new ColumnConstraints(32),
                                                  new ColumnConstraints(60),
                                                  new ColumnConstraints(40),
                                                  new ColumnConstraints(32));

            row++;
        }
        if (vessel instanceof IShip && ((IShip) vessel).getPassenger().isPresent() || vessel instanceof IConvoy && (((IConvoy) vessel).getOrlegShip()).getPassenger().isPresent()) {
            String s = messageSource.getMessage("ch.sahits.game.openpatrician.display.notice.NoticeBoardContentProvider.passenger",
                    new Object[]{}, locale.getCurrentLocal());
            Label passengerLbl = new Label(s);
            passengerLbl.setId("passengerLabel");
            passengerLbl.getStyleClass().add("noticeBoardText");
            content.add(passengerLbl, 0, row, 5, 1);
        }
        return content;
    }

    // Observables

    private ObservableValue<Boolean> shipWareObservable(ViewChangeCityPlayerProxyJFX proxy) {
        return new BooleanBinding() {
            {
                super.bind(proxy.activeShipProperty());
                INavigableVessel activeShip = proxy.getActiveShip();
                super.bind(activeShip.getLoadBinding());
                if (activeShip instanceof IShip) {
                    super.bind(((IShip) activeShip).travelState());
                } else {
                    super.bind(((IConvoy) activeShip).getOrlegShip().travelState());
                }
                super.bind(date.dayDateBinding());
            }
            private boolean value = true;
            @Override
            protected boolean computeValue() {
                value = !value;
                return value;
            }
        };
    }
}
