package ch.sahits.game.openpatrician.engine.sea;

import ch.sahits.game.event.data.HeuristicGraphInitialisationComplete;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.MapType;
import ch.sahits.game.openpatrician.model.city.ICity;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.eventbus.AsyncEventBus;
import javafx.geometry.Point2D;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;

/**
 * This singleton provides the heuristic for a specific map for the A* algorithm.
 * Initially the heuristic from all nodes to all cities is calculated, based on:
 * <ul>
 *     <li>The Eucledian distance</li>
 * </ul>
 * The heuristic can also be queried if a source has it's heuristic calculated and calculate
 * the heuristic values for that node.
 *
 * The heuristic may not be admissable, meaning that the value given in the heuristic for the
 * distance between A and B may be larger than the actual distance, however the result should still
 * be good enough.
 *
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Dec 31, 2015
 */
@ClassCategory(EClassCategory.SINGLETON_BEAN)
@Component
@Lazy
public class AStarHeuristicProvider extends BaseGraphCalulationService{
    private final Logger logger = LogManager.getLogger(getClass());
    @Autowired
    @Qualifier("serverThreadPool")
    private Executor serverThreadPool;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    private final Object lock = new Object();
    /**
     * A map of heuristic from a node to each other node in the graph.
     * Key of the map is the source and the value is a map of the various destinations and their
     * weight from the source (outer key) to the destination (inner key)
     */
    @MapType(key = Point2D.class, value = Map.class)
    private final Map<Point2D, Map<Point2D, Double>> heuristicMap = new ConcurrentHashMap<>();
    private int width;
    private int height;



    @PostConstruct
    private void initialize() {
        try {
            initImage();
            serverThreadPool.execute(() -> createGraph());
        } catch (IOException e) {
            logger.error("Failed to create black and white image from the map", e);
        }
    }
    @VisibleForTesting
    void initImage() throws IOException {
        super.initImage();
        width = mapImage.getWidth();
        height = mapImage.getHeight();
    }

    @VisibleForTesting
    void createGraph() {
        long timestamp = System.currentTimeMillis();
        synchronized (lock) {
            List<ICity> cities = map.getCities();
            for (ICity city : cities) {
                Map<Point2D, Double> hueristicForTarget = calculateHeuristicForSource(city.getCoordinates(), cities);
                heuristicMap.put(city.getCoordinates(), hueristicForTarget);
            }
            for (int x = 0; x < width; x += CHECK_DISTANCE) { // TODO: andi 1/3/16 split to different threads
                for (int y = 0; y < height; y += CHECK_DISTANCE) {
                    Point2D source = getPoint(x, y);
                    Map<Point2D, Double> hueristicForTarget = calculateHeuristicForSource(source, cities);
                    heuristicMap.put(source, hueristicForTarget);
                }
            }

        }
        long took = System.currentTimeMillis() - timestamp;
        logger.debug("Created heuristic for "+heuristicMap.size()+" sources connecting to "+heuristicMap.values().iterator().next().size()+" nodes taking "+took+"ms");
        clientServerEventBus.post(new HeuristicGraphInitialisationComplete());
    }

    private Map<Point2D,Double> calculateHeuristicForSource(Point2D source, List<ICity> cities) {
        HashMap<Point2D, Double> distance = new HashMap<>();
        for (ICity city : cities) {
            final Point2D coordinates = city.getCoordinates();
            double dist = calculateWeight(source, coordinates);
            distance.put(coordinates, dist);
        }

        return distance;
    }
    @VisibleForTesting
    protected double calculateWeight(Point2D from, Point2D to) {
        if (from.equals(to)) {
            return 0;
        }

        double distance = from.distance(to);
        // Should calculate the amount of land between the two points

        return distance;
    }


    /**
     * Check if the heuristic for a target location is calculated.
     * @param source location
     * @return true if the heuristic is present.
     */
    public boolean heuristicForSourceAvailable(Point2D source) {
        return heuristicMap.containsKey(source);
    }

    /**
     * Retrieve the heuristic map. This method is blocking while the heuristic is
     * beeing calculated to ensure there are no out of date data contained.
     * @return Heuristic map for target locations to any location in the grid.
     */
    public Map<Point2D, Map<Point2D, Double>> getHeuristic() {
        synchronized (lock) {
            return heuristicMap;
        }
    }

    /**
     * Add a new target node to the heuristic.
     * @param target location to be added.
     */
    public void addTargetNodeToHeuristic(Point2D target) {
        Preconditions.checkNotNull(target, "The target node may not be null");
         synchronized (lock) {
                 for (Entry<Point2D, Map<Point2D, Double>> entry : heuristicMap.entrySet()) {
                     if (!entry.getValue().containsKey(target)) {
                         Point2D source = entry.getKey();
                         double dist = calculateWeight(source, target);
                         entry.getValue().put(target, dist);

                     }
                 }
         }
    }

    /**
     * Add a new target node to the heuristic.
     * @param source location to be added.
     */
    public void addSourceNodeToHeuristic(Point2D source) {
        Preconditions.checkNotNull(source, "The source node may not be null");
        if (!heuristicForSourceAvailable(source)) {
            synchronized (lock) {
                serverThreadPool.execute(() -> {
                     Map<Point2D, Double> hueristicForTarget = calculateHeuristicForSource(source, map.getCities());
                     heuristicMap.put(source, hueristicForTarget);
                });
            }
        }
    }

}