package ch.sahits.game.openpatrician.engine.sea;

import ch.sahits.game.event.data.ShipPositionUpdateEvent;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.ListType;
import ch.sahits.game.openpatrician.model.IMap;
import ch.sahits.game.openpatrician.model.sea.ILocationTracker;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import com.google.common.collect.Interner;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.geometry.Point2D;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.List;

/**
 * Component that helps split up the amount of ships into segements of the navigable map.
 * This reduces the omount of ships that need to be checked if a ship nearby is searched.
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Dec 13, 2015
 */
@Component
@Lazy
@ClassCategory({EClassCategory.SINGLETON_BEAN, EClassCategory.SERIALIZABLE_BEAN})
public class LocationTracker implements ILocationTracker {
    @XStreamOmitField
    private final Logger logger = LogManager.getLogger(getClass());
    @Autowired
    private IMap map;
    @Autowired
    @Qualifier("serverClientEventBus")
    @XStreamOmitField
    private AsyncEventBus clientServerEventBus;
    @Autowired
    @XStreamOmitField
    protected Interner<Point2D> pointInterner;

    private int segmentWidth;
    @ListType(INavigableVessel.class)
    private List<WeakReference<INavigableVessel>>[] segments;
    @PostConstruct
    void initialize() {
        segmentWidth = (int) Math.rint(map.getDimension().getWidth())/10;
        segments = new List[10];
        for (int i = 0; i < segments.length; i++) {
            segments[i] = new ArrayList<>();
        }
        clientServerEventBus.register(this);
    }
    @PreDestroy
    private void unregister() {
        clientServerEventBus.unregister(this);
    }

    private int calculateHorizontalIndex(Point2D location) {
        int index = (int) Math.floor(location.getX() / segmentWidth);
        return index;
    }
    public void add(INavigableVessel ship) {
        int index = calculateHorizontalIndex(ship.getLocation());
        segments[index].add(new WeakReference(ship));
    }
    public void remove(INavigableVessel ship) {
        int index = calculateHorizontalIndex(ship.getLocation());
        if (!segments[index].removeIf(wr -> ship.equals(wr.get()))) {
             logger.warn("The ship {} could not be found and removed in segment {}", ship, index);
        }
    }
    public List<INavigableVessel> getShipsInSegment(Point2D location) {
        int index = calculateHorizontalIndex(location);
        List<WeakReference<INavigableVessel>> references = segments[index];
        ArrayList<INavigableVessel> copyList = new ArrayList<>();
        for (WeakReference<INavigableVessel> reference : references) {
            if (reference.get() != null) {
                copyList.add(reference.get());
            }
        }
        return copyList;
    }

    public List<INavigableVessel> getShipsInSegments(Point2D location, int radius) {
        int index = calculateHorizontalIndex(location);
        final Point2D locationBefore = pointInterner.intern(new Point2D(location.getX() - radius, location.getY()));
        int indexBefore = calculateHorizontalIndex(locationBefore);
        final Point2D locationAfter = pointInterner.intern(new Point2D(location.getX() + radius, location.getY()));
        int indexAfter = calculateHorizontalIndex(locationAfter);
        List<INavigableVessel> list = new ArrayList<>();
        if (indexBefore != index) {
            list.addAll(getShipsInSegment(locationBefore));
        }
        list.addAll(getShipsInSegment(location));
        if (indexAfter != index) {
            list.addAll(getShipsInSegment(locationAfter));
        }
        return list;
    }
    @Subscribe
    public void handleShipMove(ShipPositionUpdateEvent event) {
        if (event.getFromLocation().getX() != event.getToLocation().getX()) {
            int oldIndex = calculateHorizontalIndex(event.getFromLocation());
            int newIndex = calculateHorizontalIndex(event.getToLocation());
            if (oldIndex != newIndex) {
                segments[oldIndex].removeIf(wr -> event.getShip().equals(wr.get()));
                segments[newIndex].add(new WeakReference<>(event.getShip()));
            }
        }
    }
}
