package ch.sahits.game.openpatrician.engine.sea;

import ch.sahits.datastructure.GenericPair;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.DependentInitialisation;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.LazySingleton;
import ch.sahits.game.openpatrician.dialog.DialogTemplate;
import ch.sahits.game.openpatrician.model.DisplayMessage;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.event.TargetedEvent;
import ch.sahits.game.openpatrician.model.people.ICaptain;
import ch.sahits.game.openpatrician.model.people.IShipOwner;
import ch.sahits.game.openpatrician.model.service.ShipService;
import ch.sahits.game.openpatrician.model.ship.IConvoy;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.ship.IShipGroup;
import ch.sahits.game.openpatrician.model.ship.ShipFactory;
import ch.sahits.game.openpatrician.model.weapon.EWeapon;
import ch.sahits.game.openpatrician.util.StartNewGameBean;
import ch.sahits.game.openpatrician.util.model.Tristate;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.eventbus.AsyncEventBus;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Random;

/**
 * The sea fight service helps calculating the winner of a sea fight based on two sets of ships.
 * The central method is {@link #calculateOutcome(List, List)}. There are helper methods <code>explodeShipList</code>
 * to convert the {@link IShip}s to the required List of ships. There are further helper methods to collect the ships
 * lists again after the fight.<br>
 * The only case the must be handled without the help of a helper method is if a single ship (not a group or convoy) of
 * a player (AI or human) is attacked or defends and is victorious. This might result in a list of ships that should then
 * all travel to the same destination as single ships.
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Dec 08, 2015
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
@DependentInitialisation(StartNewGameBean.class)
public class SeaFightService {

    private static final int DAMAGE_PER_STRENGTH = 2;
    private static final double DEATH_TOLL_PER_DAMAGE = 0.2;

    @Autowired
    private ShipService shipStrenghtService;
    @Autowired
    private ShipFactory shipFactory;
    @Autowired
    private Random rnd;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;

    /**
     * Calculate the outcome of the fight. This includes damaging, destroying or capturing ships from either list.
     *
     * @param attackingShips list of attacking ships.
     * @param defendingShips list of defending ships.
     */
    public void calculateOutcome(List<IShip> attackingShips, List<IShip> defendingShips) {
        List<GenericPair<List<IShip>, List<IShip>>> pairings = calculatePairing(attackingShips, defendingShips);
        List<IShip> sunkenDefendingShips = new ArrayList<>();
        List<IShip> sunkenAttackingShips = new ArrayList<>();
        List<IShip> capturedDefendingShips = new ArrayList<>();
        List<IShip> capturedAttackingShips = new ArrayList<>();
        IShipOwner attacker = attackingShips.get(0).getOwner();
        IShipOwner defender = defendingShips.get(0).getOwner();

        for (Iterator<GenericPair<List<IShip>, List<IShip>>> pairingIter = pairings.iterator(); pairingIter.hasNext(); ) {
            GenericPair<List<IShip>, List<IShip>> pairing =  pairingIter.next();

            // start with the attacking ships
            boolean attackShipTurn = true;
            runAttackMove(pairings, sunkenDefendingShips, sunkenAttackingShips, capturedDefendingShips, capturedAttackingShips, pairingIter, pairing, attackShipTurn);
            // repeat for defending ships
            if (pairings.size() > 0) {
                attackShipTurn = false;
                runAttackMove(pairings, sunkenDefendingShips, sunkenAttackingShips, capturedDefendingShips, capturedAttackingShips, pairingIter, pairing, attackShipTurn);
            }
        }
        // remove ships that are in both captured lists
        for (Iterator<IShip> iterator = capturedAttackingShips.iterator(); iterator.hasNext(); ) {
            IShip ship = iterator.next();
            if (capturedDefendingShips.contains(ship)) {
                iterator.remove();
                capturedDefendingShips.remove(ship);
            }
        }
        // post message to either fleet owner with the stats of the fight
        // defeated, survived, successful

        postMessage(attackingShips, defendingShips, sunkenDefendingShips, sunkenAttackingShips, capturedDefendingShips, capturedAttackingShips, attacker, defender);

    }
    @VisibleForTesting
    void postMessage(List<IShip> attackingShips, List<IShip> defendingShips, List<IShip> sunkenDefendingShips, List<IShip> sunkenAttackingShips, List<IShip> capturedDefendingShips, List<IShip> capturedAttackingShips, IShipOwner attacker, IShipOwner defender) {
        if (attacker instanceof IHumanPlayer) {
            String titleKey = "";
            String messageKey = "";
            Object[] args = null;
            // STATE_ONE: defeated, STATE_TWO: successful, STATE_THREE: survived
            Tristate tristate;
            int lostShips = sunkenAttackingShips.size() + capturedAttackingShips.size();
            if (lostShips == 0) {
                tristate = Tristate.STATE_TWO;
            } else if (lostShips == attackingShips.size()) {
                tristate = Tristate.STATE_ONE;
            } else {
                tristate = Tristate.STATE_THREE;
            }
            switch (tristate) {
                case STATE_ONE: // defeated
                        titleKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.unsuccessfulAttackAttacker.title";
                    if (sunkenAttackingShips.size() == 1 && capturedAttackingShips.size() == 0) {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.unsuccessful.sunkOneShip";
                        args = new Object[]{sunkenAttackingShips.get(0).getName(), defender.getName()+" "+defender.getLastName()};
                    } else if (sunkenAttackingShips.size() == 0 && capturedAttackingShips.size() == 1) {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.unsuccessful.capturedOneShip";
                        args = new Object[]{capturedAttackingShips.get(0).getName(), defender.getName()+" "+defender.getLastName()};
                    } else if (sunkenAttackingShips.size() == 1 && capturedAttackingShips.size() == 1) {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.unsuccessful.sunkAndCapturedOneShip";
                        args = new Object[]{sunkenAttackingShips.get(0).getName(), capturedAttackingShips.get(0).getName(), defender.getName()+" "+defender.getLastName()};
                    } else if (sunkenAttackingShips.size() == 1 && capturedAttackingShips.size() > 1) {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.unsuccessful.sunkOneAndCapturedMultipleShip";
                        StringBuffer capturedShips = getShipNameListWithoutTheLastShip(capturedAttackingShips);
                        String lastShip = capturedAttackingShips.get(capturedAttackingShips.size()-1).getName();
                        args = new Object[]{sunkenAttackingShips.get(0).getName(), capturedShips.toString(), lastShip, defender.getName()+" "+defender.getLastName()};
                    } else if (sunkenAttackingShips.size() == 0 && capturedAttackingShips.size() > 1) {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.unsuccessful.capturedMultipleShip";
                        StringBuffer capturedShips = getShipNameListWithoutTheLastShip(capturedAttackingShips);
                        String lastShip = capturedAttackingShips.get(capturedAttackingShips.size()-1).getName();
                        args = new Object[]{capturedShips.toString(), lastShip, defender.getName()+" "+defender.getLastName()};
                    } else if (sunkenAttackingShips.size() > 1 && capturedAttackingShips.size() == 0) {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.unsuccessful.sunkMultipleShip";
                        StringBuffer sunkenShips = getShipNameListWithoutTheLastShip(sunkenAttackingShips);
                        String lastShip = sunkenAttackingShips.get(sunkenAttackingShips.size()-1).getName();
                        args = new Object[]{sunkenShips.toString(), lastShip, defender.getName()+" "+defender.getLastName()};
                    } else if (sunkenAttackingShips.size() > 1 && capturedAttackingShips.size() == 1) {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.unsuccessful.capturedOneAndSunkMultipleShip";
                        StringBuffer sunkenShips = getShipNameListWithoutTheLastShip(sunkenAttackingShips);
                        String lastShip = sunkenAttackingShips.get(sunkenAttackingShips.size()-1).getName();
                        args = new Object[]{capturedAttackingShips.get(0).getName(), sunkenShips.toString(), lastShip, defender.getName()+" "+defender.getLastName()};
                    } else {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.unsuccessful.capturedMultipleAndSunkMultipleShip";
                        StringBuffer sunkenShips = getShipNameListWithoutTheLastShip(sunkenAttackingShips);
                        String lastShipSunk = sunkenAttackingShips.get(sunkenAttackingShips.size()-1).getName();
                        StringBuffer capturedShips = getShipNameListWithoutTheLastShip(capturedAttackingShips);
                        String lastShipcaptured = capturedAttackingShips.get(capturedAttackingShips.size()-1).getName();

                        args = new Object[]{capturedShips.toString(), lastShipcaptured, sunkenShips.toString(), lastShipSunk, defender.getName()+" "+defender.getLastName()};
                    }
                    break;
                case STATE_TWO: // succesfull
                    titleKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.successfulAttackAttacker.title";
                    if (capturedDefendingShips.size() == 0 && sunkenDefendingShips.size() == 1) {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.successful.sunkOneShip";
                        args = new Object[]{defender.getName()+" "+defender.getLastName()};
                    } else if (capturedDefendingShips.size() == 1 && sunkenDefendingShips.size() == 0) {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.successful.capturedOneShip";
                        args = new Object[]{capturedDefendingShips.get(0).getName(), defender.getName()+" "+defender.getLastName()};
                    } else if (capturedDefendingShips.size() == 1 && sunkenDefendingShips.size() == 1) {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.successful.sunkAndCapturedOneShip";
                        args = new Object[]{capturedDefendingShips.get(0).getName(), defender.getName()+" "+defender.getLastName()};
                    } else if (capturedDefendingShips.size() == 0 && sunkenDefendingShips.size() > 1) {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.successful.sunkMultipleShip";
                        args = new Object[]{sunkenDefendingShips.size(), defender.getName()+" "+defender.getLastName()};
                    } else if (capturedDefendingShips.size() == 1 && sunkenDefendingShips.size() > 1) {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.successful.capturedOneAndSunkMultipleShip";
                        args = new Object[]{sunkenDefendingShips.size(), capturedDefendingShips.get(0).getName(), defender.getName()+" "+defender.getLastName()};
                    } else if (capturedDefendingShips.size() > 1 && sunkenDefendingShips.size() == 0) {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.successful.capturedMultipleShip";
                        StringBuffer capturedShips = getShipNameListWithoutTheLastShip(capturedDefendingShips);
                        String lastShipcaptured = capturedDefendingShips.get(capturedDefendingShips.size()-1).getName();
                        args = new Object[]{capturedShips.toString(), lastShipcaptured, defender.getName()+" "+defender.getLastName()};
                    } else if (capturedDefendingShips.size() > 1 && sunkenDefendingShips.size() == 1) {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.successful.sunkOneAndCapturedMultipleShip";
                        StringBuffer capturedShips = getShipNameListWithoutTheLastShip(capturedDefendingShips);
                        String lastShipcaptured = capturedDefendingShips.get(capturedDefendingShips.size()-1).getName();
                        args = new Object[]{capturedShips.toString(), lastShipcaptured, defender.getName()+" "+defender.getLastName()};
                    } else {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.successful.capturedMultipleAndSunkMultipleShip";
                        StringBuffer capturedShips = getShipNameListWithoutTheLastShip(capturedDefendingShips);
                        String lastShipcaptured = capturedDefendingShips.get(capturedDefendingShips.size()-1).getName();
                        args = new Object[]{sunkenDefendingShips.size(), capturedShips.toString(), lastShipcaptured, defender.getName()+" "+defender.getLastName()};
                    }
                    break;
                case STATE_THREE: // survived with losses
                    titleKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.successfulAttackAttackerLosses.title";
                    if (capturedDefendingShips.size() == 0 && sunkenDefendingShips.size() == 1) {
                        if (lostShips == 1) {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.sunkOneShipLostShip";
                            String lostShipName = getNameOfSingleLostShip(sunkenAttackingShips, capturedAttackingShips);
                            args = new Object[]{defender.getName()+" "+defender.getLastName(), lostShipName};
                        } else {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.sunkOneShipLostShips";
                            args = new Object[]{defender.getName()+" "+defender.getLastName(), lostShips};
                        }
                    } else if (capturedDefendingShips.size() == 1 && sunkenDefendingShips.size() == 0) {
                        if (lostShips == 1) {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.capturedOneShipLostShip";
                            String lostShipName = getNameOfSingleLostShip(sunkenAttackingShips, capturedAttackingShips);
                            args = new Object[]{capturedDefendingShips.get(0).getName(), defender.getName()+" "+defender.getLastName(), lostShipName};
                        } else {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.capturedOneShipLostShips";
                            args = new Object[]{capturedDefendingShips.get(0).getName(), defender.getName()+" "+defender.getLastName(), lostShips};
                        }
                    } else if (capturedDefendingShips.size() == 1 && sunkenDefendingShips.size() == 1) {
                        if (lostShips == 1) {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.sunkAndCapturedOneShipLostShip";
                            String lostShipName = getNameOfSingleLostShip(sunkenAttackingShips, capturedAttackingShips);
                            args = new Object[]{capturedDefendingShips.get(0).getName(), defender.getName()+" "+defender.getLastName(), lostShipName};
                        } else {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.sunkAndCapturedOneShipLostShips";
                            args = new Object[]{capturedDefendingShips.get(0).getName(), defender.getName()+" "+defender.getLastName(), lostShips};
                        }
                    } else if (capturedDefendingShips.size() == 0 && sunkenDefendingShips.size() > 1) {
                        if (lostShips == 1) {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.sunkMultipleShipLostShip";
                            String lostShipName = getNameOfSingleLostShip(sunkenAttackingShips, capturedAttackingShips);
                            args = new Object[]{sunkenDefendingShips.size(), defender.getName()+" "+defender.getLastName(), lostShipName};
                        } else {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.sunkMultipleShipLostShips";
                            args = new Object[]{sunkenDefendingShips.size(), defender.getName()+" "+defender.getLastName(), lostShips};
                        }
                    } else if (capturedDefendingShips.size() == 1 && sunkenDefendingShips.size() > 1) {
                        if (lostShips == 1) {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.capturedOneAndSunkMultipleShipLostShip";
                            String lostShipName = getNameOfSingleLostShip(sunkenAttackingShips, capturedAttackingShips);
                            args = new Object[]{sunkenDefendingShips.size(), capturedDefendingShips.get(0).getName(), defender.getName()+" "+defender.getLastName(), lostShipName};
                        } else {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.capturedOneAndSunkMultipleShipLostShips";
                            args = new Object[]{sunkenDefendingShips.size(), capturedDefendingShips.get(0).getName(), defender.getName()+" "+defender.getLastName(), lostShips};
                        }
                    } else if (capturedDefendingShips.size() > 1 && sunkenDefendingShips.size() == 0) {
                        StringBuffer capturedShips = getShipNameListWithoutTheLastShip(capturedDefendingShips);
                        String lastShipcaptured = capturedDefendingShips.get(capturedDefendingShips.size()-1).getName();
                        if (lostShips == 1) {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.capturedMultipleShipLostShip";
                            String lostShipName = getNameOfSingleLostShip(sunkenAttackingShips, capturedAttackingShips);
                            args = new Object[]{capturedShips.toString(), lastShipcaptured, defender.getName()+" "+defender.getLastName(), lostShipName};
                        } else {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.capturedMultipleShipLostShips";
                            args = new Object[]{capturedShips.toString(), lastShipcaptured, defender.getName()+" "+defender.getLastName(), lostShips};
                        }
                    } else if (capturedDefendingShips.size() > 1 && sunkenDefendingShips.size() == 1) {
                        StringBuffer capturedShips = getShipNameListWithoutTheLastShip(capturedDefendingShips);
                        String lastShipcaptured = capturedDefendingShips.get(capturedDefendingShips.size()-1).getName();
                        if (lostShips == 1) {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.sunkOneAndCapturedMultipleShipLostShip";
                            String lostShipName = getNameOfSingleLostShip(sunkenAttackingShips, capturedAttackingShips);
                            args = new Object[]{capturedShips.toString(), lastShipcaptured, defender.getName()+" "+defender.getLastName(), lostShipName};
                        } else {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.sunkOneAndCapturedMultipleShipLostShips";
                            args = new Object[]{capturedShips.toString(), lastShipcaptured, defender.getName()+" "+defender.getLastName(), lostShips};
                        }
                    } else {
                        StringBuffer capturedShips = getShipNameListWithoutTheLastShip(capturedDefendingShips);
                        String lastShipcaptured = capturedDefendingShips.get(capturedDefendingShips.size()-1).getName();
                        if (lostShips == 1) {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.capturedMultipleAndSunkMultipleShipLostShip";
                            String lostShipName = getNameOfSingleLostShip(sunkenAttackingShips, capturedAttackingShips);
                            args = new Object[]{sunkenDefendingShips.size(), capturedShips.toString(), lastShipcaptured, defender.getName()+" "+defender.getLastName(), lostShipName};
                        } else {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.capturedMultipleAndSunkMultipleShipLostShips";
                            args = new Object[]{sunkenDefendingShips.size(), capturedShips.toString(), lastShipcaptured, defender.getName()+" "+defender.getLastName(), lostShips};
                        }
                    }
                    break;
            } // end swith
            DialogTemplate template = DialogTemplate.builder()
                    .closable(true)
                    .titleKey(titleKey)
                    .messageKey(messageKey)
                    .messageArgs(args)
                    .build();
            DisplayMessage message = new DisplayMessage(titleKey, template);
            TargetedEvent tagetDisplayMsg = new TargetedEvent((IHumanPlayer)attacker, message);
            clientServerEventBus.post(tagetDisplayMsg);
        }
        if (defender instanceof IHumanPlayer) {
            String titleKey = "";
            String messageKey = "";
            Object[] args = null;
            // STATE_ONE: defeated, STATE_TWO: successful, STATE_THREE: survived
            Tristate tristate;
            int lostShips = sunkenDefendingShips.size() + capturedDefendingShips.size();
            if (lostShips == 0) {
                tristate = Tristate.STATE_TWO;
            } else if (lostShips == defendingShips.size()) {
                tristate = Tristate.STATE_ONE;
            } else {
                tristate = Tristate.STATE_THREE;
            }
            switch (tristate) {
                case STATE_ONE: // defeated
                    titleKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.successfulAttackDefender.title";
                    if (sunkenDefendingShips.size() == 1 && capturedDefendingShips.size() == 0) {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.unsuccessful.sunkOneShip";
                        args = new Object[]{sunkenDefendingShips.get(0).getName(), attacker.getName()+" "+attacker.getLastName()};
                    } else if (sunkenDefendingShips.size() == 0 && capturedDefendingShips.size() == 1) {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.unsuccessful.capturedOneShip";
                        args = new Object[]{capturedDefendingShips.get(0).getName(), attacker.getName()+" "+attacker.getLastName()};
                    } else if (sunkenDefendingShips.size() == 1 && capturedDefendingShips.size() == 1) {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.unsuccessful.sunkAndCapturedOneShip";
                        args = new Object[]{sunkenDefendingShips.get(0).getName(), capturedDefendingShips.get(0).getName(), attacker.getName()+" "+attacker.getLastName()};
                    } else if (sunkenDefendingShips.size() == 1 && capturedDefendingShips.size() > 1) {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.unsuccessful.sunkOneAndCapturedMultipleShip";
                        StringBuffer capturedShips = getShipNameListWithoutTheLastShip(capturedDefendingShips);
                        String lastShip = capturedDefendingShips.get(capturedDefendingShips.size()-1).getName();
                        args = new Object[]{sunkenDefendingShips.get(0).getName(), capturedShips.toString(), lastShip, attacker.getName()+" "+attacker.getLastName()};
                    } else if (sunkenDefendingShips.size() == 0 && capturedDefendingShips.size() > 1) {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.unsuccessful.capturedMultipleShip";
                        StringBuffer capturedShips = getShipNameListWithoutTheLastShip(capturedDefendingShips);
                        String lastShip = capturedDefendingShips.get(capturedDefendingShips.size()-1).getName();
                        args = new Object[]{capturedShips.toString(), lastShip, attacker.getName()+" "+attacker.getLastName()};
                    } else if (sunkenDefendingShips.size() > 1 && capturedDefendingShips.size() == 0) {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.unsuccessful.sunkMultipleShip";
                        StringBuffer sunkenShips = getShipNameListWithoutTheLastShip(sunkenDefendingShips);
                        String lastShip = sunkenDefendingShips.get(sunkenDefendingShips.size()-1).getName();
                        args = new Object[]{sunkenShips.toString(), lastShip, attacker.getName()+" "+attacker.getLastName()};
                    } else if (sunkenDefendingShips.size() > 1 && capturedDefendingShips.size() == 1) {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.unsuccessful.capturedOneAndSunkMultipleShip";
                        StringBuffer sunkenShips = getShipNameListWithoutTheLastShip(sunkenDefendingShips);
                        String lastShip = sunkenDefendingShips.get(sunkenDefendingShips.size()-1).getName();
                        args = new Object[]{capturedDefendingShips.get(0).getName(), sunkenShips.toString(), lastShip, attacker.getName()+" "+attacker.getLastName()};
                    } else {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.unsuccessful.capturedMultipleAndSunkMultipleShip";
                        StringBuffer sunkenShips = getShipNameListWithoutTheLastShip(sunkenDefendingShips);
                        String lastShipSunk = sunkenDefendingShips.get(sunkenDefendingShips.size()-1).getName();
                        StringBuffer capturedShips = getShipNameListWithoutTheLastShip(capturedDefendingShips);
                        String lastShipcaptured = capturedDefendingShips.get(capturedDefendingShips.size()-1).getName();

                        args = new Object[]{capturedShips.toString(), lastShipcaptured, sunkenShips.toString(), lastShipSunk, attacker.getName()+" "+attacker.getLastName()};
                    }
                    break;
                case STATE_TWO: // successful
                    titleKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.unsuccessfulAttackDefender.title";
                    if (capturedAttackingShips.size() == 0 && sunkenAttackingShips.size() == 1) {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.successful.sunkOneShip";
                        args = new Object[]{attacker.getName()+" "+attacker.getLastName()};
                    } else if (capturedAttackingShips.size() == 1 && sunkenAttackingShips.size() == 0) {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.successful.capturedOneShip";
                        args = new Object[]{capturedAttackingShips.get(0).getName(), attacker.getName()+" "+attacker.getLastName()};
                    } else if (capturedAttackingShips.size() == 1 && sunkenAttackingShips.size() == 1) {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.successful.sunkAndCapturedOneShip";
                        args = new Object[]{capturedAttackingShips.get(0).getName(), attacker.getName()+" "+attacker.getLastName()};
                    } else if (capturedAttackingShips.size() == 0 && sunkenAttackingShips.size() > 1) {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.successful.sunkMultipleShip";
                        args = new Object[]{sunkenAttackingShips.size(), attacker.getName()+" "+attacker.getLastName()};
                    } else if (capturedAttackingShips.size() == 1 && sunkenAttackingShips.size() > 1) {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.successful.capturedOneAndSunkMultipleShip";
                        args = new Object[]{sunkenAttackingShips.size(), capturedAttackingShips.get(0).getName(), attacker.getName()+" "+attacker.getLastName()};
                    } else if (capturedAttackingShips.size() > 1 && sunkenAttackingShips.size() == 0) {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.successful.capturedMultipleShip";
                        StringBuffer capturedShips = getShipNameListWithoutTheLastShip(capturedAttackingShips);
                        String lastShipcaptured = capturedAttackingShips.get(capturedAttackingShips.size()-1).getName();
                        args = new Object[]{capturedShips.toString(), lastShipcaptured, attacker.getName()+" "+attacker.getLastName()};
                    } else if (capturedAttackingShips.size() > 1 && sunkenAttackingShips.size() == 1) {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.successful.sunkOneAndCapturedMultipleShip";
                        StringBuffer capturedShips = getShipNameListWithoutTheLastShip(capturedAttackingShips);
                        String lastShipcaptured = capturedAttackingShips.get(capturedAttackingShips.size()-1).getName();
                        args = new Object[]{capturedShips.toString(), lastShipcaptured, attacker.getName()+" "+attacker.getLastName()};
                    } else {
                        messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.successful.capturedMultipleAndSunkMultipleShip";
                        StringBuffer capturedShips = getShipNameListWithoutTheLastShip(capturedAttackingShips);
                        String lastShipcaptured = capturedAttackingShips.get(capturedAttackingShips.size()-1).getName();
                        args = new Object[]{sunkenAttackingShips.size(), capturedShips.toString(), lastShipcaptured, attacker.getName()+" "+attacker.getLastName()};
                    }
                    break;
                case STATE_THREE: // survived with losses
                    titleKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.unsuccessfulAttackDefenderLosses.title";
                    if (capturedAttackingShips.size() == 0 && sunkenAttackingShips.size() == 1) {
                        if (lostShips == 1) {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.sunkOneShipLostShip";
                            String lostShipName = getNameOfSingleLostShip(sunkenDefendingShips, capturedDefendingShips);
                            args = new Object[]{attacker.getName()+" "+attacker.getLastName(), lostShipName};
                        } else {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.sunkOneShipLostShips";
                            args = new Object[]{attacker.getName()+" "+attacker.getLastName(), lostShips};
                        }
                    } else if (capturedAttackingShips.size() == 1 && sunkenAttackingShips.size() == 0) {
                        if (lostShips == 1) {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.capturedOneShipLostShip";
                            String lostShipName = getNameOfSingleLostShip(sunkenDefendingShips, capturedDefendingShips);
                            args = new Object[]{capturedAttackingShips.get(0).getName(), attacker.getName()+" "+attacker.getLastName(), lostShipName};
                        } else {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.capturedOneShipLostShips";
                            args = new Object[]{capturedAttackingShips.get(0).getName(), attacker.getName()+" "+attacker.getLastName(), lostShips};
                        }
                    } else if (capturedAttackingShips.size() == 1 && sunkenAttackingShips.size() == 1) {
                        if (lostShips == 1) {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.sunkAndCapturedOneShipLostShip";
                            String lostShipName = getNameOfSingleLostShip(sunkenDefendingShips, capturedDefendingShips);
                            args = new Object[]{capturedAttackingShips.get(0).getName(), attacker.getName()+" "+attacker.getLastName(), lostShipName};
                        } else {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.sunkAndCapturedOneShipLostShips";
                            args = new Object[]{capturedAttackingShips.get(0).getName(), attacker.getName()+" "+attacker.getLastName(), lostShips};
                        }
                    } else if (capturedAttackingShips.size() == 0 && sunkenAttackingShips.size() > 1) {
                        if (lostShips == 1) {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.sunkMultipleShipLostShip";
                            String lostShipName = getNameOfSingleLostShip(sunkenDefendingShips, capturedDefendingShips);
                            args = new Object[]{sunkenAttackingShips.size(), attacker.getName()+" "+attacker.getLastName(), lostShipName};
                        } else {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.sunkMultipleShipLostShips";
                            args = new Object[]{sunkenAttackingShips.size(), attacker.getName()+" "+attacker.getLastName(), lostShips};
                        }
                    } else if (capturedAttackingShips.size() == 1 && sunkenAttackingShips.size() > 1) {
                        if (lostShips == 1) {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.capturedOneAndSunkMultipleShipLostShip";
                            String lostShipName = getNameOfSingleLostShip(sunkenDefendingShips, capturedDefendingShips);
                            args = new Object[]{sunkenAttackingShips.size(), capturedAttackingShips.get(0).getName(), attacker.getName()+" "+attacker.getLastName(), lostShipName};
                        } else {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.capturedOneAndSunkMultipleShipLostShips";
                            args = new Object[]{sunkenAttackingShips.size(), capturedAttackingShips.get(0).getName(), attacker.getName()+" "+attacker.getLastName(), lostShips};
                        }
                    } else if (capturedAttackingShips.size() > 1 && sunkenAttackingShips.size() == 0) {
                        StringBuffer capturedShips = getShipNameListWithoutTheLastShip(capturedAttackingShips);
                        String lastShipcaptured = capturedAttackingShips.get(capturedAttackingShips.size()-1).getName();
                        if (lostShips == 1) {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.capturedMultipleShipLostShip";
                            String lostShipName = getNameOfSingleLostShip(sunkenDefendingShips, capturedDefendingShips);
                            args = new Object[]{capturedShips.toString(), lastShipcaptured, attacker.getName()+" "+attacker.getLastName(), lostShipName};
                        } else {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.capturedMultipleShipLostShips";
                            args = new Object[]{capturedShips.toString(), lastShipcaptured, attacker.getName()+" "+attacker.getLastName(), lostShips};
                        }
                    } else if (capturedAttackingShips.size() > 1 && sunkenAttackingShips.size() == 1) {
                        StringBuffer capturedShips = getShipNameListWithoutTheLastShip(capturedAttackingShips);
                        String lastShipcaptured = capturedAttackingShips.get(capturedAttackingShips.size()-1).getName();
                        if (lostShips == 1) {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.sunkOneAndCapturedMultipleShipLostShip";
                            String lostShipName = getNameOfSingleLostShip(sunkenDefendingShips, capturedDefendingShips);
                            args = new Object[]{capturedShips.toString(), lastShipcaptured, attacker.getName()+" "+attacker.getLastName(), lostShipName};
                        } else {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.sunkOneAndCapturedMultipleShipLostShips";
                            args = new Object[]{capturedShips.toString(), lastShipcaptured, attacker.getName()+" "+attacker.getLastName(), lostShips};
                        }
                    } else {
                        StringBuffer capturedShips = getShipNameListWithoutTheLastShip(capturedAttackingShips);
                        String lastShipcaptured = capturedAttackingShips.get(capturedAttackingShips.size()-1).getName();
                        if (lostShips == 1) {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.capturedMultipleAndSunkMultipleShipLostShip";
                            String lostShipName = getNameOfSingleLostShip(sunkenDefendingShips, capturedDefendingShips);
                            args = new Object[]{sunkenAttackingShips.size(), capturedShips.toString(), lastShipcaptured, attacker.getName()+" "+attacker.getLastName(), lostShipName};
                        } else {
                            messageKey = "ch.sahits.game.openpatrician.engine.sea.SeaFightService.partialsuccessful.capturedMultipleAndSunkMultipleShipLostShips";
                            args = new Object[]{sunkenAttackingShips.size(), capturedShips.toString(), lastShipcaptured, attacker.getName()+" "+attacker.getLastName(), lostShips};
                        }
                    }
                    break;
            } // end switch
            DialogTemplate template = DialogTemplate.builder()
                    .closable(true)
                    .titleKey(titleKey)
                    .messageKey(messageKey)
                    .messageArgs(args)
                    .build();
            DisplayMessage message = new DisplayMessage(titleKey, template);
            TargetedEvent tagetDisplayMsg = new TargetedEvent((IHumanPlayer)defender, message);
            clientServerEventBus.post(tagetDisplayMsg);
        }
    }
    @VisibleForTesting
    StringBuffer getShipNameListWithoutTheLastShip(List<IShip> shipList) {
        StringBuffer sunkenShips = new StringBuffer();
        for (int i = 0; i < shipList.size() - 1; i++) {
            IShip ship = shipList.get(i);
            sunkenShips.append(ship.getName());
            if (i < shipList.size() - 2) {
                sunkenShips.append(", ");
            }
        }
        return sunkenShips;
    }

    private String getNameOfSingleLostShip(List<IShip> sunkenAttackingShips, List<IShip> capturedAttackingShips) {
        return capturedAttackingShips.size() == 1 ? capturedAttackingShips.get(0).getName() : sunkenAttackingShips.get(0).getName();
    }

    private void runAttackMove(List<GenericPair<List<IShip>, List<IShip>>> pairings,
                               List<IShip> sunkenDefendingShips, List<IShip> sunkenAttackingShips,
                               List<IShip> capturedDefendingShips, List<IShip> capturedAttackingShips,
                               Iterator<GenericPair<List<IShip>, List<IShip>>> pairingIter, GenericPair<List<IShip>,
            List<IShip>> pairing, boolean attackShipTurn) {
        List<IShip> shipsOnTurn;
        List<IShip> shipsNotOnTurn;
        if (attackShipTurn) {
            shipsOnTurn = pairing.getFirst();
            shipsNotOnTurn = pairing.getSecond();
        } else {
            shipsOnTurn = pairing.getSecond();
            shipsNotOnTurn = pairing.getFirst();
        }

        int strength = calculateStrength(shipsOnTurn);
        int agilityOnTurn = calculateAgility(shipsOnTurn);
        int agilityNotOnTurn = calculateAgility(shipsNotOnTurn);
        double evasionProbability = calculateEvasionProbability(agilityOnTurn, agilityNotOnTurn);
        double accuracy = calculateAccuracy(shipsOnTurn);

        // calculate damage
        int damage = calculateDamage(strength, accuracy, evasionProbability);
        int damagePerShip = damage/shipsNotOnTurn.size();
        int damageFirstShip = damagePerShip + damage%shipsNotOnTurn.size();
        shipsNotOnTurn.get(0).damage(damageFirstShip);
        for (int i = 1; i < shipsNotOnTurn.size(); i++) {
            shipsNotOnTurn.get(i).damage(damagePerShip);
        }
        // check if ships were destroyed:
        for (Iterator<IShip> iterator = shipsNotOnTurn.iterator(); iterator.hasNext(); ) {
            IShip ship = iterator.next();
            if (ship.getDamage() == 0) {
                if (attackShipTurn) {
                    sunkenDefendingShips.add(ship);
                } else {
                    sunkenAttackingShips.add(ship);
                }
                iterator.remove();
                removeFromFleet(ship, ship.getOwner());
            }
        }
        if (shipsNotOnTurn.isEmpty()) {
            // distribute attacking ships on other pairings
            pairingIter.remove();
            if (pairings.size() > 0) {
                redistributeShips(pairings, attackShipTurn, shipsOnTurn);
            }
        } else { // this pair are still facing each other
            // calculate death toll
            int deathToll = calculateDeathToll(shipsNotOnTurn, damage);
            int deathTollPerShip = deathToll/shipsNotOnTurn.size();
            int deathTollFirstShip = deathTollPerShip + deathToll%shipsNotOnTurn.size();
            shipsNotOnTurn.get(0).setNumberOfSailors(shipsNotOnTurn.get(0).getNumberOfSailors() - deathTollFirstShip);
            for (int i = 1; i < shipsNotOnTurn.size(); i++) {
                shipsNotOnTurn.get(i).setNumberOfSailors(shipsNotOnTurn.get(i).getNumberOfSailors() - deathTollPerShip);
            }
            // decide on capturing
            IShip shipWithMaxSailors = findMaxSailorsOnShip(shipsOnTurn);
            IShip captureShip = null;
            for (IShip ship : shipsNotOnTurn) {
              if (ship.getNumberOfSailors() + 5 < shipWithMaxSailors.getNumberOfSailors()) {
                  captureShip = ship;
                  break;
              }
            }
            updateCapturedShip(capturedDefendingShips, capturedAttackingShips, attackShipTurn, shipWithMaxSailors, captureShip);
            // check if pairing has one side empty
            if (shipsNotOnTurn.isEmpty()) {
                pairingIter.remove();
                redistributeShips(pairings, attackShipTurn, shipsOnTurn);
            } else if (shipsOnTurn.isEmpty()) {
                pairingIter.remove();
                redistributeShips(pairings, !attackShipTurn, shipsNotOnTurn);
            }
        }

    }

    private void redistributeShips(List<GenericPair<List<IShip>, List<IShip>>> pairings, boolean attackShipTurn, List<IShip> survivingShips) {
        for (IShip ship : survivingShips) {
            GenericPair<List<IShip>, List<IShip>> weakestPairing = findWeakest(pairings, attackShipTurn);
            if (attackShipTurn) {
                weakestPairing.getFirst().add(ship);
            } else {
                weakestPairing.getSecond().add(ship);
            }
        }
    }

    @VisibleForTesting
    void updateCapturedShip(List<IShip> capturedDefendingShips, List<IShip> capturedAttackingShips, boolean attackShipTurn, IShip shipWithMaxSailors, IShip captureShip) {
        if (captureShip != null) {
            int calculateCaptureSailors = captureShip(shipWithMaxSailors, captureShip);
            if (calculateCaptureSailors < 0) { // the captureShip defeated shipWithMaxSailors
                final int numberOfSailors = captureShip.getNumberOfSailors();
                int halfTheSailors = numberOfSailors /2;
                captureShip.setNumberOfSailors(numberOfSailors - halfTheSailors);
                IShipOwner previousOwner = shipWithMaxSailors.getOwner();
                removeFromFleet(shipWithMaxSailors, previousOwner);
                shipWithMaxSailors.setNumberOfSailors(halfTheSailors);
                shipWithMaxSailors.setOwner(captureShip.getOwner());
                addToFleet(shipWithMaxSailors, captureShip.getOwner());
                if (attackShipTurn) { // shipWithMaxSailors belongs to attacking fleet
                   capturedAttackingShips.add(shipWithMaxSailors);
                } else {
                    capturedDefendingShips.add(shipWithMaxSailors);
                }
            } else { // the shipWithMaxSailors defeated captureShip
                final int numberOfSailors = shipWithMaxSailors.getNumberOfSailors();
                int halfTheSailors = numberOfSailors /2;
                shipWithMaxSailors.setNumberOfSailors(numberOfSailors - halfTheSailors);
                IShipOwner previousOwner = captureShip.getOwner();
                removeFromFleet(captureShip, previousOwner);
                captureShip.setNumberOfSailors(halfTheSailors);
                captureShip.setOwner(shipWithMaxSailors.getOwner());
                addToFleet(captureShip, shipWithMaxSailors.getOwner());
                if (attackShipTurn) { // shipWithMaxSailors belongs to attacking fleet
                    capturedDefendingShips.add(captureShip);
                } else {
                    capturedAttackingShips.add(captureShip);
                }
            }
        }
    }

    private void addToFleet(IShip ship, IShipOwner owner) {
        if (owner instanceof IPlayer) {
            ((IPlayer)owner).getFleet().add(ship);
        }
        // TODO: andi 12/9/15 implement the case where the owner is a pirate
    }

    private void removeFromFleet(IShip ship, IShipOwner owner) {
        if (owner instanceof IPlayer) {
            ((IPlayer)owner).getFleet().remove(ship); 
        }
        // TODO: andi 12/9/15 implement the case where the owner is a pirate 
    }

    @VisibleForTesting
    int captureShip(IShip attackingShip, IShip defendingShip) {
        int attackingHandWeapon = attackingShip.getWeaponAmount(EWeapon.HAND_WEAPON);
        int defendingHandWeapon = defendingShip.getWeaponAmount(EWeapon.HAND_WEAPON);
        while (attackingShip.getNumberOfSailors() > 0 && defendingShip.getNumberOfSailors() > 0) {
            int attackForce = attackingShip.getNumberOfSailors() + attackingHandWeapon;
            int defendForce = defendingShip.getNumberOfSailors() + defendingHandWeapon;
            double propSurvivingAttack = Math.min(Math.abs(rnd.nextGaussian()), 1);
            if (propSurvivingAttack > 0.8) {
              // weaker force kills
                if (attackForce > defendForce) {
                    final int nbSailors = attackingShip.getNumberOfSailors() - 1;
                    attackingShip.setNumberOfSailors(nbSailors);
                } else {
                    final int nbSailors = defendingShip.getNumberOfSailors() - 1;
                    defendingShip.setNumberOfSailors(nbSailors);
                }
            } else if (propSurvivingAttack > 0.5) {
                // no kill
                continue;
            } else {
                // stronger force kills
                if (attackForce > defendForce) {
                    final int nbSailors = defendingShip.getNumberOfSailors() - 1;
                    defendingShip.setNumberOfSailors(nbSailors);
                } else {
                    final int nbSailors = attackingShip.getNumberOfSailors() - 1;
                    attackingShip.setNumberOfSailors(nbSailors);
                }
            }
        }
        if (attackingShip.getNumberOfSailors() > 0) {
            return attackingShip.getNumberOfSailors();
        } else {
            return -defendingShip.getNumberOfSailors();
        }
    }

    @VisibleForTesting
    IShip findMaxSailorsOnShip(List<IShip> ships) {
      int maxSailors = ships.get(0).getNumberOfSailors();
        IShip maxSailorShip = ships.get(0);
        for (int i = 1; i < ships.size(); i++) {
           if (ships.get(i).getNumberOfSailors() > maxSailors) {
               maxSailorShip = ships.get(i);
               maxSailors = ships.get(i).getNumberOfSailors();
           }
        }
        return maxSailorShip;
    }

    @VisibleForTesting
    int calculateDeathToll(List<IShip> shipsNotOnTurn, int damage) {
        double nbDeath = damage*DEATH_TOLL_PER_DAMAGE*(rnd.nextGaussian()+1);
        int totalSailors = 0;
        for (IShip ship : shipsNotOnTurn) {
            totalSailors += ship.getNumberOfSailors();
        }
        return (int)Math.rint(Math.min(totalSailors, nbDeath));
    }

    @VisibleForTesting
    int calculateDamage(int strength, double accuracy, double evasionProbability) {
        int possibleDamage = strength * DAMAGE_PER_STRENGTH;
        double damage = possibleDamage * accuracy;
        double evasion = rnd.nextDouble();
        if (evasion < evasionProbability) {
            return 0;
        } else {
            return (int)Math.rint(damage*evasionProbability*1.5);
        }
    }

    @VisibleForTesting
    double calculateAccuracy(List<IShip> shipsOnTurn) {
        double accuracy = 0.7;
        double captainAccuracy = 0;
        int nbCaptains = 0;
        for (IShip ship : shipsOnTurn) {
            if (ship.getCaptian().isPresent()) {
                nbCaptains++;
                captainAccuracy += ship.getCaptian().get().getFightSkillLevel();
            }
        }
        if (nbCaptains > 0) {
            captainAccuracy = captainAccuracy / (nbCaptains*10.0);
        }
        return accuracy + captainAccuracy;
    }

    @VisibleForTesting
    double calculateEvasionProbability(int attackingAgility, int defendingAgility) {
        if (defendingAgility < attackingAgility) {
            return 0;
        }
        if (defendingAgility == attackingAgility) {
            return 0.2;
        }
        return  (defendingAgility - attackingAgility)/12.0 + 0.2;
    }

    private int calculateAgility(List<IShip> ships) {
        int agility = 0;
        for (IShip ship : ships) {
            agility += calculateAgility(ship);
        }
        return agility;
    }
    @VisibleForTesting
    int calculateAgility(IShip ship) {
        int agility = shipFactory.getShipSpeed(ship.getShipType());
        if (ship.getCaptian().isPresent()) {
            ICaptain captain = ship.getCaptian().get();
            agility += (int)Math.rint(captain.getNavigationSkillLevel() * 0.7);
        }
        return agility;
    }

    @VisibleForTesting
    List<GenericPair<List<IShip>, List<IShip>>> calculatePairing(List<IShip> attackingShips, List<IShip> defendingShips){
        List<GenericPair<List<IShip>, List<IShip>>> pairings = new ArrayList<>();
        if (attackingShips.size() == defendingShips.size()){
            for (int i = 0; i < attackingShips.size(); i++) {
                addPairing(attackingShips, defendingShips, pairings, i);
            }
            return pairings;
        }
        // pair off one on one and distribute the rest on the weakest pairing
        int minSize = Math.min(attackingShips.size(), defendingShips.size());
        for (int i = 0; i < minSize; i++) {
            addPairing(attackingShips, defendingShips, pairings, i);
        }
        if (attackingShips.size() < defendingShips.size()) {
            for (int i = minSize; i < defendingShips.size(); i++) {
                GenericPair<List<IShip>, List<IShip>> weakestDefendingPair = findWeakest(pairings, false);
                weakestDefendingPair.getSecond().add(defendingShips.get(i));
            }
        } else {
            for (int i = minSize; i < attackingShips.size(); i++) {
                GenericPair<List<IShip>, List<IShip>> weakestDefendingPair = findWeakest(pairings, true);
                weakestDefendingPair.getFirst().add(attackingShips.get(i));
            }
        }

        return pairings;
    }
    @VisibleForTesting
    GenericPair<List<IShip>, List<IShip>> findWeakest(List<GenericPair<List<IShip>, List<IShip>>> parings, boolean lookInAttackingShips) {
       int minStrength = Integer.MAX_VALUE;
       int minStrengthIndex = 0;
       for (int i = 0; i < parings.size(); i++) {
           GenericPair<List<IShip>, List<IShip>> paring = parings.get(i);
           List<IShip> ships;
           if (lookInAttackingShips) { // look at first
               ships = paring.getFirst();
           } else {
               ships = paring.getSecond();
           }
           int strength = 0;
           for (IShip ship : ships) {
               strength += calculateShipStrength(ship);
           }
           if (strength < minStrength) {
               minStrength = strength;
               minStrengthIndex = i;
           }
       }
       return parings.get(minStrengthIndex);
    }
    private int calculateStrength(List<IShip> ships) {
        int strength = 0;
        for (IShip ship : ships) {
            strength += calculateShipStrength(ship);
        }
        return strength;
    }

    @VisibleForTesting
    int calculateShipStrength(IShip ship) {
        int strength = shipStrenghtService.calculateShipsWeaponsStrength(ship);
        if (ship.getCaptian().isPresent()) {
            ICaptain captain = ship.getCaptian().get();
            strength += captain.getFightSkillLevel()*2;
        }
        return strength;
    }

    private void addPairing(List<IShip> attackingShips, List<IShip> defendingShips, List<GenericPair<List<IShip>, List<IShip>>> pairings, int i) {
        List<IShip> attackingShip = new ArrayList<>();
        attackingShip.add(attackingShips.get(i));
        List<IShip> defending = new ArrayList<>();
        defending.add(defendingShips.get(i));
        GenericPair<List<IShip>, List<IShip>> pair = new GenericPair<>(attackingShip, defending);
        pairings.add(pair);
    }

    /**
     * Create a list of ships from a {@link IConvoy}.
     * @param convoy representing a number of ships
     * @return List containg all ships of that group.
     */
    public List<IShip> explodeShipList(IConvoy convoy) {
        // TODO: andi 12/8/15 implement
        return null;
    }

    /**
     * Create a list of ships from a group of ships.
     * @param group of ships
     * @return List containing all ships of that group.
     */
    public List<IShip> explodeShipList(IShipGroup group) {
        // TODO: andi 12/8/15 implement
        return null;
    }

    /**
     * Create a list of ships from a single ship that is neither a {@link IConvoy}
     * or {@link IShipGroup}.
     * @param ship single ship
     * @return List containing the one ship
     */
    public List<IShip> explodeShipList(IShip ship) {
        ArrayList<IShip> ships = new ArrayList<>();
        ships.add(ship);
        return ships;
    }

    /**
     * Create a group of ships from the list of ships. As there might have been
     * a group before the fight, this group should be updated with the current
     * ship list. If there was no group, a new group will be created.
     * @param ships List of ships
     * @param oldGroup previous group of ships to be updated, may be null.
     * @return Updated <code>oldGroup</code> or a new instance containing the ships of the list
     */
    public IShipGroup regroup(List<IShip> ships, IShipGroup oldGroup) {
        // TODO: andi 12/8/15 implement
        return null;
    }

    /**
     * Update the convoy with the ships of the list.
     * @param ships List of ships
     * @param oldConvoy previous convoy that is to be updated.
     * @return Updated <code>oldConvoy</code>.
     */
    public IConvoy regroup(List<IShip> ships, IConvoy oldConvoy) {
        // TODO: andi 12/8/15 implement
        return null;
    }
}
