package ch.sahits.game.openpatrician.engine.sea;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.DependentInitialisation;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.engine.sea.model.ShipPositionUpdateTask;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.spring.EngineConfiguration;
import ch.sahits.game.openpatrician.util.StartNewGameBean;
import javafx.geometry.Point2D;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * Service keeping track of all the ships that are traveling.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Jan 04, 2016
 */
@ClassCategory(EClassCategory.SINGLETON_BEAN)
@Component
@Lazy
@DependentInitialisation(StartNewGameBean.class)
public class SeafaringService {
    private final Logger logger = LogManager.getLogger(getClass());
    @Autowired
    private AStar aStar;
    @Autowired
    @Qualifier("clientThreadPool")
    private ExecutorService clientThreadPool;
    @Autowired
    private ShipPositionUpdateTask shipUpdateTask;

    @Autowired
    @Qualifier("schedulableServerThreadPool")
    private ScheduledExecutorService schedulableServerThreadPool;
    @PostConstruct
    private void initializeTimer() {
        schedulableServerThreadPool.scheduleAtFixedRate(shipUpdateTask, 0, EngineConfiguration.CLOCK_TICK_INTERVALL_MS, TimeUnit.MILLISECONDS);
    }
    /**
     * Send a ship or convoy on its way to a destination.
     * @param vessel that is sent
     * @param destination of the travel
     * @return list of points defining the path.
     */
    public List<Point2D> travelTo(INavigableVessel vessel, Point2D destination) {
        Point2D source = vessel.getLocation();
        Future<List<Point2D>> future = clientThreadPool.submit(() -> aStar.findPath(source, destination));
        try {
            return future.get();
        } catch (InterruptedException|ExecutionException e) {
            logger.warn("Failed to execute path finding in separate thread.", e);
            return aStar.findPath(source, destination);
        }
    }
}
