package ch.sahits.game.openpatrician.engine.sea.model;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.MapType;
import javafx.geometry.Point2D;
import lombok.Getter;

import java.util.HashMap;
import java.util.Map;

/**
 * NodeData stores all information of the node needed by the AStar algorithm.
 * This information includes the value of g, h, and f. However, the value of all
 * 3 variables are dependent on source and destination, thus obtains at runtime.
 *
 * @param <T>
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Dec 31, 2015
 */
@ClassCategory(EClassCategory.MODEL)
public final class NodeData<T> {
    @Getter
    private final T nodeId;
    @MapType(key = Point2D.class, value = Double.class)
    private final Map<T, Double> heuristic;
    /**
     * Map the sistance of the node to the source. the source node is the key.
     */
    @MapType(key = Point2D.class, value = Double.class)
    private Map<T, Double> distanceFromSource = new HashMap<>();
    @Getter
    private double h;  // h is the heuristic of destination.
    @Getter
    private double f;  // f = g + h

    public NodeData (T nodeId, Map<T, Double> heuristic) {
        this.nodeId = nodeId;
        this.heuristic = heuristic;
    }

    public void setG(T source, double g) {
        distanceFromSource.put(source, g);
    }
    public double getG(T source) {
        if (distanceFromSource.containsKey(source)) {
            return distanceFromSource.get(source);
        } else {
            return Double.MAX_VALUE;
        }
    }


    public void calcF(T destination, T source) {
        this.h = heuristic.get(destination);
        this.f = getG(source) + h;
    }

}
