package ch.sahits.game.openpatrician.engine.land.city;

import ch.sahits.game.event.data.PeriodicalDailyUpdate;
import ch.sahits.game.event.data.PeriodicalTimeWeekUpdate;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.LazySingleton;
import ch.sahits.game.openpatrician.dialog.ButtonTemplate;
import ch.sahits.game.openpatrician.dialog.DialogTemplate;
import ch.sahits.game.openpatrician.engine.AbstractEngine;
import ch.sahits.game.openpatrician.model.*;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.ICreditor;
import ch.sahits.game.openpatrician.model.city.ILoaner;
import ch.sahits.game.openpatrician.model.city.LoanerList;
import ch.sahits.game.openpatrician.model.city.impl.IDebt;
import ch.sahits.game.openpatrician.model.event.TargetedEvent;
import ch.sahits.game.openpatrician.clientserverinterface.model.factory.StateFactory;
import ch.sahits.game.openpatrician.model.personal.ESocialRank;
import ch.sahits.game.openpatrician.model.personal.IReputation;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Random;

/**
 * Engine for controlling all the loaners.
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Dec 30, 2014
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class LoanerEngine extends AbstractEngine {
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    private StateFactory factory;
    @Autowired
    private Date date;
    @Autowired
    private Random rnd;
    @Autowired
    private Locale locale;
    @Autowired
    private MessageSource messageSource;
    @Autowired
    private LoanerList loaners;
    @Autowired
    private StateFactory stateFactory;

    @PostConstruct
    private void init() {
        clientServerEventBus.register(this);
    }
    @PreDestroy
    private void unregister() {
        clientServerEventBus.unregister(this);
    }
    @Subscribe
    public void handleWeeklyUpdate(PeriodicalTimeWeekUpdate event) {
        for (ILoaner loaner : loaners) {
            stateFactory.updateLoanerRequestsAndLoans(loaner);
        }
    }

    @Subscribe
    public void handleDailyUpdate(PeriodicalDailyUpdate event) {
        DateTime now = date.getCurrentDate();
        for (ILoaner loaner : loaners) {
            List<IDebt> debts = loaner.getDebts();
            for (Iterator<IDebt> iterator = debts.iterator(); iterator.hasNext(); ) {
                IDebt debt = iterator.next();
                if (debt.getDueDate().isBefore(now)) {
                    ICitizen debitor = debt.getDebitor();
                    int cash = (int) (debt.getAmount() * debt.getInterest());
                    if (debitor instanceof IPlayer) {
                        IPlayer player = (IPlayer) debitor;
                        ICompany company = player.getCompany();
                        if (company.getCash() > cash) {
                            company.updateCash(-cash);
                            if (debt.getCreditor() instanceof IPlayer) {
                                ((IPlayer)debt.getCreditor()).getCompany().updateCash(cash);
                            }
                        } else {
                            postMessageDebtNotFulfillable(debt);
                        }
                    } else {  // Debitor is not a player
                        if (debitor.getRank().equals(ESocialRank.BARGAINER) ||
                                debitor.getRank().equals(ESocialRank.CHANDLER) ||
                                debitor.getRank().equals(ESocialRank.MERCHANT) ||
                                debitor.getRank().equals(ESocialRank.TRADESMAN) ) {
                            int payBack = rnd.nextInt(4);
                            if (payBack == 0) {
                                postMessageDebtNotFulfillable(debt);
                            } else {
                                if (debt.getCreditor() instanceof IPlayer) {
                                    ((IPlayer)debt.getCreditor()).getCompany().updateCash(cash);
                                }
                            }
                        } else {
                            // Can always pay back
                            if (debt.getCreditor() instanceof IPlayer) {
                                ((IPlayer)debt.getCreditor()).getCompany().updateCash(cash);
                            }
                        }
                    }
                    iterator.remove();
                }
            }
        }
    }
    @VisibleForTesting
    void postMessageDebtNotFulfillable(final IDebt debt) {
        String titleTemplate = "ch.sahits.game.openpatrician.engine.land.city.LoanerEngine.paybackTitle";
        String messageTemplate = "ch.sahits.game.openpatrician.engine.land.city.LoanerEngine.paybackMessage";
        String remitBtnLbl = "ch.sahits.game.openpatrician.engine.land.city.LoanerEngine.remitBtn";
        String pawnBtnLbl = "ch.sahits.game.openpatrician.engine.land.city.LoanerEngine.pawnBtn";
        int sum = (int) (debt.getAmount() * debt.getInterest());
        Object[] messageArgs = new Object[] {debt.getDebitor().getName(), debt.getDebitor().getLastName(), sum};
        Runnable remit = createRemitRunnable(debt);
        Runnable pawn = createPawnRunnable(debt);
        List<ButtonTemplate> buttons = new ArrayList<>();
        ButtonTemplate remitBtn = ButtonTemplate.builder()
                .action(remit)
                .labelKey(remitBtnLbl)
                .largeButton(true)
                .build();
        ButtonTemplate pawnBtn = ButtonTemplate.builder()
                .action(pawn)
                .labelKey(pawnBtnLbl)
                .largeButton(true)
                .build();
        buttons.add(remitBtn);
        buttons.add(pawnBtn);
        DialogTemplate dialogTemplate = DialogTemplate.builder()
                .titleKey(titleTemplate)
                .messageKey(messageTemplate)
                .messageArgs(messageArgs)
                .buttons(buttons)
                .closable(false)
                .build();
        DisplayMessage displayMessage = new DisplayMessage(titleTemplate);
        displayMessage.setDialogTemplate(dialogTemplate);
        if (debt.getDebitor() instanceof IHumanPlayer) {
            TargetedEvent message = new TargetedEvent((IHumanPlayer) debt.getDebitor(), displayMessage);
            clientServerEventBus.post(message);
        }
    }
    @VisibleForTesting
    Runnable createRemitRunnable(final IDebt debt) {
        return new Runnable() {
            @Override
            public void run() {
                ICreditor creditor = debt.getCreditor();
                if (creditor instanceof IPlayer) {
                    ICity city = debt.getDebitor().getHometown();
                    IReputation reputation = city.getReputation((IPlayer) creditor);
                    reputation.update(500);
                }
            }
        };
    }
    @VisibleForTesting
    Runnable createPawnRunnable(final IDebt debt) {
        return new Runnable() {
            @Override
            public void run() {
                if (debt.getDebitor() instanceof IHumanPlayer) {
                    // pawn a ship and auction it
                    int sum = (int) (debt.getAmount() * debt.getInterest());
                    IPlayer player = (IPlayer) debt.getDebitor();
                    IShip shipMatch = null;
                    int maxDelta = Integer.MAX_VALUE;
                    for (IShip ship : player.getFleet()) {
                       int shipValue = ship.getValue();
                        if  (Math.abs(shipValue - sum) < maxDelta) {
                            maxDelta = Math.abs(shipValue - sum);
                            shipMatch = ship;
                        }
                    }
                    // todo: andi 1/2/15: auction the ship: Ticket: 146
                    // Auction ship postponed
                } else if (debt.getCreditor() instanceof IHumanPlayer){
                    int value = rnd.nextInt(debt.getAmount()/2) + debt.getAmount();
                    value = (int) Math.min(value, debt.getAmount()*debt.getInterest());
                    ((IPlayer)debt.getCreditor()).getCompany().updateCash(value);
                    ICitizen debitor = debt.getDebitor();
                    DisplayMessage msg = new DisplayMessage("ch.sahits.game.openpatrician.engine.land.city.LoanerEngine.pawnMsg1", new Object[]{debitor.getName(), debitor.getLastName(), value});
                    TargetedEvent message = new TargetedEvent((IHumanPlayer) debt.getCreditor(), msg);
                    clientServerEventBus.post(message);
                }
            }
        };
    }

    void addNewLoaner(ICity city) {
        factory.createLoaner(city);
    }

    @Override
    public List<AbstractEngine> getChildren() {
        return new ArrayList<>();
    }


    /**
     * Find all the debt of a player with a given loaner.
     * @param loaner
     * @param player
     * @return
     */
    public List<IDebt> findDebts(ILoaner loaner, IPlayer player) {
        ArrayList<IDebt> filteredList = new ArrayList<>();
        for (IDebt debt : loaner.getDebts()) {
            if (player.equals(debt.getDebitor())) {
                filteredList.add(debt);
            }
        }
        return filteredList;
    }

    /**
     * Find all the debts that were granted by a player.
     * @param loaner
     * @param creditor
     * @return
     */
    public List<IDebt> findLoans(ILoaner loaner, IPlayer creditor) {
        ArrayList<IDebt> filteredList = new ArrayList<>();
        for (IDebt debt : loaner.getDebts()) {
            if (creditor.equals(debt.getCreditor())) {
                filteredList.add(debt);
            }
        }
        return filteredList;    }
}
