package ch.sahits.game.openpatrician.engine.player;

import ch.sahits.game.event.data.RepairFinishedEvent;
import ch.sahits.game.event.data.ai.HireSailorEvent;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.LazySingleton;
import ch.sahits.game.openpatrician.clientserverinterface.service.MapService;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.map.IMap;
import ch.sahits.game.openpatrician.model.player.IAITradeStrategy;
import ch.sahits.game.openpatrician.model.player.IAITradeStrategyType;
import ch.sahits.game.openpatrician.model.player.ICityProductionConsumptionKnowledge;
import ch.sahits.game.openpatrician.model.player.IProductionConsumptionKnowledge;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.ITradeStep;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import com.google.common.base.Preconditions;
import com.google.common.eventbus.Subscribe;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.util.Pair;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.ListIterator;
import java.util.stream.Collectors;

/**
 * Base implementation of the trading strategy. The base strategy maintains a list of TradeSteps, that
 * are executed in order. Once the list is empty the next steps need to be defined. It might be that the
 * last executed step complets a trade cycle and the next step would be the repetition of the first.
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public abstract class BasePlayerTradeStrategy implements IAITradeStrategy {
    private final Logger logger = LogManager.getLogger(getClass());
    @Autowired
    private ApplicationContext context;
    @Autowired
    private IMap map;
    @Autowired
    @XStreamOmitField
    private MapService mapService;

    protected IAITradeStrategyType tradeStrategyType;

    protected boolean isMatchingTradeStrategy(IAIPlayer player) {
        return player.getTradeStrategyType().equals(tradeStrategyType);
    }

    /**
     * Retrieve the next trade step that should be executed and remove it from the list of an AI player.
     * @param player on witch to retrieve the next task
     * @return
     */
    public ITradeStep getNextStep(IAIPlayer player, INavigableVessel vessel) {
        return player.getNextTradeStep(vessel);
    }

    /**
     * Check whether the AI player has more trade steps
     * @param player on which to check
     * @return
     */
    public boolean hasMoreTradeSteps(IAIPlayer player, INavigableVessel vessel) {
        return player.hasMoreTradeSteps(vessel);
    }

    /**
     * Add a new trade step for a player.
     * @param player for which to add the trade step
     * @param step to be added.
     */
    public void append(IAIPlayer player, INavigableVessel vessel, ITradeStep step) {
        player.addTradeStep(step, vessel);
    }

    /**
     * Add a new trade step to the begining of the list.
     * @param player for which to add the trade step
     * @param step to be added.
     * @param vessel
     */
    public void inject(IAIPlayer player, INavigableVessel vessel, ITradeStep step) {
        player.injectTradeStep(step, vessel);
    }

    /**
     * Figure out the ware that are most needed in the city. The need is calculated upon the knowledge
     * of the production, consumption and sotred amount. If a ware is not consumed at all it is not needed at
     * all, compared to wares that have more consumption than stored + produced.
     * @param knowledge base knowledge
     * @return sorted list Pair of wares and their importants. The lower the number the more important.
     */
    public List<Pair<IWare, Number>> getMostNeededWares(ICityProductionConsumptionKnowledge knowledge) {
        Preconditions.checkNotNull(knowledge, "The city production knowledge may not be null");
        List<Pair<IWare, Number>> need = new ArrayList<>();
        for (IWare ware : EWare.values()) {
            int stored = knowledge.getStoredAmount(ware);
            int produced = knowledge.getProductionAmount(ware);
            int consumed = knowledge.getConsumptionAmount(ware);
            int value;
            if (consumed == 0) {
                value = Integer.MAX_VALUE;
            } else {
                value = stored + produced - consumed;
            }
            need.add(new Pair<>(ware, value));
        }
        return need.stream().sorted(new WareNeedComparator()).collect(Collectors.toList());
    }

    /**
     * Check if the <code>ware</code> is needed in a city.
     * @param knowledge of the city that should be checked.
     * @param ware for which should be checked.
     * @return true if there is not much of the ware around.
     */
    public boolean isNeeded(ICityProductionConsumptionKnowledge knowledge, IWare ware) {
        List<Pair<IWare, Number>> needs = getMostNeededWares(knowledge);
        for (Pair<IWare, Number> need : needs) {
            if (need.getKey().equals(ware) && need.getValue().intValue() < 10) {
                return true;
            } else if (need.getKey().equals(ware) || need.getValue().intValue() > 10) {
                return false;
            }

        }
        return false;
    }

    /**
     * Figure out the ware that are most needed in the city. The need is calculated upon the knowledge
     * of the production, consumption and sotred amount. If a ware is not consumed at all it is not needed at
     * all, compared to wares that have more consumption than stored + produced. Also consider the amount of that ware
     * that is loaded on the <code>vessel</code>. That ware is treated the same way as if it stored in the city, with the
     * effect that on the first call the most needed ware might be BEER, but once an amount is bought the most needed ware might
     * change.
     * @param knowledge base knowledge
     * @param vessel on which the wares are loaded.
     * @return sorted list Pair of wares and their importants. The lower the number the more important.
     */
    public List<Pair<IWare, Number>> getMostNeededWares(ICityProductionConsumptionKnowledge knowledge, INavigableVessel vessel) {
        List<Pair<IWare, Number>> sortedNeeds = getMostNeededWares(knowledge);
        for (ListIterator<Pair<IWare, Number>> iterator = sortedNeeds.listIterator(); iterator.hasNext(); ) {
            Pair<IWare, Number> need = iterator.next();
            if (vessel.getWare(need.getKey()).getAmount() > 0) {
                int value = need.getValue().intValue() + vessel.getWare(need.getKey()).getAmount();
                iterator.remove();
                iterator.add(new Pair<>(need.getKey(), value));
            }
        }
        return sortedNeeds.stream().sorted(new WareNeedComparator()).collect(Collectors.toList());
    }

    /**
     * Figure out what wares are needed in <code>nextStop</code>, that can be provided in the current town.
     * @param knowledge base knowledge
     * @param knowledgeCurrentTown knowledge of the current town
     * @param nextStop stop of the next city.
     * @return list of wares that can be delivered from the current city to the <code>nextStop</code>
     */
    protected List<IWare> getWaresNeedIn(IProductionConsumptionKnowledge knowledge, ICityProductionConsumptionKnowledge knowledgeCurrentTown, ICity nextStop) {
        List<Pair<IWare, Number>> sortedNeeds;
        ICityProductionConsumptionKnowledge knowledgeFirstTown = knowledge.getKnowlege(nextStop);
        sortedNeeds = getMostNeededWares(knowledgeFirstTown);
        List<IWare> deliverWare = new ArrayList<>(); // wares to be bought in hometown and sold in nextStop
        for (Pair<IWare, Number> need : sortedNeeds) {
            if (knowledgeCurrentTown.getProductionAmount(need.getKey()) > 0) {
                deliverWare.add(need.getKey());
            }
        }
        return deliverWare;
    }

    /**
     * Figure out what the next stop should be to supply the ware in the list <code>waresOfInterest</code>.
     * This list is ordered so that wares in the front are more important.
     * @param baseTown city from which the travel starts
     * @param knowledge data
     * @param waresOfInterest ordered list of needed wares
     * @param exclude city to which should not be traveled.
     * @return City that can provide at least some of the wares and is not to far away.
     */
    protected ICity findNextStop(ICity baseTown, IProductionConsumptionKnowledge knowledge, List<IWare> waresOfInterest, ICity exclude) {
        List<ICity> cities = knowledge.findListWithProductionsMinimalDistance(baseTown, waresOfInterest.get(0)); // TODO ahotz 08.06.2016: do not only consider the first ware
        if (cities.get(0).equals(exclude) && cities.size() > 1) {
            return cities.get(1);
        } else {
            return cities.get(0);
        }
    }

    /**
     * Create the travel step to a different city.
     * @param vessel that is travelling
     * @param travelToCity destination city
     * @return TravelToTradeStep
     */
    protected TravelToTradeStep createTravelToStep(INavigableVessel vessel, ICity travelToCity) {
        TravelToTradeStep travelTo = context.getBean(TravelToTradeStep.class);
        travelTo.setDestinationCity(travelToCity);
        travelTo.setVessel(vessel);
        return travelTo;
    }

    /**
     * Create the trade step to check for repairs.
     * @param vessel that should be checked.
     * @param city in which should be checked.
     * @return CheckForRepairTradeStep
     */
    protected CheckForRepairTradeStep createCheckRepairStep(INavigableVessel vessel, ICity city) {
        CheckForRepairTradeStep repairStep = context.getBean(CheckForRepairTradeStep.class);
        repairStep.setCity(city);
        repairStep.setVessel(vessel);
        return repairStep;
    }



    /**
     * Create a new instance forthe hire sailor step.
     * @param vessel
     * @param city
     * @return
     */
    protected HireSailorsStep createHireSailorStep(INavigableVessel vessel, ICity city) {
        HireSailorsStep hireSailors = context.getBean(HireSailorsStep.class);
        hireSailors.setShip(vessel);
        hireSailors.setCity(city);
        return hireSailors;
    }

    /**
     * Create a new instance of the TakeLoan trade step.
     * @param player that takes a loan
     * @param city in which the loan is taken
     * @return
     */
    protected TakeLoanTradeStep createCheckAndTakeLoanStep(IAIPlayer player, ICity city) {
        TakeLoanTradeStep takeLoan = context.getBean(TakeLoanTradeStep.class);
        takeLoan.setCity(city);
        takeLoan.setPlayer(player);
        return takeLoan;
    }

    protected AggregatedBuyTradeStep createAggregatedBuyTradeStep(INavigableVessel vessel, ICity city, List<IWare> waresToBuy) {
        AggregatedBuyTradeStep buyStepHometown = context.getBean(AggregatedBuyTradeStep.class);
        buyStepHometown.setCity(city);
        buyStepHometown.setVessel(vessel);
        buyStepHometown.setExecuteNext(true);
        for (IWare ware : waresToBuy) {
            int maxPrice = ware.getMaxBuyPriceOffensive();
            buyStepHometown.addBuyStep(ware, maxPrice);
        }
        return buyStepHometown;
    }

    protected AggregatesSellTradeStep createAggregatedSellStep(INavigableVessel vessel, ICity city, List<IWare> waresToSell) {
        AggregatesSellTradeStep sellStep = context.getBean(AggregatesSellTradeStep.class);
        sellStep.setVessel(vessel);
        sellStep.setCity(city);
        sellStep.setExecuteNext(true);
        for (IWare ware : waresToSell) {
            sellStep.addSellStep(ware);
        }
        return sellStep;
    }

    protected AggregatesDumpTradeStep createAggregatedDumpStep(INavigableVessel vessel, ICity city, List<IWare> waresToSell) {
        AggregatesDumpTradeStep sellStep = context.getBean(AggregatesDumpTradeStep.class);
        sellStep.setVessel(vessel);
        sellStep.setCity(city);
        sellStep.setExecuteNext(true);
        for (IWare ware : waresToSell) {
            sellStep.addSellStep(ware);
        }
        return sellStep;
    }

    /**
     * The implementation of the execute trade steps makes the following assumptions:
     * <ul>
     *     <li>When called there is at least one trade step defined</li>
     * </ul>
     * Typically the trade steps are defined up to the travel step. When the ship
     * then reaches the port the next steps need to be defined.
     * @param player
     * @param vessel
     * @return
     */
    @Override
    public boolean executeTradeSteps(IAIPlayer player, INavigableVessel vessel) {
        boolean proceedWithExecutionOfNextStep = true;
        if (hasMoreTradeSteps(player, vessel)) {
            try {
                logger.trace("Executed more steps for {} of {} {}", vessel.getName(), player.getName(), player.getLastName());
                ITradeStep nextStep = getNextStep(player, vessel);
                logger.trace("Next step for {} of {} {}: {}", vessel.getName(), player.getName(), player.getLastName(), nextStep);
                if (nextStep.execute()) {
                    logger.trace("Execute recursive after " + nextStep.getClass().getName());
                    proceedWithExecutionOfNextStep = executeTradeSteps(player, vessel);
                } else {
                    proceedWithExecutionOfNextStep = false;
                    logger.trace("Do not execute any further steps after: {}", nextStep);
                }
            } catch (IllegalStateException e) {
                throw e;
            } catch (RuntimeException e) {
                logger.error("Failed in execution of trade step for strategy: "+player.getTradeStrategyType()+" player "+player.getName()+" "+player.getLastName()+", vessel: "+vessel.getName(), e);
            }
        } else {
            throw new IllegalStateException(String.format("There are no further steps defined for %s of %s %s for strategy %s", vessel.getName(), player.getName(), player.getLastName(), getClass().getName()));
        }
        player.updateTradeWaitingStatus(vessel, !proceedWithExecutionOfNextStep);
        return proceedWithExecutionOfNextStep;
    }

    /**
     * Remove any remaining trade steps.
     * @param vessel for which the trade steps should be checked
     * @param player for which the trade steps should be checked.
     */
    protected void clearRemainingTradeSteps(INavigableVessel vessel, IAIPlayer player) {
        while(hasMoreTradeSteps(player, vessel)) {
            getNextStep(player, vessel);
        }
    }

    /**
     * Handle the event of the ship coming back from repair and the trade steps have to
     * be picked up again.
     * @param event
     */
    @Subscribe
    public void handleRepairFinished(RepairFinishedEvent event) {
        INavigableVessel vessel = event.getShip();
        IAIPlayer player = (IAIPlayer) vessel.getOwner();
        if (isMatchingTradeStrategy(player)) {
            Preconditions.checkArgument(hasMoreTradeSteps(player, vessel), "There most be steps defined after repair for "+vessel.getName()+" of "+player.getName()+" "+player.getLastName());
            player.updateTradeWaitingStatus(vessel, false);
            executeTradeSteps(player, vessel);
        }
    }

    /**
     * Andling the hiring of sailors. This event is fired when the last try to hire sailors failed to insufficient
     * amount of sailors available.
     * @param event
     */
    @Subscribe
    public void handleHireSailors(HireSailorEvent event) {
        INavigableVessel ship = event.getShip();
        ICity city = event.getCity();
        IAIPlayer player = (IAIPlayer) ship.getOwner();
        if (isMatchingTradeStrategy(player)) {
            HireSailorsStep hireSailor = createHireSailorStep(ship, city);
            inject(player, ship, hireSailor);
            executeTradeSteps(player, ship); // thre might not be any tradesteps defined for the ship if it belongs to a group.
        }
    }

    /**
     * Find all wares that are loaded on the ship.
     * @param vessel
     * @return
     */
    protected List<IWare> getLoadedWares(INavigableVessel vessel) {
        return vessel.getLoadedWares()
                .stream()
                .filter(ware -> vessel.getWare(ware).getAmount() > 0)
                .collect(Collectors.toList());
    }

    protected ArrayList<IWare> findWaresOfInterest(List<Pair<IWare, Number>> sortedNeeds) {
        ArrayList<IWare> waresOfInterest = new ArrayList<>(); // wares to be delivered to hometown
        for (Pair<IWare, Number> need : sortedNeeds) {
            if (need.getValue().doubleValue() < 0 || waresOfInterest.isEmpty()) {
                waresOfInterest.add(need.getKey());
            } else if (waresOfInterest.size() < 2) { // Have at least 2 entries in the list
                waresOfInterest.add(need.getKey());
            } else {
                break; // list is sorted by value increasing order
            }
        }
        return waresOfInterest;
    }


    private static class WareNeedComparator implements Comparator<Pair<IWare, Number>> {

        @Override
        public int compare(Pair<IWare, Number> firstPair, Pair<IWare, Number> secondPair) {
            return (int)Math.rint(firstPair.getValue().doubleValue() - secondPair.getValue().doubleValue());
        }
    }
}
