package ch.sahits.game.openpatrician.engine.player;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.LazySingleton;
import ch.sahits.game.openpatrician.clientserverinterface.service.MapService;
import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.event.IShipEntersPortEvent;
import ch.sahits.game.openpatrician.model.map.ITradeRoute;
import ch.sahits.game.openpatrician.model.map.ITradeRouteStop;
import ch.sahits.game.openpatrician.model.product.ITradeMissionData;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import com.google.common.eventbus.AsyncEventBus;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;

/**
 * Trade strategy that is based on a preset trade route
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Jun 18, 2016
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class TradeRouteTradeStrategy extends BasePlayerTradeStrategy {
    private final Logger logger = LogManager.getLogger(getClass());

    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    private ShipService shipService;
    @Value("${aiplayer.money.low}")
    private int lowMoney = 2000;
    @Autowired
    private MapService mapService;

    public TradeRouteTradeStrategy() {
        this.tradeStrategyType = EAITradeStrategyType.TRADE_ROUTE;
    }
    @PostConstruct
    private void init() {
        clientServerEventBus.register(this);
    }
    @PreDestroy
    private void destroy() {
        clientServerEventBus.unregister(this);
    }

    @Override
    public void initialzeTradeCycle(IAIPlayer player, INavigableVessel vessel) {
        // TODO: andi 6/18/16 It can happen that a ware amounts more than can be sold over several iterations
        // At the end of the cycle clear and chose a new route.
        Optional<ICity> optCity = shipService.findCity(vessel);
        if (!optCity.isPresent()) {
            logger.debug("The vessel {} is not in any city but {}", vessel.getName(), vessel.getLocation());
        }
        ICity city = optCity.get();
        ITradeMissionData missionData = player.getTradeMission(vessel);
        if (missionData == null || !(missionData instanceof TradeRouteMissionData)) {
            missionData = selectTradeRoute(city);
            player.setTradeMission(vessel, missionData);
        }
        ITradeRoute route = ((TradeRouteMissionData)missionData).getTradeRoute();
        // sell all loaded wares
        List<IWare> loadedWares = getLoadedWares(vessel);
        if (!loadedWares.isEmpty()) {
            AggregatesSellTradeStep sellStep = createAggregatedSellStep(vessel, city, loadedWares);
            append(player, vessel, sellStep);
        }

        TakeLoanTradeStep takeLoanStep = createCheckAndTakeLoanStep(player, city);
        append(player, vessel, takeLoanStep);
        ITradeRouteStop stop = null;
        ICity nextStop = null;
        for (Iterator<ITradeRouteStop> iterator = route.getTradeStops().iterator(); iterator.hasNext(); ) {
            ITradeRouteStop routeStop =  iterator.next();
            if (routeStop.getTradeStop().equals(city)){
                stop = routeStop;
                if (iterator.hasNext()) {
                    routeStop =  iterator.next();
                    nextStop = routeStop.getTradeStop();
                } else {
                    nextStop = route.getTradeStops().iterator().next().getTradeStop(); // first
                }
                break;
            }
        }
        List<IWare> buyWares = new ArrayList<>(stop.getWaresToBuy());
        AggregatedBuyTradeStep buyStepHometown = createAggregatedBuyTradeStep(vessel, city, buyWares);
        append(player, vessel, buyStepHometown);
        CheckForRepairTradeStep repairStep = createCheckRepairStep(vessel, city);
        append(player, vessel, repairStep);
        HireSailorsStep hireSailors = createHireSailorStep(vessel, city);
        append(player, vessel, hireSailors);
        TravelToTradeStep travelTo = createTravelToStep(vessel, nextStop);
        append(player, vessel, travelTo);
    }

    private TradeRouteMissionData selectTradeRoute(ICity city) {
        List<ITradeRoute> routes = mapService.findTradeRoutesFirstStop(city);
        if (routes.isEmpty()) {
            routes = mapService.findTradeRoutesSecondStop(city);
        }
        Collections.shuffle(routes);
        ITradeRoute route = routes.get(0);
        ITradeMissionData missionData = new TradeRouteMissionData();
        ((TradeRouteMissionData)missionData).setTradeRoute(route);
        return ((TradeRouteMissionData)missionData);
    }

    @Override
    public void handleShipArrivesInPort(IShipEntersPortEvent event) {
        INavigableVessel vessel = event.getShip();
        IAIPlayer player = (IAIPlayer) vessel.getOwner();
        initialzeTradeCycle(player, vessel);
    }

}
