package ch.sahits.game.openpatrician.engine.sea;

import ch.sahits.game.event.data.PeriodicalDailyUpdate;
import ch.sahits.game.event.data.ShipBecomesUnavailableEvent;
import ch.sahits.game.event.data.ShipEntersPortEvent;
import ch.sahits.game.event.data.ShipNearingPortEvent;
import ch.sahits.game.event.data.ai.BlockadeShipRequestEvent;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.DependentInitialisation;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.clientserverinterface.model.factory.ShipFactory;
import ch.sahits.game.openpatrician.clientserverinterface.service.MapService;
import ch.sahits.game.openpatrician.engine.AbstractEngine;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.DisplayMessage;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.IModelTranslationService;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.PlayerList;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.cityhall.CityHallList;
import ch.sahits.game.openpatrician.model.event.TargetedEvent;
import ch.sahits.game.openpatrician.model.people.IShipOwner;
import ch.sahits.game.openpatrician.model.sea.BlockadeState;
import ch.sahits.game.openpatrician.model.sea.IBlockade;
import ch.sahits.game.openpatrician.model.sea.impl.Blockade;
import ch.sahits.game.openpatrician.model.ship.IConvoy;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.util.StartNewGameBean;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map.Entry;
import java.util.Random;

/**
 * Engine handling the blockade.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Apr 14, 2016
 */
@Lazy
@Service
@ClassCategory(EClassCategory.SINGLETON_BEAN)
@DependentInitialisation(StartNewGameBean.class)
public class BlockadeEngine extends AbstractEngine {
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    private BlockadeState blockadeState;
    @Autowired
    private ShipFactory shipFactory;
    @Autowired
    private Date date;
    @Autowired
    private CityHallList cityHalls;
    @Autowired
    private Random rnd;
    @Autowired
    private ApplicationContext context;
    @Autowired
    private PlayerList players;
    @Autowired
    private SeafaringService seafaringService;
    @Autowired
    private SeaFightService seafightService;
    @Autowired
    private IModelTranslationService modelTranslationService;
    @Autowired
    private MapService mapService;

    @Override
    public List<AbstractEngine> getChildren() {
        return Collections.emptyList();
    }
    @PostConstruct
    private void initialize() {
        clientServerEventBus.register(this);
    }
    @PreDestroy
    private void destroy() {
        clientServerEventBus.unregister(this);
    }

    public void initializeNewBlockade(ICity city) {
        DateTime assemblyDate = date.getCurrentDate().plusDays(21);
        int variance = rnd.nextInt(120) - 60;
        DateTime endBlockage = assemblyDate.plusDays(120 + variance);
        Blockade blockade = (Blockade) context.getBean("blockade", assemblyDate, endBlockage);
        ICity assemblyPoint = mapService.getAldermanCity();
        boolean anyShipRequested = false;
        for (IPlayer player : players) {
            int fleetSize = player.getFleet().size();
            int nbShips = Math.min(fleetSize/10, 3);
            if (nbShips > 0) {
                anyShipRequested = true;
                blockade.requestShip(player, nbShips);
                if (player instanceof IHumanPlayer) {
                    String dateAsString = modelTranslationService.toDisplayString(assemblyDate);
                    DisplayMessage msg = new DisplayMessage("ch.sahits.game.openpatrician.engine.sea.BlockadeEngine.blockadesissolve", new Object[]{city.getName(), nbShips, dateAsString, assemblyPoint.getName()});
                    TargetedEvent targetMsg = new TargetedEvent((IHumanPlayer) player, msg);
                    clientServerEventBus.post(targetMsg);
                } else {
                    BlockadeShipRequestEvent event = new BlockadeShipRequestEvent(assemblyPoint, nbShips, (IAIPlayer) player);
                    clientServerEventBus.post(event); // this could also be sent on the serverside only
                }
            }
        }
        if (anyShipRequested) {
            blockadeState.addBlockade(city, blockade);
        }
    }


    @Subscribe
    public void handleDailyUpdate(PeriodicalDailyUpdate dayChange) {
         // handle assembly dates and enb blockade dates
        DateTime now = date.getCurrentDate();
        for (Entry<ICity, IBlockade> blockadeEntry : blockadeState.entrySet()) {
            IBlockade blockade = blockadeEntry.getValue();
            ICity blockadeCity = blockadeEntry.getKey();
            if (blockade.getAssemblyDate().isBefore(now) && blockade.getBlockadeConvoy() == null) {
                // find the ships to put to duty
                ICity aldermanCity = mapService.getAldermanCity();
                List<IPlayer> players = blockade.getPlayersWithRequestedShips();
                for (IPlayer player : players) {
                    ArrayList<IShip> ships = new ArrayList<>();
                    int requestedNumber = blockade.getNumberOfRequestedShips(player);
                    for (IShip ship : player.getFleet()) {
                        if (ship.getLocation().equals(aldermanCity.getCoordinates())) {
                            ships.add(ship);
                            if (ships.size() == requestedNumber) {
                                break;
                            }
                        }
                    }
                    if (ships.size() < requestedNumber) {
                        int missingShips = requestedNumber - ships.size();
                        int variance = rnd.nextInt(4000) - 1500;
                        int fine = (10000 + variance) * missingShips;
                        if (player instanceof IHumanPlayer) {
                            DisplayMessage msg = new DisplayMessage("ch.sahits.game.openpatrician.engine.sea.BlockadeEngine.fineMessage", new Object[]{fine, blockadeCity.getName()});
                            clientServerEventBus.post(new TargetedEvent((IHumanPlayer) player, msg));
                        }
                        player.getCompany().updateCash(-fine);
                    }
                    if (ships.size() > 0) {
                        blockade.addShips(player, ships);
                    }
                }
                if (blockade.getShips().size() > 0) {
                    createConvoy(blockade, aldermanCity);
                    IConvoy convoy = blockade.getBlockadeConvoy();
                    seafaringService.travelBetweenCities(convoy, blockadeCity);
                } else {
                    blockadeState.finishBlockade(blockadeCity);
                }
            }
            if (blockade.getEndBlockade().isBefore(now) && blockade.getBlockadeConvoy() != null && blockade.getBlockadeConvoy().getLocation().equals(blockadeCity.getCoordinates())) {
                ICity returnTo = mapService.getAldermanCity();
                seafaringService.travelBetweenCities(blockade.getBlockadeConvoy(), returnTo);
            }
        }
    }
    @Subscribe
    public void handleShipBreaksBlockade(ShipNearingPortEvent event) {
        ICity city = event.getCity();
        for (Entry<ICity, IBlockade> entry : blockadeState.entrySet()) {
            if (entry.getKey().equals(city)) {
                IConvoy blocadingConvoy = entry.getValue().getBlockadeConvoy();
                INavigableVessel blockadeBreaker = event.getShip();
                BlockadeSeafightContext context = new BlockadeSeafightContext(entry.getValue());
               seafightService.calculateOutcome(blocadingConvoy, blockadeBreaker, context);
            }
        }
    }
    @Subscribe
    public void handleBlockadeConvoyReturns(ShipEntersPortEvent event) {
        // disolve the convoy and remove blockade from blockade state
        INavigableVessel vessel = event.getShip();
        if (vessel instanceof IConvoy) {
            DateTime now = date.getCurrentDate();
            ICity aldermanCity = mapService.getAldermanCity();
            for (Entry<ICity, IBlockade> blockadeEntry : blockadeState.entrySet()) {
                IBlockade blockade = blockadeEntry.getValue();
                if (blockade.getEndBlockade().isBefore(now)) {
                    IConvoy convoy = blockade.getBlockadeConvoy();
                    if (convoy.equals(vessel) && event.getCity().equals(aldermanCity)) {
                        final ICity blockadedCity = blockadeEntry.getKey();
                        dissolveConvoy(blockade, aldermanCity, blockadedCity);
                        blockadeState.finishBlockade(blockadedCity);
                        // TODO: andi 4/27/16 auction the captured ships 
                        return;
                    }
                }
            }
        }
    }

    void dissolveConvoy(IBlockade blockade, ICity city, ICity blockadedCity) {
        IConvoy convoy = blockade.getBlockadeConvoy();
        for (IShip ship : convoy.getShips()) {
            ship.setAvailable(true);
            IShipOwner owner = ship.getOwner();
            convoy.removeShip(ship);
            if (owner instanceof IHumanPlayer) {
                DisplayMessage msg = new DisplayMessage("ch.sahits.game.openpatrician.engine.sea.BlockadeEngine.blockadeRequest", new Object[]{blockadedCity.getName(), ship.getName(), city.getName()});
                TargetedEvent targetMsg = new TargetedEvent((IHumanPlayer) owner, msg);
                clientServerEventBus.post(targetMsg);
            }
            if (owner instanceof IPlayer) {
                IPlayer player = (IPlayer) owner;
                player.addSelectableVessel(ship);
            }
            clientServerEventBus.post(new ShipEntersPortEvent(ship, city));
        }
    }

    void createConvoy(IBlockade blockade, ICity city) {
        List<IShip> allShips = blockade.getShips();

        IConvoy convoy = shipFactory.getConvoy(allShips.get(0), false);
        for (int i = 1; i < allShips.size(); i++) {
            convoy.addShip(allShips.get(i));
        }
        for (IShip ship : allShips) {
            ship.setAvailable(false);
            IPlayer owner = (IPlayer) ship.getOwner();
            if (owner instanceof IHumanPlayer) {
                clientServerEventBus.post(new TargetedEvent((IHumanPlayer) owner, new ShipBecomesUnavailableEvent(ship, city)));
            }
            owner.removeSelectableVessel(ship);
        }
        blockade.setConvoy(convoy);
    }
}
