package ch.sahits.game.openpatrician.engine.sea;

import ch.sahits.game.event.data.ShipPositionUpdateEvent;
import ch.sahits.game.event.data.ClockTick;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.DependentInitialisation;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.dialog.DialogTemplate;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.DisplayMessage;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.map.IMap;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.map.MapSegmentedImage;
import ch.sahits.game.openpatrician.model.map.PirateMapSegmentImage;
import ch.sahits.game.openpatrician.model.map.TreasureMapSegmentImage;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.cityhall.IAcceptedAldermanTask;
import ch.sahits.game.openpatrician.model.city.cityhall.ICapturePirateNest;
import ch.sahits.game.openpatrician.model.city.cityhall.impl.AldermanOffice;
import ch.sahits.game.openpatrician.model.event.TargetedEvent;
import ch.sahits.game.openpatrician.model.personal.IReputation;
import ch.sahits.game.openpatrician.model.ship.IConvoy;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.server.MapLocationDetectionModel;
import ch.sahits.game.openpatrician.server.MapSegmentDataCheck;
import ch.sahits.game.openpatrician.util.StartNewGameBean;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;
import javafx.geometry.Point2D;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Random;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Dec 07, 2015
 */
@ClassCategory(EClassCategory.SINGLETON_BEAN)
@Component
@Lazy
@DependentInitialisation(StartNewGameBean.class)
public class MapLocationDetector {
    @Value("${seamap.visibility.distance}")
    private int locationDistance = 5;
    @Autowired
    private MapLocationDetectionModel dataModel;

    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    @Qualifier("syncServerClientEventBus")
    private EventBus syncServerClientEventBus;
    @Autowired
    @Qualifier("timerEventBus")
    private AsyncEventBus timerEventBus;
    @Autowired
    private Date date;
    @Autowired
    private AldermanOffice aldermanOffice;
    @Autowired
    private IMap map;
    @Autowired
    private Random rnd;
    @Autowired
    private LocationTracker locationTracker;

    private boolean shipsUpdated = false;

    @PostConstruct
    private void init() {
        clientServerEventBus.register(this);
        timerEventBus.register(this);
        syncServerClientEventBus.register(this);
    }
    @PreDestroy
    private void unregister() {
        clientServerEventBus.unregister(this);
        timerEventBus.unregister(this);
        syncServerClientEventBus.unregister(this);
    }


    @Subscribe
    public void checkShipUpdate(ShipPositionUpdateEvent event) {
        shipsUpdated = true;
    }
    @VisibleForTesting
    void handleShipLocationUpdate() {
        for (MapSegmentedImage mapSegment : dataModel.getMapSegments()) {
            Point2D location = mapSegment.getLocation();
            List<INavigableVessel> shipsToCheck = locationTracker.getShipsInSegments(location, locationDistance);
            for (INavigableVessel ship : shipsToCheck) {
                if (location.distance(ship.getLocation()) <= locationDistance && !dataModel.isInCheckedList(ship)) {
                    DateTime dealine = date.getCurrentDate().plusDays(1);
                    MapSegmentDataCheck check = new MapSegmentDataCheck(ship, mapSegment, dealine);
                    dataModel.getCheckList().put(check.getShip(), check);
                }
            }
        }
    }

    @Subscribe
    public void checkShipLeaving(ClockTick tick) {
        if (shipsUpdated) {
            handleShipLocationUpdate();
            shipsUpdated = false;
        }
        for (Iterator<INavigableVessel> iterator = dataModel.getCheckList().keySet().iterator(); iterator.hasNext(); ) {
            INavigableVessel ship = iterator.next();
            MapSegmentDataCheck check = dataModel.getCheckList().get(ship);
            if (ship.getLocation().distance(check.getMapSegment().getLocation()) > locationDistance) {
                iterator.remove();
            } else { // still in the radius
                DateTime now = date.getCurrentDate();
                if (now.isAfter(check.getDeadline())) {
                    iterator.remove();
                    if (check.getMapSegment() instanceof TreasureMapSegmentImage) {
                        TreasureMapSegmentImage treasureMap = (TreasureMapSegmentImage) check.getMapSegment();
                        DisplayMessage msg = new DisplayMessage("ch.sahits.game.openpatrician.engine.sea.MapLocationDetector.treasure", new Object[]{treasureMap.getTreasureAmount()});
                        TargetedEvent displayMessage = new TargetedEvent((IHumanPlayer) ship.getOwner(), msg);
                        clientServerEventBus.post(displayMessage);
                        // post a message
                        ((IPlayer) ship.getOwner()).getCompany().updateCash(treasureMap.getTreasureAmount());
                        ((IPlayer) ship.getOwner()).setSegmentedMap(null);
                        dataModel.removeSegment(treasureMap);
                    }
                    if (check.getMapSegment() instanceof PirateMapSegmentImage) {
                        PirateMapSegmentImage pirateNest = (PirateMapSegmentImage) check.getMapSegment();
                        // calculate if pirate can be overpowered
                        int survivingShips;
                        int pirateFortification = 1;
                        if (ship instanceof IConvoy) {
                            IConvoy convoy = (IConvoy) ship;
                            int randomPirateDestroy = rnd.nextInt(1);
                            survivingShips = convoy.getShips().size() - (pirateNest.getAmountOfShips() + pirateFortification - randomPirateDestroy);

                        } else {
                            int randomPirateDestroy = rnd.nextInt(1);
                            int nbShips = 1;
                            survivingShips =  nbShips - (pirateNest.getAmountOfShips() + pirateFortification - randomPirateDestroy);
                        }
                        if (survivingShips > 0) {
                            dataModel.removeSegment(pirateNest);
                            DisplayMessage msg = new DisplayMessage("ch.sahits.game.openpatrician.engine.sea.MapLocationDetector.pirateDefeated", new Object[]{pirateNest.getName()});
                            TargetedEvent displayMessage = new TargetedEvent((IHumanPlayer) ship.getOwner(), msg);
                            clientServerEventBus.post(displayMessage);
                            if (ship instanceof IConvoy) {
                                removeDefeatedShips((IConvoy) ship, survivingShips);
                            }
                            List<IAcceptedAldermanTask> acceptedTasks = aldermanOffice.getWorkedOnTasks();
                            for (Iterator<IAcceptedAldermanTask> taskIterator = acceptedTasks.iterator(); taskIterator.hasNext(); ) {
                                IAcceptedAldermanTask task = taskIterator.next();
                                if (task.getPlayer().equals(ship.getOwner())) {
                                    if (task.getTask() instanceof ICapturePirateNest && ((ICapturePirateNest) task.getTask()).getPirateNestMap().equals(pirateNest)) {
                                        taskIterator.remove();
                                        if (task.getDeadline().isAfter(now)) {
                                            for (ICity city : map.getCities()) {
                                                IReputation reputation = city.getReputation((IPlayer) ship.getOwner());
                                                reputation.update(150);
                                            }
                                        }
                                    }
                                }
                            }
                        } else { // all ships lost
                            DisplayMessage msg = new DisplayMessage("ch.sahits.game.openpatrician.engine.sea.MapLocationDetector.defeatedByPirate", new Object[]{pirateNest.getName()});
                            TargetedEvent displayMessage = new TargetedEvent((IHumanPlayer) ship.getOwner(), msg);
                            clientServerEventBus.post(displayMessage);
                            if (ship instanceof IConvoy) {
                                final IConvoy convoy = (IConvoy) ship;
                                removeDefeatedShips(convoy, 0);
                            } else {
                                ((IPlayer) ship.getOwner()).getFleet().remove(ship);
                            }
                        }
                    } // end pirate map

                }
            } // end still in radius
        }
    }
    @VisibleForTesting
    void removeDefeatedShips(IConvoy ship, int survivingShips) {
        IConvoy convoy = ship;
        List<IShip> ships = convoy.getShips();
        int destroyedShips = ships.size() - survivingShips;
            if (destroyedShips < ships.size()) {
                ships.remove(convoy.getOrlegShip());
                Collections.shuffle(ships);
            } else {
                ((IPlayer)convoy.getOwner()).getFleet().remove(convoy);
            }
            Multimap<IPlayer, IShip> destoyedShipsMap = ArrayListMultimap.create();
            while (destroyedShips > 0) {
                IShip s = ships.get(destroyedShips-1);
                destoyedShipsMap.put((IPlayer) s.getOwner(), s);
                convoy.removeShip(s);
                ((IPlayer)s.getOwner()).getFleet().remove(s);
                destroyedShips--;
            }
            for (IPlayer player : destoyedShipsMap.keySet()) {
                if (player instanceof IHumanPlayer) {
                    ArrayList<IShip> destroyedShipsOfPlayer = new ArrayList<>(destoyedShipsMap.get(player));
                    if (destroyedShipsOfPlayer.size() > 1) {
                        String titleKey = "ch.sahits.game.openpatrician.engine.sea.MapLocationDetector.shiploss.manytitle";
                        String messageKey = "ch.sahits.game.openpatrician.engine.sea.MapLocationDetector.shiploss.manySunk";
                        StringBuffer sunkenShips = new StringBuffer();
                        for (int i = 0; i < destroyedShipsOfPlayer.size() - 1; i++) {
                            IShip s = destroyedShipsOfPlayer.get(i);
                            sunkenShips.append(s.getName());
                            if (i < destroyedShipsOfPlayer.size() - 2) {
                                sunkenShips.append(", ");
                            }
                        }
                        String lastShipName =  destroyedShipsOfPlayer.get(destroyedShipsOfPlayer.size() - 1).getName();
                        Object[] args = new Object[]{sunkenShips.toString(), lastShipName};
                        DialogTemplate template = DialogTemplate.builder()
                                .closable(true)
                                .titleKey(titleKey)
                                .messageKey(messageKey)
                                .messageArgs(args)
                                .build();
                        DisplayMessage message = new DisplayMessage(titleKey, template);
                        TargetedEvent tagetDisplayMsg = new TargetedEvent((IHumanPlayer)player, message);
                        clientServerEventBus.post(tagetDisplayMsg);
                    } else {
                        String titleKey = "ch.sahits.game.openpatrician.engine.sea.MapLocationDetector.shiploss.title";
                        String messageKey = "ch.sahits.game.openpatrician.engine.sea.MapLocationDetector.shiploss.oneSunk";
                        String lastShipName =  destroyedShipsOfPlayer.get(0).getName();
                        Object[] args = new Object[]{lastShipName};
                        DialogTemplate template = DialogTemplate.builder()
                                .closable(true)
                                .titleKey(titleKey)
                                .messageKey(messageKey)
                                .messageArgs(args)
                                .build();
                        DisplayMessage message = new DisplayMessage(titleKey, template);
                        TargetedEvent tagetDisplayMsg = new TargetedEvent((IHumanPlayer)player, message);
                        clientServerEventBus.post(tagetDisplayMsg);
                    }
                }
            }
    }


}
