package ch.sahits.game.openpatrician.engine.sea;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.DependentInitialisation;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.clientserverinterface.model.PathInterpolatorMap;
import ch.sahits.game.openpatrician.clientserverinterface.model.VesselPositionUpdateData;
import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.engine.sea.model.ShipPositionUpdateTask;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.spring.EngineConfiguration;
import ch.sahits.game.openpatrician.util.StartNewGameBean;
import com.google.common.base.Preconditions;
import javafx.geometry.Point2D;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * Service keeping track of all the ships that are traveling.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Jan 04, 2016
 */
@ClassCategory(EClassCategory.SINGLETON_BEAN)
@Component
@Lazy
@DependentInitialisation(StartNewGameBean.class)
public class SeafaringService {
    private final Logger logger = LogManager.getLogger(getClass());
    @Autowired
    private AStar aStar;
    @Autowired
    @Qualifier("clientThreadPool")
    private ExecutorService clientThreadPool;
    @Autowired
    private ShipPositionUpdateTask shipUpdateTask;
    @Autowired
    private ShipService shipService;
    @Autowired
    private IPathConverter pathConverter;
    @Autowired
    @Qualifier("schedulableServerThreadPool")
    private ScheduledExecutorService schedulableServerThreadPool;
    @Autowired
    private PathInterpolatorMap interpolators;
    @PostConstruct
    private void initializeTimer() {
        schedulableServerThreadPool.scheduleAtFixedRate(shipUpdateTask, 0, EngineConfiguration.CLOCK_TICK_INTERVALL_MS, TimeUnit.MILLISECONDS);
    }
    /**
     * Send a ship or convoy on its way to a destination.
     * @param vessel that is sent
     * @param destination of the travel
     * @return list of points defining the path.
     */
    public List<Point2D> travelTo(INavigableVessel vessel, Point2D destination) {
//        Preconditions.checkArgument(shipService.checkNumberOfSailors(vessel), "The vessel "+vessel.getName()+" has not enough sailors");
        Point2D source = vessel.getLocation();
        Preconditions.checkArgument(!source.equals(destination), "Destination and source may not be the same");
        Future<List<Point2D>> future = clientThreadPool.submit(() -> aStar.findPath(source, destination));
        try {
            return future.get();
        } catch (InterruptedException|ExecutionException e) {
            logger.warn("Failed to execute path finding in separate thread from "+ source+" to "+destination+" of "+vessel.getName()+" by "+vessel.getOwner().getName()+" "+vessel.getOwner().getLastName()+".", e);
            e.printStackTrace();
            return aStar.findPath(source, destination);
        }
    }


    /**
     * Travel from one city to another.
     * @param vessel
     * @param toCity
     */
    public void travelBetweenCities(INavigableVessel vessel, ICity toCity) {
        List<Point2D> path = travelTo(vessel, toCity.getCoordinates());
        pathConverter.createPath(vessel, path, 1);
        VesselPositionUpdateData vesselPositionUpdateData = interpolators.get(vessel);
        vesselPositionUpdateData.setDestinationCity(true);
        vesselPositionUpdateData.setSourceCity(true);
    }

    /**
     * Travel from a location that is not a city to a city.
     * @param vessel
     * @param toCity
     */
    public void travelToCity(INavigableVessel vessel, ICity toCity) {
        List<Point2D> path = travelTo(vessel, toCity.getCoordinates());
        pathConverter.createPath(vessel, path, 1);
        VesselPositionUpdateData vesselPositionUpdateData = interpolators.get(vessel);
        vesselPositionUpdateData.setDestinationCity(true);
        vesselPositionUpdateData.setSourceCity(false);
    }

    /**
     * Travel between two locations, where neither is a city.
     * @param vessel
     * @param location
     */
    public void travelNotBetweenCities(INavigableVessel vessel, Point2D location) {
        List<Point2D> path = travelTo(vessel, location);
        pathConverter.createPath(vessel, path, 1);
        VesselPositionUpdateData vesselPositionUpdateData = interpolators.get(vessel);
        vesselPositionUpdateData.setDestinationCity(false);
        vesselPositionUpdateData.setSourceCity(false);
    }

    /**
     * Travel from a city to a place that is not a city.
     * @param vessel
     * @param location
     */
    public void travelFromCity(INavigableVessel vessel, Point2D location) {
        List<Point2D> path = travelTo(vessel, location);
        pathConverter.createPath(vessel, path, 1);
        VesselPositionUpdateData vesselPositionUpdateData = interpolators.get(vessel);
        vesselPositionUpdateData.setDestinationCity(false);
        vesselPositionUpdateData.setSourceCity(true);
    }
}
