package ch.sahits.game.openpatrician.spring;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.DependentInitialisation;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.engine.ClockTickTimer;
import ch.sahits.game.openpatrician.engine.time.EUpdateIntervalRegistration;
import ch.sahits.game.openpatrician.engine.time.PeriodicalTimeUpdaterV2;
import ch.sahits.game.openpatrician.util.MapInitializedBean;
import ch.sahits.game.openpatrician.util.StartNewGameBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.PropertyPlaceholderConfigurer;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.*;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.Resource;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;

/**
 * Spring configuration for the engine module.
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Feb 02, 2014
 */
@Configuration
@ComponentScan({"ch.sahits.game.openpatrician.engine",
                "ch.sahits.game.openpatrician.javafx.util"})
@ClassCategory(EClassCategory.STARTUP)
public class EngineConfiguration {
    /** Intervall for the clock tick timer */
    public final static int CLOCK_TICK_INTERVALL_MS = 100;
    @Autowired
    private ApplicationContext context;
    @Bean
    public static PropertyPlaceholderConfigurer applicationServerProperties() {
        PropertyPlaceholderConfigurer bean = new PropertyPlaceholderConfigurer();
        bean.setIgnoreUnresolvablePlaceholders(true);
        bean.setLocations(new Resource[]{new ClassPathResource("model-settings.properties")});
        return bean;
    }

    @Bean
    @Scope("prototype")
    public StartNewGameBean startNewGameBean() {
        return new StartNewGameBean();
    }
    @Bean
    @Scope("prototype")
    public MapInitializedBean mapInitializedBean() {
        return new MapInitializedBean();
    }
    @Bean
    public ClockTickTimer clockTickTimer() {
        return new ClockTickTimer(CLOCK_TICK_INTERVALL_MS, ClockTickTimer.DURATION_INFINITY);
    }
    @Bean
    public ExecutorService paralleizationExecutor() {
        int threads = Runtime.getRuntime().availableProcessors();
        return Executors.newFixedThreadPool(threads);
    }
    @Bean
    public ScheduledExecutorService schedulableServerThreadPool() {
        int threads = Runtime.getRuntime().availableProcessors();
        return Executors.newScheduledThreadPool(threads/2 + 1);
    }
    /**
     * Create a periodical updater for end of day which is a singleton.
     *
     * @return
     */
    @Bean
    @Lazy
    @DependentInitialisation(value = StartNewGameBean.class, registerWithBeanName = true)
    public PeriodicalTimeUpdaterV2 periodicalEndOfDayUpdater() {
        return new PeriodicalTimeUpdaterV2(EUpdateIntervalRegistration.END_OF_DAY);
    }

    /**
     * Create a periodical updater for end of week which is a singleton.
     *
     * @return
     */
    @Bean
    @Lazy
    @DependentInitialisation(value = StartNewGameBean.class, registerWithBeanName = true)
    public PeriodicalTimeUpdaterV2 periodicalEndOfWeekUpdater() {
        return new PeriodicalTimeUpdaterV2(EUpdateIntervalRegistration.END_OF_WEEK);
    }

    /**
     * Create a periodical updater for end of month which is a singleton.
     *
     * @return
     */
    @Bean
    @Lazy
    @DependentInitialisation(value = StartNewGameBean.class, registerWithBeanName = true)
    public PeriodicalTimeUpdaterV2 periodicalEndOfMonthUpdater() {
        return new PeriodicalTimeUpdaterV2(EUpdateIntervalRegistration.END_OF_MONTH);
    }
    /**
     * Create a periodical updater for end of year which is a singleton.
     *
     * @return
     */
    @Bean
    @Lazy
    @DependentInitialisation(value = StartNewGameBean.class, registerWithBeanName = true)
    public PeriodicalTimeUpdaterV2 periodicalEndOfYearUpdater() {
        return new PeriodicalTimeUpdaterV2(EUpdateIntervalRegistration.END_OF_YEAR);
    }
    /**
     * Create a periodical updater for weekly updates which is a singleton.
     * @return
     */
    @Bean
    @Lazy
    @DependentInitialisation(value = StartNewGameBean.class, registerWithBeanName = true)
    public PeriodicalTimeUpdaterV2 periodicalWeeklyUpdater() {
        return new PeriodicalTimeUpdaterV2(EUpdateIntervalRegistration.WEEK);
    }
}
