package ch.sahits.game.openpatrician.engine;

import ch.sahits.game.event.data.GraphInitialisationComplete;
import ch.sahits.game.event.data.ShipEntersPortEvent;
import ch.sahits.game.event.data.ai.BlockadeShipRequestEvent;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.LazySingleton;
import ch.sahits.game.openpatrician.clientserverinterface.service.MapService;
import ch.sahits.game.openpatrician.engine.player.BlockadeStrategy;
import ch.sahits.game.openpatrician.engine.sea.SeafaringService;
import ch.sahits.game.openpatrician.model.AIPlayerList;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.people.IShipOwner;
import ch.sahits.game.openpatrician.model.player.IAITradeStrategy;
import ch.sahits.game.openpatrician.model.player.IProductionConsumptionKnowledge;
import ch.sahits.game.openpatrician.model.product.IBlockadeMission;
import ch.sahits.game.openpatrician.model.product.ISpecialMission;
import ch.sahits.game.openpatrician.model.product.ITradeMissionData;
import ch.sahits.game.openpatrician.model.sea.BlockadeState;
import ch.sahits.game.openpatrician.model.sea.IBlockade;
import ch.sahits.game.openpatrician.model.sea.TravellingVessels;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;
import java.util.List;
import java.util.Map.Entry;
import java.util.Optional;

import static java.util.Collections.emptyList;

/**
 * This implements the engine for an AI Player
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Jan 23, 2014
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class AIPlayerEngine extends AbstractEngine {
    private final Logger logger = LogManager.getLogger(getClass());
    @Autowired
    private AIPlayerList players;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    private BlockadeState blockadeState;
    @Autowired
    private BlockadeStrategy blockadeStrategy;
    @Autowired
    private SeafaringService seafaringService;
    @Autowired
    private Date date;
    @Autowired
    private MapService mapService;
    @Autowired
    private TravellingVessels vessels;

    @Override
    public List<AbstractEngine> getChildren() {
        return emptyList();
    }
    @PostConstruct
    private void init() {
        for (IAIPlayer player : players) {
            ICity hometown = player.getHometown();
            IProductionConsumptionKnowledge knowlege = player.getProductionAndConsumptionKnowledge();
            knowlege.updateKnowledge(hometown);
            List<IShip> shipList = player.getFleet();
            for (IShip ship : shipList) {
                try {
                    initializeStrategyForVessel(player, ship);
                } catch (RuntimeException e) {
                    logger.error("Failed to initialize strategy for ship  "+ship.getName()+" of "+player.getName()+" "+player.getLastName(), e);
                }
            }
        }
        clientServerEventBus.register(this);
    }

    private void initializeStrategyForVessel(IAIPlayer player, INavigableVessel ship) {
        IAITradeStrategy tradeStrategy = player.getTradeStrategyType().getStrategy();
        tradeStrategy.initialzeTradeCycle(player, ship);
    }

    @Subscribe
    public void handleBlockadeStart(BlockadeShipRequestEvent event) {
        IAIPlayer player = event.getPlayer();
        int requiredShips = event.getNumberOfShips();
        for (int i = 0; i < requiredShips; i++) {
            Optional<IShip> optShip = blockadeStrategy.selectShip(player);
            if (optShip.isPresent()) {
                IShip vessel = optShip.get();
                player.setTradeMission(vessel, new IBlockadeMission() {});
                seafaringService.travelTo(vessel, event.getAssemblyCity().getCoordinates());
            } else {
                break;
            }
        }
    }
    @Subscribe
    public void handleShipArrivesInCity(ShipEntersPortEvent event) {
        INavigableVessel vessel = event.getShip();
        IShipOwner owner = vessel.getOwner();
        vessels.remove(vessel);
        if (owner instanceof IAIPlayer) {
            IAIPlayer player = (IAIPlayer) owner;
            IProductionConsumptionKnowledge knowlege = player.getProductionAndConsumptionKnowledge();
            knowlege.updateKnowledge(event.getCity());
            ITradeMissionData missionData = player.getTradeMission(vessel);
            if (missionData instanceof ISpecialMission) {
                if (missionData instanceof IBlockadeMission) {
                    handleBlockadeMission(vessel, player, event.getCity());
                } else {
                    throw new IllegalStateException("The special mission "+missionData+" is not implemented");
                }
            } else {
                player.updateTradeWaitingStatus(vessel, false);
                IAITradeStrategy strategy = player.getTradeStrategyType().getStrategy();
                try {
                    strategy.handleShipArrivesInPort(event);
                } catch (RuntimeException e) {
                    logger.error("Failed to handle ship arrival of "+vessel.getName()+" of "+player.getName()+" "+player.getLastName(), e);
                }
            }
        }
    }
    @Subscribe
    public void handleTradeStrategyStart(GraphInitialisationComplete event) {
        for (IAIPlayer player : players) {
            List<INavigableVessel> shipList = player.getSelectableVessels();
            IAITradeStrategy tradeStrategy = player.getTradeStrategyType().getStrategy();
            for (INavigableVessel ship : shipList) {
                // If the vessel is traveling do not execute
                if (!player.waitingForTradeStepToFinish(ship)) {
                    tradeStrategy.executeTradeSteps(player, ship);
                }
            }
        }
    }

    private void handleBlockadeMission(INavigableVessel vessel, IAIPlayer player, ICity city) {
        DateTime now = date.getCurrentDate();
        ICity aldermanCity = mapService.getAldermanCity();
        if (city.equals(aldermanCity)) {
            if (blockadeState.entrySet().isEmpty()) {
                initializeStrategyForVessel(player, vessel);
            } else {
                for (Entry<ICity, IBlockade> entry : blockadeState.entrySet()) {
                    IBlockade blockade = entry.getValue();
                    if (blockade.getAssemblyDate().isBefore(now)) {
                        // Missed assemby date
                        initializeStrategyForVessel(player, vessel);
                    }
                }
            }
        } else {
            initializeStrategyForVessel(player, vessel);
        }
    }



    // TODO ahotz 21.05.2016: When new ship is being build add trade steps for it
    // TODO ahotz 21.05.2016: When ship leaves convoy add trade steps for it
    // TODO ahotz 21.05.2016: When ship joins convoy remove trade steps for the ship
    // TODO: andi 5/22/16 When ship is captured or destroyed 

}
