package ch.sahits.game.openpatrician.engine.player;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.MapType;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.common.base.annotation.IgnoreOnDeserialisation;
import ch.sahits.game.openpatrician.model.player.ICityProductionConsumptionKnowledge;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Model of the knowledge of a player about the production and the consumption of a city.
 */
@Prototype
@ClassCategory({EClassCategory.PROTOTYPE_BEAN, EClassCategory.SERIALIZABLE_BEAN})
public class CityProductionConsumptionKnowledge implements ICityProductionConsumptionKnowledge {



    @MapType(key = IWare.class, value = Integer.class)
    private Map<IWare, Integer> productionAmounts = new HashMap<>();

    @MapType(key = IWare.class, value = Integer.class)
    private Map<IWare, Integer> consumptionAmounts = new HashMap<>();

    @MapType(key = IWare.class, value = Integer.class)
    private Map<IWare, Integer> storedAmounts = new HashMap<>();
    @PostConstruct
    @IgnoreOnDeserialisation
    private void init() {
        for (EWare ware : EWare.values()) {
            if (!storedAmounts.containsKey(ware)) {
                storedAmounts.put(ware, 0);
            }
        }
    }

    /**
     * Update the knowlege about the weekly produced amounts.
     * @param ware
     * @param amount
     */
    public void updateProduction(IWare ware, int amount) {
        productionAmounts.put(ware, amount);
    }
    public List<IWare> getSortedProductionWares() {
        ArrayList<IWare> list = new ArrayList<>();
        productionAmounts.entrySet()
                .stream()
                .sorted(Map.Entry.<IWare, Integer>comparingByValue().reversed())
                .forEach(entry -> list.add(entry.getKey()));
        return list;
    }


    @Override
    public int getProductionAmount(IWare ware) {
        return productionAmounts.get(ware);
    }
    /**
     * Update the knowledge about the weekly consumed amounts.
     * @param ware
     * @param amount
     */
    public void updateConsumption(IWare ware, int amount) {
        consumptionAmounts.put(ware, amount);
    }

    @Override
    public int getConsumptionAmount(IWare ware) {
        return consumptionAmounts.get(ware);
    }


    /**
     * Update the knowledge about the stored amounts.
     * @param ware
     * @param amount
     */
    public void updateStored(IWare ware, int amount) {
        storedAmounts.put(ware, amount);
    }

    @Override
    public int getStoredAmount(IWare ware) {
        return storedAmounts.get(ware);
    }
}
