package ch.sahits.game.openpatrician.engine.player;

import ch.sahits.game.event.data.PeriodicalTimeDayUpdate;
import ch.sahits.game.event.data.PeriodicalTimeWeekUpdate;
import ch.sahits.game.event.data.ShipArrivesAtDestinationEvent;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.LazySingleton;
import ch.sahits.game.openpatrician.engine.AbstractEngine;
import ch.sahits.game.openpatrician.engine.land.city.ReputationCalculator;
import ch.sahits.game.openpatrician.model.IBalanceSheet;
import ch.sahits.game.openpatrician.model.ICompany;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.PlayerList;
import ch.sahits.game.openpatrician.model.building.IBuilding;
import ch.sahits.game.openpatrician.model.building.ISteward;
import ch.sahits.game.openpatrician.model.building.IStorage;
import ch.sahits.game.openpatrician.model.building.ITownHouse;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.building.IWorkShop;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.map.IMap;
import ch.sahits.game.openpatrician.model.personal.IReputation;
import ch.sahits.game.openpatrician.model.product.AmountablePrice;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.sea.TravellingVessels;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.util.spring.DependentPropertyInitializer;
import ch.sahits.game.openpatrician.util.spring.DependentValue;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * Base engine representing a player. If the player is AI the correseponding
 * engine will be a subclass of this engine.
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Jun 16, 2015
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class PlayerEngine extends AbstractEngine {
    private final Logger logger = LogManager.getLogger(getClass());
    @DependentValue("sailor.cost.per.day")
    private double dailySailorCost = 0.5;

    @Autowired
    private PlayerList players;
    @Autowired
    private HumanPlayerEngine humanPlayerEngine;
    @Autowired
    private IMap map;
    @Autowired
    private ReputationCalculator repCalc;
    @Autowired
    private TravellingVessels vessels;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    private DependentPropertyInitializer propertyInitializer;
    @PostConstruct
    private void init() {
        try {
            propertyInitializer.initializeAnnotatedFields(this);
        } catch (IllegalAccessException e) {
            logger.warn("Failed to initialize DependentValue annotated fields");
        }
        clientServerEventBus.register(this);
    }
    @PreDestroy
    private void unregister() {
        clientServerEventBus.unregister(this);
    }
    @Subscribe
    public void handleDailyUpdate(PeriodicalTimeDayUpdate event){
        dailyUpdateBalanceSheet();
        updateCompanyValue();
    }
    /**
     * Handling the daily update events.
     * @param event
     */
    @Subscribe
    public void handleWeeklyUpdate(PeriodicalTimeWeekUpdate event) {
        for (IPlayer player : players) {
            for (ICity city : map.getCities()) {
                try {
                    IReputation reputation = city.getReputation(player);
                    int wareReputation = repCalc.calculateWareRepputation(city, player);
                    reputation.addWareReputation(wareReputation);
                } catch (RuntimeException e) {
                    logger.error("Failed to update reputation for "+player.getName()+" "+player.getLastName()+" in "+city.getName(), e);
                }
            }
        }
    }
    @VisibleForTesting
    void updateCompanyValue() {
        for (IPlayer player : players) {
            ICompany company = player.getCompany();
            long newCompanyValue = company.getCash();
            // Add value of all ships
            List<IShip> ships = player.getFleet();
            for (IShip ship : ships) {
                newCompanyValue += ship.getValue();
                Set<IWare> loaded = ship.getLoadedWares();
                for (IWare ware : loaded) {
                    AmountablePrice<IWare> ap = ship.getWare(ware);
                    newCompanyValue+=ap.getAVGPrice()*ap.getAmount();
                }
            }

            // Add value of all buildings

            for (ICity city :  map.getCities()) {
                List<IBuilding> buildings = player.findBuildings(city);
                for (IBuilding building : buildings) {
                    newCompanyValue += building.getValue();
                    if (building instanceof ITradingOffice){
                        ITradingOffice office = (ITradingOffice) building;
                        for (IWare ware : EWare.values()) {
                            AmountablePrice<IWare> ap = office.getWare(ware);
                            newCompanyValue+=ap.getAVGPrice()*ap.getAmount();
                        }
                    }
                }

            }
            company.setCompanyValue(newCompanyValue);
        }
    }


    @Override
    public List<AbstractEngine> getChildren() {
        ArrayList<AbstractEngine> engines = new ArrayList<>();
        engines.add(humanPlayerEngine);
        return engines;
    }
    private void dailyUpdateBalanceSheet() {
        // property taxes are handled by the city hall engine
        // office trading is handled in the AutomaticTradingEngine
        List<ICity> cities = map.getCities();
        for (IPlayer player : players) {
            long sum = 0;
            int stewardCosts = 0;
            int rentalIncome = 0;
            int wageCosts = 0;
            int otherCosts = 0;

            for (ICity city : cities) {
                // steward costs
                ITradingOffice office = player.findTradingOffice(city);
                if (office != null) {
                    ISteward steward = office.getSteward();
                    if (steward != null) {
                        stewardCosts -= (int) Math.rint(steward.getSalary() / 7.0);
                        sum += stewardCosts;
                    }
                    // rentalIncome
                    List<ITownHouse> townHouses = player.findBuildings(city, ITownHouse.class);
                    for (IBuilding building : townHouses) {
                        rentalIncome += ((ITownHouse) building).computeRentalIncome();
                        sum += rentalIncome;
                    }
                    // wages
                    List<IWorkShop> workshops = player.findBuildings(city, IWorkShop.class);
                    IStorage storage = office.getStorage();
                    for (IBuilding building : workshops) {
                        int workers = ((IWorkShop) building).getWorkers();
                        int salaryPerWorker = ((IWorkShop) building).getSalaryPerWorker();
                        wageCosts -= (int) Math.rint(salaryPerWorker * workers / 7.0);
                    }
                    wageCosts -= storage.guardCostsPerDayBinding().get();
                    List<IShip> fleet = player.getFleet();
                    for (IShip ship : fleet) {
                        double sailorSalary = ship.getNumberOfSailors() * dailySailorCost;
                        wageCosts -= sailorSalary;
                        if (ship.getCaptian().isPresent()) {
                            wageCosts -= ship.getCaptian().get().getSalary();
                        } else if (sailorSalary > 0) {
                            wageCosts -= dailySailorCost;
                        }
                    }
                    sum += wageCosts;

                    // other costs
                    otherCosts -= storage.costsPerDayBinding().get();
                    sum += otherCosts;
                    // all other exceptional costs are deduced where they occur.

                    player.getCompany().updateCash(sum);
                    if (player instanceof IHumanPlayer) {
                        ITradingOffice mainOffice = player.findTradingOffice(player.getHometown());
                        IBalanceSheet balanceSheet = mainOffice.getCurrentWeek();
                        balanceSheet.updateRentalIncome(rentalIncome);
                        balanceSheet.updateStewardCosts(stewardCosts);
                        balanceSheet.updateSalaries(wageCosts);
                        balanceSheet.updateOtherExpensesRegular(otherCosts);
                    }

                }
            }
        }
    }
    @Subscribe
    public void handleShipReachesDestination(ShipArrivesAtDestinationEvent event) {
        INavigableVessel vessel = event.getShip();
        vessels.remove(vessel);
    }
}
