package ch.sahits.game.openpatrician.engine.sea.model;

import ch.sahits.game.event.EGameStatusChange;
import ch.sahits.game.event.GameStateChange;
import ch.sahits.game.event.data.ClockTickIntervalChange;
import ch.sahits.game.event.data.PauseGame;
import ch.sahits.game.event.data.ResumeGame;
import ch.sahits.game.event.data.ShipArrivesAtDestinationEvent;
import ch.sahits.game.event.data.ShipEntersPortEvent;
import ch.sahits.game.event.data.ShipLeavingPort;
import ch.sahits.game.event.data.ShipNearingPortEvent;
import ch.sahits.game.event.data.ShipPositionUpdateEvent;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.LazySingleton;
import ch.sahits.game.openpatrician.clientserverinterface.model.PathInterpolatorMap;
import ch.sahits.game.openpatrician.clientserverinterface.model.VesselPositionUpdateData;
import ch.sahits.game.openpatrician.clientserverinterface.service.MapService;
import ch.sahits.game.openpatrician.clientserverinterface.service.PathInterpolator;
import ch.sahits.game.openpatrician.clientserverinterface.service.PointService;
import ch.sahits.game.openpatrician.engine.sea.IPathConverter;
import ch.sahits.game.openpatrician.model.DisplayMessage;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.map.IMap;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.event.PausableSchedulableTask;
import ch.sahits.game.openpatrician.model.event.TargetedEvent;
import ch.sahits.game.openpatrician.model.ship.IGroupableVessel;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.spring.EngineConfiguration;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;
import javafx.geometry.Point2D;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.Iterator;
import java.util.Map.Entry;

/**
 * Task to periodically update the ships positions.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Jan 12, 2016
 */
@ClassCategory(EClassCategory.SINGLETON_BEAN)
@LazySingleton
public class ShipPositionUpdateTask extends PausableSchedulableTask {
    private final Logger logger = LogManager.getLogger(getClass());
    @Autowired
    private PathInterpolatorMap interpolators;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    @Qualifier("syncServerClientEventBus")
    private EventBus syncServerClientEventBus;
    @Autowired
    @Qualifier("timerEventBus")
    private AsyncEventBus timerEventBus;
    @Autowired
    private IMap map;
    @Autowired
    private IPathConverter pathConverter;
    @Autowired
    private PointService pointService;
    @Autowired
    private MapService mapService;

    private long pausedTimeStamp = -1;


    @PostConstruct
    protected void register() {
        clientServerEventBus.register(this);
        super.register();
        pause(new PauseGame());
    }
    @PreDestroy
    public void unregister() {
        clientServerEventBus.unregister(this);
        super.unregister();
    }
    @Override
    public void handle(long now) {
        synchronized (interpolators) {
            for (Iterator<Entry<INavigableVessel, VesselPositionUpdateData>> iterator = interpolators.entrySet().iterator(); iterator.hasNext(); ) {
                Entry<INavigableVessel, VesselPositionUpdateData> entry = iterator.next();
                final VesselPositionUpdateData value = entry.getValue();
                final INavigableVessel vessel = entry.getKey();

                PathInterpolator interpolator = value.getInterpolator();
                double traveldFraction = interpolator.getTraveldFraction();
                double travelPerTick = interpolator.getTravelFractionPerTick();

                updateShipDamage(value, vessel, travelPerTick);

                double animationFraction = Math.min(1, traveldFraction + travelPerTick);
                Point2D newLocation = interpolator.interpolate(animationFraction);
                Point2D oldLocation = vessel.getLocation();
                vessel.setLocation(newLocation);
                syncServerClientEventBus.post(new ShipPositionUpdateEvent(vessel, oldLocation, newLocation));

                logger.trace("interpolate path for {} move to {}, fraction={}", vessel.getName(), newLocation, animationFraction);
                if (value.isSourceCity()) {
                    final Point2D source = value.getSource();
                    if (oldLocation.distance(source)< 10 && newLocation.distance(source) >= 10) {
                        // leaving city
                        ICity city = map.findCity(source);
                        clientServerEventBus.post(new ShipLeavingPort(vessel, city));
                    }
                }
                final Point2D destination = value.getDestination();
                if (value.isDestinationCity()) {
                    if (newLocation.distance(destination) <= 10 && oldLocation.distance(destination) > 10) {
                        // nearing city
                        ICity city = map.findCity(destination);
                        clientServerEventBus.post(new ShipNearingPortEvent(vessel, city));
                    }
                }
                if (pointService.equals(destination, newLocation)) {
                    vessel.setLocation(destination);
                    if (value.isDestinationCity()) {
                        // Send event arrive in city
                        ICity city = map.findCity(destination);
                        clientServerEventBus.post(new ShipEntersPortEvent(vessel, city));
                        if (vessel.getOwner() instanceof IHumanPlayer) {
                            DisplayMessage msg = new DisplayMessage("ch.sahits.game.openpatrician.engine.sea.model.ShipPositionUpdateTask.shipArrivedInCity",
                                    new Object[]{vessel.getName(), city.getName()});
                            clientServerEventBus.post(new TargetedEvent((IHumanPlayer) vessel.getOwner(), msg));
                        }
                    } else {
                        clientServerEventBus.post(new ShipArrivesAtDestinationEvent(vessel, destination));
                        if (vessel.getOwner() instanceof IHumanPlayer) {
                            DisplayMessage msg = new DisplayMessage("ch.sahits.game.openpatrician.engine.sea.model.ShipPositionUpdateTask.shipArrivedAtDestination",
                                    new Object[]{vessel.getName()});
                            clientServerEventBus.post(new TargetedEvent((IHumanPlayer) vessel.getOwner(), msg));
                        }
                    }
                    // remove vessel from list
                    iterator.remove(); // what and how many entries are removed?
                }

            }
        }
    }

    /**
     * Handle the update of damage to the ship.
     * @param value vessel data that needs to be updated with the distance that is not yet considered in the damage
     * @param vessel that should be damaged
     * @param travelPerTick number of pixels that were traveled.
     */
    @VisibleForTesting
    void updateShipDamage(VesselPositionUpdateData value, INavigableVessel vessel, double travelPerTick) {
        double distanceInKm = mapService.convertToDistenceInKm(travelPerTick);
        double distanceSinceLastDamageUpdate = value.getTraveledDistanceSinceLastDamageUpdate() + distanceInKm;
        if (vessel instanceof IShip) {
            IShip ship = (IShip) vessel;
            damageShip(value, distanceSinceLastDamageUpdate, ship);
        } else if (vessel instanceof IGroupableVessel){
            for (IShip ship : ((IGroupableVessel) vessel).getShips()) {
                // The traveled distance since the last update is determined by the last ship
                damageShip(value, distanceSinceLastDamageUpdate, ship);
            }
        }
    }

    /**
     * Update the damage of the ship that is taken by traveling.
     * @param value vessel data that needs to be updated with the distance that is not yet considered in the damage
     * @param distanceSinceLastDamageUpdate traveled distance in km since the last update of the damage
     * @param ship instance of the ship that is to be damaged.
     */
    private void damageShip(VesselPositionUpdateData value, double distanceSinceLastDamageUpdate, IShip ship) {
        int kmPerDamage = ship.getDistanceInKmForOneHealthPointReduction();
        if (distanceSinceLastDamageUpdate > kmPerDamage) {
            int damagePoint = calculateDamagePoints(kmPerDamage, distanceSinceLastDamageUpdate);
            ship.damage(damagePoint);
            double remainingDistance = distanceSinceLastDamageUpdate - (damagePoint * kmPerDamage);
            value.setTraveledDistanceSinceLastDamageUpdate(remainingDistance);
        } else {
            value.setTraveledDistanceSinceLastDamageUpdate(distanceSinceLastDamageUpdate);
        }
    }

    /**
     * Calculate the damage for the traveld distance
     * @param kmPerDamage amount of km that will cause 1 damage point.
     * @param distanceSinceLastDamageUpdate distance traveled since the damage was updated the last time
     * @return number of damage points.
     */
    private int calculateDamagePoints(int kmPerDamage, double distanceSinceLastDamageUpdate) {
        int damage = 0;
        double tempDistance = distanceSinceLastDamageUpdate;
        while (tempDistance >= kmPerDamage) {
            damage++;
            tempDistance -= kmPerDamage;
        }
        return damage;
    }

    @Subscribe
    public void handleGameSpeedChange(ClockTickIntervalChange intervallChange) {
        synchronized (interpolators) {
            for (Iterator<Entry<INavigableVessel, VesselPositionUpdateData>> iterator = interpolators.entrySet().iterator(); iterator.hasNext(); ) {
                Entry<INavigableVessel, VesselPositionUpdateData> entry = iterator.next();
                final VesselPositionUpdateData data = entry.getValue();
                final PathInterpolator interpolator = data.getInterpolator();
                double traveledFraction = interpolator.getTraveldFraction();
                double traveledTimeMS = data.getDurationInMs() * traveledFraction;
                double fractionToTravel = 1 - traveledFraction;
                long remainingDuration = pathConverter.calculateDuration(entry.getKey(), interpolator, fractionToTravel);
                long newDuration = Math.round(traveledTimeMS + remainingDuration);
                long nbTicks = remainingDuration / EngineConfiguration.CLOCK_TICK_INTERVALL_MS;
                double fractionPerTick = fractionToTravel/nbTicks;
                interpolator.setTravelFractionPerTick(fractionPerTick);
                data.setDurationInMs(newDuration);
                logger.trace("Update the game speed, fraction to travel={}, nbTicks={}, remainingDuration={}, update fraction per tick={}", fractionToTravel, nbTicks, remainingDuration, fractionPerTick);
            }
        }
    }

    @Override
    public void pause(PauseGame evt) {
        logger.debug("Pause Task");
        pausedTimeStamp = System.currentTimeMillis();
        super.pause(evt);
    }

    @Override
    public void resume(ResumeGame evt) {
        logger.debug("Resume Task, paused timestamp: {}", pausedTimeStamp);
        long pausedDurationMS = System.currentTimeMillis() - pausedTimeStamp; // nano time is not a timestamp
        synchronized (interpolators) {
            for (VesselPositionUpdateData data : interpolators.values()) {
                long originalStartTimestampMS = data.getAnimationStartMS();
                long newStartTimestampNS = originalStartTimestampMS + pausedDurationMS; // during that time it was not animated
                data.setAnimationStartMS(newStartTimestampNS);
                logger.trace("Change start time from {} -> {}", originalStartTimestampMS, newStartTimestampNS);
            }
        }
        super.resume(evt);
        pausedTimeStamp = -1;
    }
    @Subscribe
    public void loadedGame(GameStateChange change) {
        if (change.getStatusChange() == EGameStatusChange.GAME_LOADED) {
            logger.trace("Reset the time, so it is calculated correctly on resume, so that the distance to travel is correct");
            if (interpolators.isEmpty()) {
                logger.trace("There are no ships traveling, so there is no need.");
            } else {
                // find the longest route as that gives the most accurate result
                for (Entry<INavigableVessel, VesselPositionUpdateData> entry : interpolators.entrySet()) {
                    final VesselPositionUpdateData vesselData = entry.getValue();
                    // total travel duration
                    long totalDuration = vesselData.getDurationInMs();
                    // traveled fraction
                    double traveledFraction = vesselData.getInterpolator().getTraveldFraction();
                    // calculate the time that was traveled
                    long traveledTime = Math.round(totalDuration * traveledFraction);
                    // set original start timestamp as paused timestamp - traveled time
                    long newStartTime = pausedTimeStamp - traveledTime;
                    vesselData.setAnimationStartMS(newStartTime);
                    logger.trace("Update the ship {} with new start animation time {}; totalDuration={}, traveledFraction={}, traveledTime={}ms", entry.getKey().getName(), newStartTime, totalDuration, traveledFraction, traveledTime);
                }
            }
        }
    }
}
